/*
 * Copyright 2014 Mopote.com The Project licenses this file to you under the Apache License, version 2.0 (the
 * "License"); you may not use this file except in compliance with the License. You may obtain a copy of the License at:
 * http://www.apache.org/licenses/LICENSE-2.0 Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language governing permissions and limitations under the
 * License.
 */
package cn.com.duiba.tuia.media.common.utils;

import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;

import org.bouncycastle.util.encoders.Hex;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import cn.com.duiba.tuia.media.common.constants.ErrorCode;
import cn.com.duiba.tuia.media.common.exception.TuiaMediaException;

/**
 * @since 1.0
 * @author Allen
 */
public final class SignatureUtils {

    /** The logger. */
    private static Logger        logger               = LoggerFactory.getLogger(SignatureUtils.class);

    private static final char[]  HEX_ARRAY            = "0123456789ABCDEF".toCharArray();
    private static final String  ENCRYPTION_ALGORITHM = "SHA-1";
    private static final boolean DEFAULT_PUB          = true;

    private SignatureUtils() {
    }

    /**
     * 
     * toHex:(这里用一句话描述这个方法的作用). <br/>
     *
     * @author ZFZ
     * @param data
     * @return
     * @since JDK 1.6
     */
    public static String toHex(byte[] data) {
        return new String(Hex.encode(data));
    }

    /**
     * 
     * bytesToHexString:(这里用一句话描述这个方法的作用). <br/>
     *
     * @author ZFZ
     * @param bytes
     * @return
     * @since JDK 1.6
     */
    public static String bytesToHexString(byte[] bytes) {
        char[] hexChars = new char[bytes.length * 2];
        for (int j = 0; j < bytes.length; j++) {
            int v = bytes[j] & 0xFF;
            hexChars[j * 2] = HEX_ARRAY[v >>> 4];
            hexChars[j * 2 + 1] = HEX_ARRAY[v & 0x0F];
        }
        return new String(hexChars);
    }

    /**
     * 
     * hexStringToBytes:(这里用一句话描述这个方法的作用). <br/>
     *
     * @author ZFZ
     * @param s
     * @return
     * @since JDK 1.6
     */
    public static byte[] hexStringToBytes(String s) {
        int len = s.length();
        byte[] data = new byte[len / 2];
        for (int i = 0; i < len; i += 2) {
            data[i / 2] = (byte) ((Character.digit(s.charAt(i), 16) << 4) + Character.digit(s.charAt(i + 1), 16));
        }
        return data;
    }

    /**
     * 
     * digest:(这里用一句话描述这个方法的作用). <br/>
     *
     * @author ZFZ
     * @param strSrc
     * @return
     * @throws TuiaMediaException
     * @since JDK 1.6
     */
    public static String digest(String strSrc) throws TuiaMediaException {
        return digest(strSrc, DEFAULT_PUB);
    }

    /**
     * 
     * digest:(这里用一句话描述这个方法的作用). <br/>
     *
     * @author ZFZ
     * @param strSrc
     * @param pub
     * @return
     * @throws TuiaMediaException
     * @since JDK 1.6
     */
    public static String digest(String strSrc, boolean pub) throws TuiaMediaException {
        return digest(strSrc, "MD5", pub);
    }

    /**
     * 
     * digest:(这里用一句话描述这个方法的作用). <br/>
     *
     * @author ZFZ
     * @param strSrc
     * @param encName
     * @return
     * @throws TuiaMediaException
     * @since JDK 1.6
     */
    public static String digest(String strSrc, String encName) throws TuiaMediaException {
        return digest(strSrc, encName, DEFAULT_PUB);
    }

    /**
     * 
     * digest:(使用指定算法生成消息摘要，默认是md5). <br/>
     *
     * @author ZFZ
     * @param strSrc
     * @param encName
     * @param pub
     * @return
     * @throws TuiaMediaException
     * @since JDK 1.6
     */
    public static String digest(String strSrc, String encName, boolean pub) throws TuiaMediaException {
        MessageDigest md = null;
        String strDes = null;
        String eName = encName;
        byte[] bt = strSrc.getBytes();
        try {
            if (eName.isEmpty()) {
                eName = "MD5";
            }
            md = MessageDigest.getInstance(eName);
            md.update(bt);
            byte[] digest = md.digest();
            strDes = pub ? toHex(digest) : bytesToHexString(digest);
        } catch (NoSuchAlgorithmException e) {
            logger.error(" SignatureUtils digest is error , the e = [{}]", e);
            throw new TuiaMediaException(ErrorCode.E0002003);
        }
        return strDes;
    }

    /**
     * 
     * signature:(这里用一句话描述这个方法的作用). <br/>
     *
     * @author ZFZ
     * @param appId
     * @param key
     * @param digest
     * @param millis
     * @return
     * @throws TuiaMediaException
     * @since JDK 1.6
     */
    public static String signature(String appId, String key, String digest, long millis) throws TuiaMediaException {
        return signature(appId, key, digest, millis, DEFAULT_PUB);
    }

    /**
     * 
     * signature:(这里用一句话描述这个方法的作用). <br/>
     *
     * @author ZFZ
     * @param appId
     * @param key
     * @param digest
     * @param millis
     * @param pub
     * @return
     * @throws TuiaMediaException
     * @since JDK 1.6
     */
    public static String signature(String appId, String key, String digest, long millis, boolean pub)
                                                                                                     throws TuiaMediaException {
        String timestamp = String.valueOf(millis);
        String signature = null;
        if (isNotBlank(key) && isNotBlank(timestamp) && isNotBlank(appId)) {
            String value = SignatureArithmeticUtils.get(appId, key, digest, String.valueOf(millis));
            signature = digest(value, ENCRYPTION_ALGORITHM, pub);
        }
        return signature;
    }

    /**
     * 
     * isValidate:(这里用一句话描述这个方法的作用). <br/>
     *
     * @author ZFZ
     * @param signature
     * @param appId
     * @param secretKey
     * @param digest
     * @param millis
     * @return
     * @throws TuiaMediaException
     * @since JDK 1.6
     */
    public static boolean isValidate(String signature, String appId, String secretKey, String digest, long millis)
                                                                                                                  throws TuiaMediaException {
        return isValidate(signature, appId, secretKey, digest, millis, DEFAULT_PUB);
    }

    /**
     * 
     * isValidate:(这里用一句话描述这个方法的作用). <br/>
     *
     * @author ZFZ
     * @param signature
     * @param appId
     * @param secretKey
     * @param digest
     * @param millis
     * @param pub
     * @return
     * @throws TuiaMediaException
     * @since JDK 1.6
     */
    public static boolean isValidate(String signature, String appId, String secretKey, String digest, long millis,
                                     boolean pub) throws TuiaMediaException {
        String calculatedSignature = signature(appId, secretKey, digest, millis, pub);
        return isBlank(calculatedSignature) ? false : calculatedSignature.equals(signature);
    }

    /**
     * 
     * isNotBlank:(这里用一句话描述这个方法的作用). <br/>
     *
     * @author ZFZ
     * @param str
     * @return
     * @since JDK 1.6
     */
    public static boolean isNotBlank(String str) {
        return !isBlank(str);
    }

    /**
     * 
     * isBlank:(这里用一句话描述这个方法的作用). <br/>
     *
     * @author ZFZ
     * @param str
     * @return
     * @since JDK 1.6
     */
    public static boolean isBlank(String str) {
        int strLen;
        if (str == null || (strLen = str.length()) == 0) {
            return true;
        }
        for (int i = 0; i < strLen; i++) {
            if (!Character.isWhitespace(str.charAt(i))) {
                return false;
            }
        }
        return true;
    }
}
