package cn.com.duiba.tuia.pangea.center.api.utils;

import cn.com.duiba.tuia.pangea.center.api.dto.dayu.SceneConfig;
import cn.com.duiba.tuia.pangea.center.api.enums.ArgumentType;
import cn.com.duiba.tuia.pangea.center.api.enums.ConfigTypeEnum;
import cn.com.duiba.tuia.pangea.center.api.enums.DiversionTypeEnum;
import cn.com.duiba.tuia.pangea.center.api.rsp.DayuResult;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.collections.CollectionUtils;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Random;
import java.util.stream.Collectors;

/**
 * Created by lizhihui on 2017-08-09 17:50.
 */
@Slf4j
public class SceneUtils {

    private SceneUtils() {
    }

    /**
     * 解析配置文件(多layer层流量分割)
     *
     * @param sceneConfig 配置文件
     * @param params 参数
     * @return 实验结果
     */
    public static DayuResult resolve(SceneConfig sceneConfig, Map<ArgumentType, Object> params) {
        return SceneUtils.getDayuResult(sceneConfig, params);
    }

    private static DayuResult getDayuResult(SceneConfig sceneConfig, Map<ArgumentType, Object> params) {
        DayuResult result = new DayuResult();
        result.setArguments(new HashMap<>());
        result.setExpIds(new ArrayList<>());

        SceneUtils.getExpIdsAndArguments(sceneConfig, null, params, result, 0);
        // 对实验id进行排序,从小到大
        result.setExpIds(result.getExpIds().stream().sorted().collect(Collectors.toList()));
        return result;
    }

    /**
     * 获取本次请求经历的实验id和参数(这个方法不建议改.除非业务改了)
     *
     * @param sceneConfig 场景配置信息
     * @param parentType 父层类型(调用时必须传null)
     * @param params 参数集合
     * @param result 结果类
     * @param hash 计算出来的hash值(调用时可以随意传值)
     */
    private static void getExpIdsAndArguments(SceneConfig sceneConfig, String parentType,
                                              Map<ArgumentType, Object> params, DayuResult result, Integer hash) {

        if (sceneConfig == null) {
            return;
        }

        // 先获取下层配置,如果下层没有配置了,直接返回
        List<SceneConfig> configList = sceneConfig.getConfigList();
        if (!sceneConfig.getType().equals(ConfigTypeEnum.EXPERIMENT.getType())
            && (CollectionUtils.isEmpty(configList))) {
            return;
        }

        // 如果父层类型为null,则说明这层是rootDomain(这个null是调用方传入的) 此时调用子层hash=100表示必定命中
        // 递归处理type=null（rootDomain）、rootDomain的子层先为layer、experiment
        if (parentType == null) {
            // sceneConfig为rootDomain时，这里configList为layer集合，sceneConfig.getType()=domain
            configList.forEach(config -> getExpIdsAndArguments(config, sceneConfig.getType(), params, result, 100));

        } else {// 父层只可能是domain或者layer
            // rootDomain的子层一般为 domain，进入if后根据 传参deviceId或consumerId创建hash
            if (parentType.equals(ConfigTypeEnum.DOMAIN.getType())) {

                // 如果父层是domain,那么这层只能是layer，这里的sceneConfig是layer，sceneConfig.getDiversion()=DiversionTypeEnum.DEVICE_ID_HASH创建场景时设置
                // layer没设置名称，使用layer的次序i
                Integer finalHash = calculateHash(params, sceneConfig.getDiversion(), sceneConfig.getName());
                // 遍历实验层
                configList.forEach(config -> getExpIdsAndArguments(config, sceneConfig.getType(), params, result,
                                                                   finalHash));
            } else {// 到这里parentType只能是layer,layer下面可能是domain或者experiment

                // 如果是domain,则继续遍历
                if (sceneConfig.getType().equals(ConfigTypeEnum.DOMAIN.getType())
                    && doesHit(sceneConfig.getFlowRate(), hash)) {
                    configList.forEach(config -> getExpIdsAndArguments(config, sceneConfig.getType(), params, result,
                                                                       hash));
                } else if (doesHit(sceneConfig.getFlowRate(), hash)) { // getFlowRate 流量粪桶字符串,格式: 1,10
                    // 代码走到这里.说明这个config的类型是experiment

                    // 实验链路里增加该实验id
                    Long id = sceneConfig.getId();
                    result.getExpIds().add(id);

                    // 参数集合里增加该参数
                    result.getArguments().putAll(sceneConfig.getArguments());
                }
            }
        }
    }

    /**
     * 计算该数字是否命中该流量
     *
     * @param flow 流量粪桶字符串,格式: 1,10
     * @param number 流量数值
     * @return 是否在分桶范围内 true-是 false 否
     */
    private static boolean doesHit(String flow, Integer number) {
        if(!flow.contains(",")){
            return true;//百分之百
        }
        Integer start = Integer.parseInt(flow.split(",")[0]);
        Integer end = Integer.parseInt(flow.split(",")[1]);

        return number >= start && number <= end;
    }

    /**
     * 计算hash值
     *
     * @param params 参数m
     * @param hashType 流量切割类型
     * @param layerName layer的名称
     * @return 计算后的概率值
     */
    private static int calculateHash(Map<ArgumentType, Object> params, Integer hashType, String layerName) {
        int hash;

        if (hashType.equals(DiversionTypeEnum.CONSUMER_ID_HASH.getType())) {

            Long consumerId = (Long) params.get(ArgumentType.CONSUMER_ID);
            hash = HashAlgorithm.dekHash(consumerId.toString() + layerName);

        } else if (hashType.equals(DiversionTypeEnum.DEVICE_ID_HASH.getType())) {

            String deviceId = (String) params.get(ArgumentType.DEVICE_ID);
            hash = HashAlgorithm.apHash(deviceId + layerName);

        } else {
            // 0-99
            hash = new Random().nextInt(100);
        }
        // 上述hash算法会计算出负值.所以要进行正负转换,然后100取模,+1
        return (hash < 0 ? -hash : hash) % 100 + 1;
    }
}
