package cn.com.duiba.tuia.pangea.center.api.localservice.apollopangu;

import cn.com.duiba.tuia.pangea.center.api.dto.apollo.ApolloKeyValueDTO;
import cn.com.duiba.tuia.pangea.center.api.remoteservice.RemoteApolloKeyValueService;
import cn.com.duiba.tuia.pangea.center.api.utils.HashAlgorithm;
import cn.com.duiba.wolf.utils.NumberUtils;
import com.alibaba.fastjson.JSON;
import com.google.common.cache.CacheBuilder;
import com.google.common.cache.CacheLoader;
import com.google.common.cache.LoadingCache;
import com.google.common.util.concurrent.ListenableFuture;
import com.google.common.util.concurrent.ListenableFutureTask;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.codec.digest.DigestUtils;
import org.apache.commons.collections.MapUtils;
import org.apache.commons.lang.StringUtils;
import org.jetbrains.annotations.NotNull;
import org.springframework.stereotype.Service;

import javax.annotation.Resource;
import java.util.*;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;

/**
 * @Description:
 * @author: 阿海
 * @create: 2020-07-28 17:29
 * 1. 根据key ===>  获取对应的内容
 * keyType>0可以：
 * 1. 根据key，资源id ===>  获取对应的内容
 * 2. 根据key，资源id ，用户标志 ====> 判断是否命中分流
 */
@Service
@Slf4j
public class ApolloPanGuServiceImpl implements ApolloPanGuService {

    @Resource
    private RemoteApolloKeyValueService remoteApolloKeyValueService;

    @Resource
    private ExecutorService executorService;

    /**
     * 全链路日志追加：
     * （1）分流的关键标志：0未命中分流，>=1命中分流
     * {"abtflag":{"sub.test.flag":1, "sub.advert.flag":0}}
     * （2）分流的后业务是否走降级，只打降级的：1降级
     * {"abthytrix":{"sub.test.flag":0, "sub.advert.flag":0}}
     * （3）【待定】层域划分，打了abtno=1的标记的后继链路，全部不能走分流
     */
    public static String AB_TEST_FLAG = "abtflag";
    public static String AB_TEST_HYTRIX = "abthytrix";
    public static String AB_TEST_NO = "abtno";

    /**
     * @param keyStr 关键字keyStr
     * @return 获取Apollo配置的值
     * （1）配置不存在，返回null
     * （2）配置存在返回正常的
     */
    @Override
    public String getIdMapStrByKeyStr(String keyStr) {
        return APOLLO_PANGU_STRING_CACHE.getUnchecked(keyStr);
    }

    /**
     * [{ids:'8420,8417',value:'20'},{ids:'8416', value:'30'}]
     * 【在配置序列内】返回为null
     *
     * @param keyStr     关键字
     * @param resourceId 资源Id
     * @return 从Map中解析===>获取配置的值,如30
     */
    @Override
    public String getIdMapStrByKeyStrResourceId(String keyStr, String resourceId) {
        Map<String, String> map = APOLLO_PANGU_MAP_CACHE.getUnchecked(keyStr);
        // {'8420':'20','8417':'20','8416':'30'}
        if (MapUtils.isEmpty(map)) {
            return null;
        }
        return map.get(resourceId);
    }

    /**
     * 【错误配置】返回为null
     *
     * @param keyStr
     * @return 判断当前配置是否为1，一般为开关使用
     */
    @Override
    public Boolean getIdMapStrIsOne(String keyStr) {
        String unchecked = APOLLO_PANGU_STRING_CACHE.getUnchecked(keyStr);
        if (StringUtils.isEmpty(unchecked)) {
            return false;
        }
        boolean flag = unchecked.equals("1");
        log.info("Apollo盘古配置，getIdMapStrIsOne,{}，boolean，{}", keyStr, flag);
        return flag;
    }

    /**
     *
     * 当前分流支持：
     * （1）正交分流，盐值不同
     * （2）同层互斥，GroupNumber默认为1，GroupNumber=2代表可以等分3份流量A/B/C测试
     * （3）固定的坑位，广告位，活动，皮肤，广告，这样子后期就不用分流了
     *
     * @param keyStr
     * @param resourceId
     * @param deviceId
     * @param groupNumber 额外分组的数量（比如分流20%，分组数是3，将分流为3*6=18%）
     * @return; 返回值
     *
     * （1）【0】默认是返回不分流
     * （2）【0】配置了未命中分流
     * （3）【1】配置了命中分流
     * （4）【2】配置了命中分流 xxx版本
     *
     */
    @Override
    public int getIdMapStrDeviceIdHashFlag(String keyStr, String resourceId, String deviceId, Integer groupNumber) {

        // 1. RPC获取分流比例,如20%==>纯数字20
        String valueFromApollo = getIdMapStrByKeyStrResourceId(keyStr, resourceId);
        if (StringUtils.isEmpty(valueFromApollo)) {
            return 0;
        }
        if (!NumberUtils.isNumeric(valueFromApollo)) {
            log.error("Apollo盘古配置，正交分流返回非数字,不命中分流,{}", valueFromApollo);
            return 0;
        }

        // 2. 分流的举例比例，uv+盐值的hash
        int ratioApollo = Integer.valueOf(valueFromApollo);

        // 3. 获取分流结果：关键字+设备号+分组+分流比例
        int deviceHashShunt = 0;
        try {
            deviceHashShunt = sGetDeviceHashShunt(keyStr, deviceId, groupNumber, ratioApollo);
        } catch (Exception e) {
            log.error("sGetDeviceHashShunt 分流异常：", e);
        }

        return deviceHashShunt;
    }

    /**
     * 获取分流结果：关键字+设备号+分组+分流比例
     *
     * @param keyStr
     * @param deviceId
     * @param groupNumber
     * @param ratioApollo
     * @return 分流结果关键字：0未命中分流，1命中分流策略1,而2命中分流策略2
     */
    public static int sGetDeviceHashShunt(String keyStr, String deviceId, Integer groupNumber, int ratioApollo) {

        // 1. 【分流Hash】加盐值，防止分流异常，比如hash值是15
        final int valueHash = sGetHashValue(keyStr, deviceId);

        // 2. 【未命中】分流，余数hash值30，分流比例20
        if (ratioApollo <= 0 || valueHash >= ratioApollo) {
            return 0;
        }

        // 3.【命中分流】不需要额外的分组，GroupNumber只有1份
        if (null == groupNumber || groupNumber <= 1) {
            return 1;
        }


        // 4. 【复杂应用】场景，比如算法分流20%==>，其中又要55分流，10%返回1（如9）,10%返回2（如15）
        if (ratioApollo >= 100) {
            ratioApollo = 100;
        }
        // 5. GroupNumber 是2份，每份为10
        int step = ratioApollo / groupNumber;
        for (int i = 0; i < groupNumber; i++) {
            // step=10，表示ratioApollo是否在0~10之内,0在前面已经返回掉了
            if (step * i <= valueHash && valueHash < step * (i + 1)) {
                return i + 1;
            }
        }

        return 0;
    }

    /**
     * 关键字和设备号获取hash值，对100求余数
     *
     * @param keyStr
     * @param deviceId
     * @return 返回0~99的hash余数
     */
    public static int sGetHashValue(String keyStr, String deviceId) {
        String HashStr = DigestUtils.md5Hex(deviceId + keyStr);
        int hash = HashAlgorithm.dekHash(HashStr);
        return (hash < 0 ? -hash : hash) % 100;
    }


    /**
     * -------------------------------------------------------------------------------------
     * 【Apollo缓存1】：keyStr=====>直接获取内容
     * 推荐使用：keyType=0的场景
     * 返回值，如开关的：1
     */
    private CacheLoader<String, String> APOLLO_STRING_LOADER = new CacheLoader<String, String>() {
        @Override
        public String load(String keyStr) {
            if (StringUtils.isEmpty(keyStr)) {
                return "";
            }
            String keyIdMapStr = getKeyIdMapStr(keyStr);
            log.info("APOLLO_STRING_LOADER缓存查询，keyStr，{}，keyIdMapStr，{}", keyStr, keyIdMapStr);
            return keyIdMapStr;
        }

        @Override
        public ListenableFuture<String> reload(String keyStr, String oldValue) {
            ListenableFutureTask<String> task = ListenableFutureTask.create(() -> load(keyStr));
            executorService.submit(task);
            return task;
        }
    };

    private LoadingCache<String, String> APOLLO_PANGU_STRING_CACHE = CacheBuilder.newBuilder()
            .initialCapacity(1024).concurrencyLevel(20).refreshAfterWrite(1,
                    TimeUnit.SECONDS).expireAfterWrite(20,
                    TimeUnit.HOURS).build(APOLLO_STRING_LOADER);

    /**
     * 【Apollo缓存2】：keyStr=====>直接获取内容转化后的HashMap
     * 推荐使用：keyType>0的场景
     * {'8420':'20','8417':'20','8416':'30'}
     *
     * @Param 如广告位Id，以及对应的分流比例
     */
    private CacheLoader<String, Map<String, String>> APOLLO_MAP_LOADER = new CacheLoader<String, Map<String, String>>() {
        @Override
        public Map<String, String> load(@NotNull String keyStr) {
            // 【RPC接口】盘古处理，根据 keyStr 获取value信息==>转化成Map
            return getStringMapFromRPC(keyStr);
        }

        @Override
        public ListenableFuture<Map<String, String>> reload(String keyStr, Map<String, String> oldValue) {
            ListenableFutureTask<Map<String, String>> task = ListenableFutureTask.create(() -> load(keyStr));
            executorService.submit(task);
            return task;
        }
    };

    private LoadingCache<String, Map<String, String>> APOLLO_PANGU_MAP_CACHE = CacheBuilder.newBuilder()
            .initialCapacity(1024).concurrencyLevel(20).refreshAfterWrite(1,
                    TimeUnit.SECONDS).expireAfterWrite(20,
                    TimeUnit.HOURS).build(APOLLO_MAP_LOADER);

    /**
     * 【RPC接口】盘古处理，根据 keyStr 获取value信息==>转化成Map
     *
     * @param keyStr
     * @return 返回必须不能是null
     */
    private Map<String, String> getStringMapFromRPC(String keyStr) {
        Map<String, String> keyIdMap = new HashMap<>();
        if (StringUtils.isEmpty(keyStr)) {
            return keyIdMap;
        }
        // 1.【RPC】接口查询
        String keyIdMapStr = getKeyIdMapStr(keyStr);
        if (StringUtils.isEmpty(keyIdMapStr)) {
            log.error("getStringMapFromRPC 查询为空，keyStr，{}", keyStr);
            return keyIdMap;
        }

        // 2. 数据格式转化
        log.info("getStringMapFromRPC 查询结果OK，keyStr，{}，keyIdMapStr，{}", keyStr, keyIdMapStr);
        // {'8420':'20','8417':'20','8416':'30'}
        try {
            Map keyIdMapTemp = (Map) JSON.parse(keyIdMapStr);
            for (Object key : keyIdMapTemp.keySet()) {
                keyIdMap.put(key + "", keyIdMapTemp.get(key) + "");
            }
        } catch (Exception e) {
            log.error("getStringMapFromRPC 缓存转换失败", e);
        }

        return keyIdMap;
    }


    /**
     * 【RPC接口】盘古，根据 keyStr 获取value信息
     * 配置存在返回具体
     */
    private String getKeyIdMapStr(String keyStr) {
        ApolloKeyValueDTO apolloKeyValueDTO = new ApolloKeyValueDTO();
        apolloKeyValueDTO.setKeyStr(keyStr);
        apolloKeyValueDTO.setKeyOnOff(1);
        apolloKeyValueDTO = remoteApolloKeyValueService.selectApolloKeyValue(apolloKeyValueDTO);
        return null == apolloKeyValueDTO ? "" : apolloKeyValueDTO.getKeyIdMapStr();
    }

    /**
     * 分流测试--------------------------------------------------
     * 1. 随机用户1~10000的100个用户设备号,pv访问
     * 2. 走了DPA组件化素材，分流20%进行AB测
     * 3. 走了DPA组件化活动，分流30%进行AB测（或者ABC）
     */

    public static void main(String[] args) {
        // 1. 随机用户1~1000的100个用户
        List<Integer> userList = new ArrayList<>();
        for (int i = 0; i < 1000; i++) {
            int randomUserId = (int) (Math.random() * 1000000);
            userList.add(randomUserId);
        }
        System.out.println(userList);

        // 2. 走了DPA组件化素材，分流20%进行AB测
        Map<String, Set<Integer>> mapSck = new HashMap<>();
        userList.forEach(dto -> {
            Integer dpaActivity = sGetDeviceHashShunt("DPA素材1", dto + "", 2, 50);
            Set<Integer> integers = mapSck.get(dpaActivity + "");
            if (integers == null) {
                integers = new HashSet<>();
            }
            integers.add(dto);
            mapSck.put(dpaActivity + "", integers);
        });
        System.out.println("DPA素材 ---------------------");
        Set<Integer> sckSet0 = mapSck.get("0");
        Set<Integer> sckSet1 = mapSck.get("1");
        Set<Integer> sckSet2 = mapSck.get("2");
        System.out.println("未分流，命中数：" + sckSet0.size());
        System.out.println("分流1，命中数：" + sckSet1.size());
        System.out.println("分流2，命中数：" + sckSet2.size());

        // 3. 走了DPA组件化活动，分流30%进行ABC测
        Map<String, Set<Integer>> mapAct = new HashMap<>();
        userList.forEach(dto -> {
            Integer dpaActivity = sGetDeviceHashShunt("DPA活动2", dto + "", 2, 50);
            Set<Integer> integers = mapAct.get(dpaActivity + "");
            if (integers == null) {
                integers = new HashSet<>();
            }
            integers.add(dto);
            mapAct.put(dpaActivity + "", integers);
        });
        System.out.println("DPA活动 ---------------------");
        Set<Integer> actSet0 = mapAct.get("0");
        Set<Integer> actSet1 = mapAct.get("1");
        Set<Integer> actSet2 = mapAct.get("2");
        System.out.println("未分流，命中数：" + actSet0.size());
        System.out.println("分流1，命中数：" + actSet1.size());
        System.out.println("分流2，命中数：" + actSet2.size());

        System.out.println("DPA素材，是否包含DPA活动 --------------------");
        System.out.println("DPA素材分流1，中包含DPA活动分流0的数量：" + actSet1.stream().filter(sckSet0::contains).collect(Collectors.toList()).size());
        System.out.println("DPA素材分流1，中包含DPA活动分流1的数量：" + actSet1.stream().filter(sckSet1::contains).collect(Collectors.toList()).size());
        System.out.println("DPA素材分流1，中包含DPA活动分流2的数量：" + actSet1.stream().filter(sckSet2::contains).collect(Collectors.toList()).size());

    }
}
