package cn.com.duiba.tuia.purchase.web.api.localservice;

import cn.com.duiba.tuia.purchase.web.api.dto.PbpProductDto;
import cn.com.duiba.tuia.purchase.web.api.dto.PbpTaskDto;
import cn.com.duiba.tuia.purchase.web.api.dto.PutPlanDto;
import cn.com.duiba.tuia.purchase.web.api.remoteservice.RemotePurchaseService;
import com.github.benmanes.caffeine.cache.Caffeine;
import com.github.benmanes.caffeine.cache.LoadingCache;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang.StringUtils;
import org.apache.commons.lang.math.RandomUtils;
import org.apache.commons.lang3.ObjectUtils;
import org.springframework.stereotype.Service;

import javax.annotation.Resource;
import java.util.Collections;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.concurrent.TimeUnit;

@Service
@Slf4j
public class OcpcPurchaseService {

    @Resource
    private RemotePurchaseService remotePurchaseService;


    private LoadingCache<Long, Optional<PutPlanDto>> caffeine = Caffeine.newBuilder()
            .maximumSize(6000)
            .initialCapacity(20)
            .refreshAfterWrite(5, TimeUnit.SECONDS)
            .expireAfterWrite(20, TimeUnit.SECONDS)
            .build(this::queryByChannelId);

    private LoadingCache<Long, Optional<PbpProductDto>> productCaffeineCache = Caffeine.newBuilder()
            .maximumSize(6000)
            .initialCapacity(20)
            .refreshAfterWrite(5, TimeUnit.SECONDS)
            .expireAfterWrite(20, TimeUnit.SECONDS)
            .build(this::queryByTaskId);


    public Optional<PutPlanDto> getByChannelId(Long channelId) {
        return caffeine.get(channelId);
    }

    public Optional<PbpProductDto> getByTaskId(Long taskId) {
        return productCaffeineCache.get(taskId);
    }

    /**
     * 根据渠道号查询对应的计划 不存在则返回空计划
     *
     * @param channelId
     * @return
     */
    private Optional<PutPlanDto> queryByChannelId(Long channelId) {
        PutPlanDto putPlanDto = remotePurchaseService.getPlanByChannelId(channelId);
        //加一层空对象缓存 防止恶意请求
        if (putPlanDto == null) {
            return Optional.empty();
        }
        return Optional.of(putPlanDto);
    }

    private Optional<PbpProductDto> queryByTaskId(Long taskId) {
        PbpTaskDto pbpTaskDto = remotePurchaseService.getTaskById(taskId);
        //加一层空对象缓存 防止恶意请求
        if (Objects.isNull(pbpTaskDto)) {
            return Optional.empty();
        }
        PbpProductDto pbpProductDto = remotePurchaseService.getProductById(pbpTaskDto.getProductId());
        //加一层空对象缓存 防止恶意请求
        if (Objects.isNull(pbpProductDto)) {
            return Optional.empty();
        }
        return Optional.of(pbpProductDto);
    }

    public String getEventType(Long channelId, String eventType) {
        Optional<PutPlanDto> optional = getByChannelId(channelId);
        if (!optional.isPresent()) {
            log.info("not found channelId {}", channelId);
            return null;
        }
        PutPlanDto putPlan = optional.get();
        Long putPlanId = putPlan.getId();
        if (!putPlan.getIsCallBack()) {
            log.info("the callback configuration is not enabled：channelId is {}，planId is {}，eventType is {}",
                    channelId, putPlanId, eventType);
            return null;
        }
        Map<String, String> actionIdMap = ObjectUtils.defaultIfNull(putPlan.getActionIdMap(), Collections.emptyMap());
        if (!actionIdMap.containsKey(eventType)) {
            log.info("tuia event type not supported：channelId is {}，planId is {}，eventType is {}", channelId, putPlanId, eventType);
            return null;
        }
        if (StringUtils.isBlank(actionIdMap.get(eventType))) {
            log.info("media event type is not defined：channelId is {}，planId is {}，eventType is {}", channelId, putPlanId, eventType);
            return null;
        }
        //概率没命中 万分比
        int randomNumber = RandomUtils.nextInt(10000);
        if (randomNumber >= putPlan.getProbabilityBack()) {
            log.info("概率回传未命中：channelId is {}，planId is {}，eventType is {}，PR is {}, randomNumber is {}",
                    channelId, putPlanId, eventType, putPlan.getProbabilityBack(), randomNumber);
            return null;
        }
        return actionIdMap.get(eventType);
    }

}

