package cn.com.tuia.advert.message.consumer;

import java.util.*;
import java.util.Map.Entry;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ExecutorService;

import javax.annotation.Resource;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.BeansException;
import org.springframework.context.ApplicationContext;
import org.springframework.context.ApplicationContextAware;
import org.springframework.data.redis.connection.Message;
import org.springframework.data.redis.connection.MessageListener;
import org.springframework.data.redis.core.RedisTemplate;

import com.google.common.collect.Lists;

import cn.com.tuia.advert.message.RedisMessageChannel;

/**
 * redis消息接受
 *
 * @author ZhouFeng zhoufeng@duiba.com.cn
 * @version $Id: RedisMsgSubscriber.java , v 0.1 2017/3/16 下午8:45 ZhouFeng Exp $
 */
public class RedisMsgSubscriber implements MessageListener, ApplicationContextAware {

    private static Logger    log = LoggerFactory.getLogger(RedisMsgSubscriber.class);

    @Resource
    private ExecutorService executorService;
    
    @Resource
    private RedisTemplate<String, String> redisTemplate;

    private static final Map<RedisMessageChannel, List<AbstractRedisMessageHandle>> handlesMap = new ConcurrentHashMap<>();

    private RedisMsgSubscriber(){
        //redis private subscriber
    }
    
    @Override
    public void setApplicationContext(ApplicationContext applicationContext) throws BeansException {
        // 从IOC中取出所有AbstractRedisMessageHandle对象，根据其注解建立与消息的映射关系
        Map<String, AbstractRedisMessageHandle> map = applicationContext.getBeansOfType(AbstractRedisMessageHandle.class);
        for(Entry<String, AbstractRedisMessageHandle> entry : map.entrySet()){
            AbstractRedisMessageHandle handle = entry.getValue();
            RedisMessageHandle annotation = handle.getClass().getAnnotation(RedisMessageHandle.class);
            if (annotation == null) {
                continue;
            }
            RedisMessageChannel channel = annotation.value();
            List<AbstractRedisMessageHandle> handleList = handlesMap.get(channel);
            if(null == handleList){
                handleList = Lists.newArrayList();
                handlesMap.put(channel, handleList);
            }
            handleList.add(handle);
        }
    }

    @Override
    public void onMessage(Message message, byte[] pattern) {

        byte[] body = message.getBody();// 请使用valueSerializer
        byte[] channel = message.getChannel();
        // 请参考配置文件，本例中key，value的序列化方式均为string。
        // 其中key必须为stringSerializer。和redisTemplate.convertAndSend对应
        String msgContent = (String) redisTemplate.getValueSerializer().deserialize(body);
        String topic = redisTemplate.getStringSerializer().deserialize(channel);

        List<AbstractRedisMessageHandle> handleList = null;

        RedisMessageChannel channelEnum = RedisMessageChannel.getByChannel(topic);

        if (channelEnum == null) {
            log.error("channel:[{}] not dedeclared", channel);
            return;
        }

        if (handlesMap != null && !handlesMap.isEmpty()) {
            handleList = handlesMap.get(channelEnum);
        }

        if (handleList == null) {
            log.error("not find handle of channel:[{}]", channel);
            return;
        }
        for(AbstractRedisMessageHandle handle : handleList){
            executorService.submit(handle.createHandle(channelEnum, msgContent));
        }
    }
}
