package cn.com.duiba.wolf.dubbo;

import java.net.URL;
import java.security.CodeSource;
import java.util.Enumeration;
import java.util.HashSet;
import java.util.Set;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * Created by inter12 on 12/5/14.
 */
public class VersionCheck {

    private VersionCheck() {
    }

    private static final String VERSION = getVersion(VersionCheck.class, "1.1.5");

    private static final Logger logger  = LoggerFactory.getLogger(VersionCheck.class);

    static {
        // 检查是否存在重复的jar包
        VersionCheck.checkDuplicate(VersionCheck.class);
    }

    public static String getVersion() {
        return VERSION;
    }

    public static void checkDuplicate(Class<?> cls) {
        checkDuplicate(cls.getName().replace('.', '/') + ".class");
    }

    public static String getVersion(Class<?> cls, String defaultVersion) {
        try {
            // 首先查找MANIFEST.MF规范中的版本号
            String version = cls.getPackage().getImplementationVersion();
            if (version == null || version.length() == 0) {
                version = cls.getPackage().getSpecificationVersion();
            }
            if (version == null || version.length() == 0) {
                // 如果规范中没有版本号，基于jar包名获取版本号
                CodeSource codeSource = cls.getProtectionDomain().getCodeSource();
                if (codeSource == null) {
                    logger.info("No codeSource for class {} when getVersion, use default version {}" ,cls.getName(), defaultVersion);
                } else {
                    String file = codeSource.getLocation().getFile();
                    if (file != null && file.length() > 0 && file.endsWith(".jar")) {
                        file = file.substring(0, file.length() - 4);
                        int i = file.lastIndexOf('/');
                        if (i >= 0) {
                            file = file.substring(i + 1);
                        }
                        i = file.indexOf('-');
                        if (i >= 0) {
                            file = file.substring(i + 1);
                        }
                        while (file.length() > 0 && !Character.isDigit(file.charAt(0))) {
                            i = file.indexOf('-');
                            if (i >= 0) {
                                file = file.substring(i + 1);
                            } else {
                                break;
                            }
                        }
                        version = file;
                    }
                }
            }
            // 返回版本号，如果为空返回缺省版本号
            return version == null || version.length() == 0 ? defaultVersion : version;
        } catch (Exception e) { // 防御性容错
            // 忽略异常，返回缺省版本号
            logger.error("return default version, ignore exception " + e.getMessage(), e);
            return defaultVersion;
        }
    }

    public static void checkDuplicate(String path) {
        try {
            // 在ClassPath搜文件
            Enumeration<URL> urls = VersionCheck.class.getClassLoader().getResources(path);
            Set<String> files = new HashSet<>();
            while (urls.hasMoreElements()) {
                URL url = urls.nextElement();
                if (url != null) {
                    String file = url.getFile();
                    if (file != null && file.length() > 0) {
                        files.add(file);
                    }
                }
            }
            // 如果有多个，就表示重复
            if (files.size() > 1) {
                logger.error("Duplicate class {} in {} jar {}", path, files.size(), files);
            }
        } catch (Exception e) { // 防御性容错
            logger.error(e.getMessage(), e);
        }
    }

}
