package cn.com.duiba.wolf.cache;

import cn.com.duiba.wolf.cache.exception.RuntimeMemcachedException;
import cn.com.duiba.wolf.log.DegradeLogger;
import cn.com.duiba.wolf.utils.CatUtils;
import cn.com.duiba.wolf.utils.NumberUtils;
import net.rubyeye.xmemcached.*;
import net.rubyeye.xmemcached.auth.AuthInfo;
import net.rubyeye.xmemcached.command.BinaryCommandFactory;
import net.rubyeye.xmemcached.command.KestrelCommandFactory;
import net.rubyeye.xmemcached.command.TextCommandFactory;
import net.rubyeye.xmemcached.exception.MemcachedException;
import net.rubyeye.xmemcached.transcoders.SerializingTranscoder;
import net.rubyeye.xmemcached.utils.AddrUtil;

import org.apache.commons.lang.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.DisposableBean;
import org.springframework.beans.factory.InitializingBean;

import java.io.IOException;
import java.io.Serializable;
import java.lang.reflect.Field;
import java.net.InetSocketAddress;
import java.util.*;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;

/**
 * CacheClient的一种实现,xmemcached客户端
 * @author huangwq
 */
public class XMemcacheClient implements InitializingBean,DisposableBean, CacheClient {

    private static Logger           logger                     = DegradeLogger.wrap(LoggerFactory.getLogger(XMemcacheClient.class));

    /** 缓存类的名称 */
    private final String            CACHE_NAME                 = XMemcacheClient.class.getName();

    /** 表示缓存永不失效 */
    private static final int        CACHE_NO_EXPIRY            = 0;

    /** 计数器的默认初始值 */
    private static final int        COUNTER_DEFAULT_INIT_VALUE = 0;

    private int                     connectionPoolSize;
    private int                     connectionTimeout          = 5000;
    private int                     operationTimeout           = 300;//(int) MemcachedClient.DEFAULT_OP_TIMEOUT;

    // 默认保存2天
    private static final int        DEFAULT_EXP                = 3600 * 24 * 2;

    private MemcachedSessionLocator sessionLocator;
    private MemcachedClient         memcachedClient;

    private String                  servers;
    private String                  authInfos;
    private String                  username;
    private String                  password;
    private boolean enableHeartBeat;
    //TEXT/BINARY
    private String protocol;
    
    @Override
	public boolean add(String key, int exp, Object value) throws Exception{
		return memcachedClient.add(key, exp, value);
	}

    @Override
    public <T> T get(String key) {
        T t = getInner(key);
        if(t != null && t instanceof NullCache){//如果取回的数据是NullCache,表示是之前故意放入的,特殊处理,返回null
            return null;
        }
        return t;
    }

    private <T> T getInner(String key) {
        key = getStringNoBlank(key);
        try {
            return memcachedClient.get(key);
        } catch (IllegalArgumentException e) {
            logger.error(e.getMessage() + ",key is:" + key);
            throw new RuntimeMemcachedException(e);
        } catch (Exception e) {
            logger.error("",e);
            CatUtils.logError(e);
            return null;
        }
    }

    @Override
    public long getLong(String key) {
        key = getStringNoBlank(key);
        try {
            Object obj = memcachedClient.get(key);
            if (obj == null) {
                return 0;
            } else if (obj instanceof String) {
                return NumberUtils.parseLong((String) obj, 0);
            } else if (obj instanceof Integer) {
                return (Integer) obj;
            } else if (obj instanceof Long) {
                return (Long) obj;
            }
        } catch (IllegalArgumentException e) {
            logger.error(e.getMessage() + ",key is:" + key);
            throw new RuntimeMemcachedException(e);
        } catch (Exception e) {
            throw new RuntimeMemcachedException(e);
        }
        return 0;
    }

    public Map<String, Object> get(Collection<String> keys) {
        Map<String, Object> value = null;
        try {
            value = memcachedClient.get(keys);
        } catch (Exception e) {
            throw new RuntimeMemcachedException(e);
        }
        return value;
    }

//    @Override
//    public void set(String key, Object value) {
//        set(key, value, DEFAULT_EXP, TimeUnit.SECONDS);
//    }

//    @Override
//    public boolean set(String key, Object value, int expSeconds) {
//        return set(key, value, expSeconds, TimeUnit.SECONDS);
//    }

    @Override
    public boolean set(String key, Object value, int expiry, TimeUnit unit) {
        if (value == null) {
            return false;
        }
        key = getStringNoBlank(key);

        int exp = getValidExpiryTime(expiry, unit);
        try {
            return memcachedClient.set(key, exp, value);
        } catch (Exception e) {
            CatUtils.logError(e);
            //throw new RuntimeMemcachedException(e);
            logger.error("",e);
            return false;
        }
    }

    @Override
    public <T> T getWithCacheLoader(String key, int exp, TimeUnit timeUnit, CacheLoader<T> cacheLoader){
        return getWithCacheLoader(key, exp, timeUnit, false, cacheLoader);
    }

    @Override
    public <T> T getWithCacheLoader(String key, int exp, TimeUnit timeUnit, boolean isCacheNull ,CacheLoader<T> cacheLoader){
        T value = getInner(key);
        if(value == null){
            value = cacheLoader.load();
            if(isCacheNull) {
                setWithNull(key, value, exp, timeUnit);
            }else if(value != null){
                set(key, value, exp, timeUnit);
            }
        }

        if(value instanceof NullCache){//如果取回的数据是NullCache,表示是之前故意放入的,特殊处理,返回null
            value = null;
        }

        return value;
    }

    private void setWithNull(String key, Object value, int expiry, TimeUnit unit) {
        if (value == null) {
            value = NullCache.NULL;
        }
        set(key, value, expiry, unit);
    }

    @Override
    public long incr(String key, long delta) {
        return incr(key, delta, delta, DEFAULT_EXP, TimeUnit.MILLISECONDS);
    }

    @Override
    public long incr(String key, long delta, long expiry, TimeUnit unit) {
        return incr(key, delta, delta, expiry, unit);
    }

    @Override
    public long incr(String key, long delta, long initValue, long expiry, TimeUnit unit) {
        // long count = COUNTER_DEFAULT_INIT_VALUE;

        int exp = getValidExpiryTime(expiry, unit);
        try {
            // count =
            long count = memcachedClient.incr(key, delta, initValue, MemcachedClient.DEFAULT_OP_TIMEOUT, exp);
            logger.debug("key:{},afterInce:{}", key, count);
            return count;
        } catch (Exception e) {
            //logger.error("memcachedClient incr() error,key:" + key + this.get(key), e);
            throw new RuntimeMemcachedException(e);
        }
    }

    @Override
    public long decr(String key, long delta) {
        return decr(key, delta, COUNTER_DEFAULT_INIT_VALUE, DEFAULT_EXP, TimeUnit.MILLISECONDS);
    }

    @Override
    public long decr(String key, long delta, long expiry, TimeUnit unit) {
        return decr(key, delta, COUNTER_DEFAULT_INIT_VALUE, expiry, unit);
    }

    @Override
    public long decr(String key, long delta, long initValue, long expiry, TimeUnit unit) {
        // long count = COUNTER_DEFAULT_INIT_VALUE;

        int exp = getValidExpiryTime(expiry, unit);
        try {
            return memcachedClient.decr(key, delta, initValue, MemcachedClient.DEFAULT_OP_TIMEOUT, exp);
        } catch (Exception e) {
            logger.error("memcachedClient decr() error,key:" + key, e);
            throw new RuntimeMemcachedException(e);
        }
    }

    @Override
    public <T> GetsResponse<T> gets(String key){
        try {
            return memcachedClient.gets(key);
        } catch (Exception e) {
            throw new RuntimeMemcachedException(e);
        }
    }

    @Override
    public <T> Map<String, GetsResponse<T>> gets(Collection<String> keys){
        try {
            return memcachedClient.gets(keys);
        } catch (Exception e) {
            throw new RuntimeMemcachedException(e);
        }
    }

    @Override
    public <T> boolean cas(String key, CASOperation<T> casOperation){
        try {
            return memcachedClient.cas(key, casOperation);
        } catch (Exception e) {
            throw new RuntimeMemcachedException(e);
        }
    }

    @Override
    public <T> boolean cas(String key, int expSeconds, CASOperation<T> casOperation){
        try {
            return memcachedClient.cas(key, expSeconds, casOperation);
        } catch (Exception e) {
            throw new RuntimeMemcachedException(e);
        }
    }

    @Override
    public boolean cas(String key, int exp, TimeUnit timeUnit, Object value, long currentCas){
        try {
            int expSeconds = getValidExpiryTime(exp, timeUnit);
            return memcachedClient.cas(key,expSeconds,value,currentCas);
        } catch (Exception e) {
            throw new RuntimeMemcachedException(e);
        }
    }

    @Override
    public boolean casByValue(String key, Object expectedValue, Object newValue, int expSeconds){
        try {
            GetsResponse<Object> t = memcachedClient.gets(key);
            if(t == null){//key not exists
                return false;
            }
            if(t.getValue().equals(expectedValue)) {
                return memcachedClient.cas(key, expSeconds, newValue, t.getCas());
            }
        } catch (Exception e) {
            throw new RuntimeMemcachedException(e);
        }
        return false;
    }

    @Override
    public boolean remove(String key) {
        try {
            return memcachedClient.delete(getStringNoBlank(key));
        } catch (Exception e) {
            //throw new RuntimeMemcachedException(e);
            CatUtils.logError(e);
            logger.error("",e);
            return false;
        }
    }

    @Override
    public boolean remove(String... keys) {
        throw new UnsupportedOperationException();
    }

    public void setSessionLocator(MemcachedSessionLocator sessionLocator) {
        this.sessionLocator = sessionLocator;
    }

    public int getConnectionPoolSize() {
        return connectionPoolSize;
    }

    public void setConnectionPoolSize(int connectionPoolSize) {
        this.connectionPoolSize = connectionPoolSize;
    }

    public int getConnectionTimeout() {
        return connectionTimeout;
    }

    public void setConnectionTimeout(int connectionTimeout) {
        this.connectionTimeout = connectionTimeout;
        if (memcachedClient != null) {
            memcachedClient.setConnectTimeout(connectionTimeout);
        }
    }

    public int getOperationTimeout() {
        return operationTimeout;
    }

    public void setOperationTimeout(int operationTimeout) {
        this.operationTimeout = operationTimeout;
        if (memcachedClient != null) {
            memcachedClient.setOpTimeout(operationTimeout);
        }
    }

    public String getServers() {
        return servers;
    }

    public void setServers(String servers) {
        this.servers = servers;
    }

    public String getAuthInfos() {
        return authInfos;
    }

    public void setAuthInfos(String authInfos) {
        this.authInfos = authInfos;
    }



//    private Integer[] getWeights(int serverCount) {
//        Integer[] weights = new Integer[serverCount];
//        for (int i = 0; i < serverCount; i++) {
//            weights[i] = 1;
//        }
//        return weights;
//    }

    /**
     * @Title: getStringNoBlank
     * @Description: 清空字符串内空格
     * @param @param str
     * @param @return 设定文件
     * @author yulc
     * @return String 返回类型
     * @throws
     */
    public static String getStringNoBlank(String str) {
        if (str != null && !"".equals(str)) {
            return str.replaceAll("\\s*|\t|\r|\n", "").replaceAll("　", "").replaceAll("&nbsp;", "");
        } else {
            return str;
        }
    }

    private void initClient() {
        if (memcachedClient != null) {
            logger.info("{} already initialized", CACHE_NAME);
            return;
        }

        String[] servers = StringUtils.split(this.getServers(), ",");
        String[] authInfos = StringUtils.split(this.getAuthInfos(), ",");
        List<InetSocketAddress> addressList = new ArrayList<>();

        Map<InetSocketAddress, AuthInfo> authInfoMap = new HashMap<>();
        int i = 0;
        for (String server : servers) {
            InetSocketAddress address = AddrUtil.getOneAddress(server);
            addressList.add(address);
            if(authInfos != null && authInfos.length > 0) {//优先使用authInfos中配置的用户名和密码,如果没有再使用username、password配置
                if (authInfos != null && authInfos.length > i) {
                    String authInfo = authInfos[i];
                    String[] temp = authInfo.split(":");
                    if (temp.length >= 1) {
                        authInfoMap.put(address, AuthInfo.plain(temp[0], temp[1]));
                        logger.info("auth {} with username:{},password:{}", server, temp[0], temp[1]);
                    }
                }
            }else if(!StringUtils.isBlank(username) && !StringUtils.isBlank(password)){
                authInfoMap.put(address, AuthInfo.plain(username, password));
            }
            i++;
        }

        MemcachedClientBuilder builder = new XMemcachedClientBuilder(addressList);// weights
        builder.setAuthInfoMap(authInfoMap);

        if (sessionLocator != null) {
            builder.setSessionLocator(sessionLocator);
        }

        SerializingTranscoder transcoder = new SerializingTranscoder();
        transcoder.setCompressionThreshold(1024);
        builder.setTranscoder(transcoder);

        //TEXT/BINARY 默认BINARY
        if (StringUtils.isBlank(protocol) || "BINARY".equals(protocol)) {
            builder.setCommandFactory(new BinaryCommandFactory());
        }else if("TEXT".equals(protocol)){
            builder.setCommandFactory(new TextCommandFactory());
        }else if("KESTREL".equals(protocol)){
            builder.setCommandFactory(new KestrelCommandFactory());
        }
        builder.setKeyProvider(new KeyProvider() {

            @Override
            public String process(String key) {
                if (key == null) {
                    return null;
                }
                return key.trim();// "prefix_" +
            }
        });
        builder.setHealSessionInterval(2000);//连接断开两秒后尝试重连
        // builder.setSessionLocator(new KetamaMemcachedSessionLocator());// 一致性哈希（consistent hash)
        // builder.setSessionLocator(new ElectionMemcachedSessionLocator());
        // builder.setTranscoder(new SerializingTranscoder());
        //builder.getConfiguration().setCheckSessionTimeoutInterval();
        //builder.getConfiguration().setSessionIdleTimeout();

        // builder.getTranscoder().setPackZeros(true);
        // builder.getTranscoder().setPrimitiveAsString(false);

        // memcachedClient.setOptimizeGet(true);
        // memcachedClient.setOptimizeMergeBuffer(true);
        // memcachedClient.setPrimitiveAsString(false);
        // memcachedClient.setSanitizeKeys(false);

        if (connectionPoolSize > 0) {
            builder.setConnectionPoolSize(connectionPoolSize);// 设置多个连接会有bug。1个足够了
        }
        // builder.setSocketOption(StandardSocketOption.SO_RCVBUF, 32 * 1024); // 设置接收缓存区为32K，默认16K
        // builder.setSocketOption(StandardSocketOption.SO_SNDBUF, 16 * 1024); // 设置发送缓冲区为16K，默认为8K

        try {
            memcachedClient = builder.build();

            memcachedClient.setConnectTimeout(connectionTimeout);
            memcachedClient.setOpTimeout(operationTimeout);
            memcachedClient.setEnableHeartBeat(enableHeartBeat);
            logger.info("{} initialized",addressList);
        } catch (IOException e) {
            logger.error("Initialize " + getClass() + " error", e);
            throw new RuntimeMemcachedException("Initialize " + getClass() + " error", e);
        }

        try {
            // 去除shutdownhook,以防kill程序时memcached先关闭了而导致的一些问题
            Field field = memcachedClient.getClass().getDeclaredField("shutdownHookThread");
            field.setAccessible(true);
            Thread shutdownHookThread = (Thread)field.get(memcachedClient);
            Runtime.getRuntime().removeShutdownHook(shutdownHookThread);
        } catch (Exception e) {
            logger.error("warn: getField:shutdownHookThread error", e);
        }
    }

    /**
     * 添加 Memcached 服务器.
     *
     * @param servers Memcached 服务器地址, 每个服务器的格式为: [host1]:[port1] [host2]:[port2]. 例如, 192.168.0.1:11211.
     * @throws IOException 如果在添加过程中失败, 会抛出此异常
     */
    public void addServer(String... servers) throws IOException {
        if (servers == null || servers.length == 0) {
            throw new IllegalArgumentException("servers can't be null");
        }

        StringBuilder hostList = new StringBuilder();
        for (String server : servers) {
            hostList.append(server).append(" ");
        }
        hostList.deleteCharAt(hostList.length() - 1);

        memcachedClient.addServer(hostList.toString());
        logger.info("Memcached server(s) added: [{}]", hostList);
    }

    /**
     * 添加Memcached服务器.
     *
     * @param servers Memcached服务器地址, 每个服务器的格式为: [host1]:[port1] [host2]:[port2]. 例如, 192.168.0.1:11211.
     * @param weights 服务器的权重, 例如：{1, 1, 2}
     * @throws IOException 如果在添加过程中失败, 会抛出此异常
     */
    public void addServers(String[] servers, int[] weights) throws IOException {
        if (servers == null || servers.length == 0) {
            throw new IllegalArgumentException("servers can't be null");
        }

        if (weights == null || (weights != null && weights.length < servers.length)) {
            throw new IllegalArgumentException("weights can't be less than servers");
        }

        for (int i = 0; i < servers.length; i++) {
            InetSocketAddress address = AddrUtil.getOneAddress(servers[i]);
            memcachedClient.addServer(address, weights[i]);
        }

    }

    /**
     * 删除 Memcached 服务器, server 的格式为: [ip]:[port]. 例如, 192.168.0.1:11211.
     *
     * @param servers Memcached 服务器地址
     * @throws IOException 如果在删除过程中失败, 会抛出此异常
     */
    public void removeServer(String... servers) {
        if (servers == null || servers.length == 0) {
            throw new IllegalArgumentException("servers can't be null");
        }

        StringBuilder hostList = new StringBuilder();
        for (String server : servers) {
            hostList.append(server).append(" ");
        }
        hostList.deleteCharAt(hostList.length() - 1);

        memcachedClient.removeServer(hostList.toString());
        logger.info("Memcached server(s) removed: [{}]",hostList);
    }

    private int getValidExpiryTime(long expiry, TimeUnit unit) {
        int exp = (int) unit.toSeconds(expiry);
        return (exp > CACHE_NO_EXPIRY) ? exp : DEFAULT_EXP;
    }

    @Override
    public void afterPropertiesSet() throws Exception {
        this.initClient();
    }

    @Override
    public void flushAll() {
        try {
            memcachedClient.flushAll();
        } catch (Exception e) {
            throw new RuntimeMemcachedException(e);
        }
    }

    public void setProtocol(String protocol) {
        this.protocol = protocol;
    }

    public void setEnableHeartBeat(boolean enableHeartBeat) {
        this.enableHeartBeat = enableHeartBeat;
    }

    public String getUsername() {
        return username;
    }

    public void setUsername(String username) {
        this.username = username;
    }

    public String getPassword() {
        return password;
    }

    public void setPassword(String password) {
        this.password = password;
    }

    @Override
    public void destroy() throws Exception {
        memcachedClient.shutdown();
    }

    /**
     * 表示缓存为空, 防止命中失效,如果把此对象的实例放入缓存。取回数据时会返回null
     *
     * Created by wenqi.huang on 16/7/6.
     */
    private static class NullCache implements Serializable {
        private static final long serialVersionUID = 8197695308667194378L;

        private NullCache(){}

        public static final NullCache NULL = new NullCache();
    }

}
