package cn.com.duiba.wolf.perf.timeprofile;

import cn.com.duiba.wolf.dubbo.InitOperation;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.servlet.http.HttpServletRequest;
import java.util.*;

/**
 * 用于记录代码耗时，当代码耗时超过指定阈值时打印告警日志
 */
public class DBTimeProfile {

    private static final Logger                               log       = LoggerFactory.getLogger(DBTimeProfile.class);
    private static final ThreadLocal<LinkedList<TimeProfileElement>> stack     = new ThreadLocal<>();
    private static final ThreadLocal<LinkedList<TimeProfileElement>>  logs      = new ThreadLocal<>();
    private static final ThreadLocal<HttpServletRequest>              request   = new ThreadLocal<>();
    private static final ThreadLocal<Integer>              currentThreshold   = new ThreadLocal<>();
    private static volatile int                                  threshold = 100;// 超时时间，单位毫秒
    private static final String timeout = "DBTimeProfile timeout ";

    static{
        InitOperation.init();
    }

    private DBTimeProfile(){}

    private static void enter() {
        enter("");
    }

    /**
     * 在需要监控耗时的方法体前调用
     * @param tag 方法体的名字
     */
    public static void enter(String tag) {
        if (stack.get() == null) {
            return;
        }
        TimeProfileElement t = new TimeProfileElement();
        t.logEnterTime();
        t.setTag(tag);
        t.setDeep(stack.get().size());

        stack.get().push(t);
    }

    /**
     * 在需要监控耗时的方法体后调用
     */
    public static void release() {
        LinkedList<TimeProfileElement> s1 = stack.get();
        if (s1 == null) {
            return;
        }
        if(s1.size() <= 1){
            //throw new IllegalStateException("release failed,will stop this timeProfile, enter/release 必须成对出现");
            log.warn("release failed,will stop this timeProfile, enter/release 必须成对出现");
            clear();
            return;
        }
        TimeProfileElement t = stack.get().pop();
        t.logReleaseTime();
        logs.get().add(t);
    }

    /**
     * 在Filter等入口处调用,开始计时，当接入了spring-boot-ext项目时，你不需要调用此方法，框架会调用。
     */
    public static void start() {
        stack.set(new LinkedList<>());
        logs.set(new LinkedList<>());
        enter();
    }

    /**
     * 在Filter等入口处调用,结束当前计时，如果超过日志，会打印告警日志，当接入了spring-boot-ext项目时，你不需要调用此方法，框架会调用。
     */
    public static void end() {
        end("undefined");
    }

    /**
     * 在Filter等入口处调用,结束当前计时，如果超过日志，会打印告警日志，当接入了spring-boot-ext项目时，你不需要调用此方法，框架会调用。
     */
    public static void end(String methodName) {
        LinkedList<TimeProfileElement> s1 = stack.get();
        if(s1 == null){
            return;
        }
        if(s1.size() != 1){
            log.warn("timeProfile methodName:{} failed, enter/release 必须成对出现, stackSize:{}", methodName, s1.size());
            clear();
            return;
        }
        TimeProfileElement t = s1.pop();
        t.logReleaseTime();
        logs.get().add(t);

        long timeConsume = t.getCostTimeMillis();
        Integer threshold = currentThreshold.get();
        if(threshold == null){
            threshold = DBTimeProfile.threshold;
        }
        if (t.getCostTimeMillis() > threshold) {
            // 输出日志
            StringBuilder sb = new StringBuilder();
            if (request.get() != null) {
                String url = RequestTool.getRequestUrl(request.get());
                sb.append(timeout).append(timeConsume).append("ms >").append(threshold).append("ms, url=").append(url);
            } else {
                sb.append(timeout).append(timeConsume).append("ms > methodName:").append(methodName);
            }
            List<TimeProfileElement> list = new ArrayList<>(logs.get());


            Collections.reverse(list);
            //listSort(list);
            for (TimeProfileElement s : list) {
                sb.append("\r\n\t");
                for (int i = 0; i < s.getDeep(); i++) {
                    sb.append("-");
                }
                Long consume = s.getCostTimeMillis();
                sb.append(consume * 100 / timeConsume).append("%");
                sb.append("  ").append(consume).append("ms");
                if (s.getTag() != null) {
                    sb.append("  ").append(s.getTag());
                }
            }
            log.warn("{}",sb);
        }
        clear();
    }

    private static void clear(){
        request.set(null);
        stack.set(null);
        logs.set(null);
        currentThreshold.set(null);
    }

    /**
     * 设置报警阈值，这是全局配置，只要在应用启动时设置一次即可,单位：ms
     * <br/>
     * 当接入spring-boot-ext框架时，请使用配置:duiba.profile.threshold 来设置
     * @param threshold
     */
    public static void setThreshold(int threshold) {
        DBTimeProfile.threshold = threshold;
    }

    /**
     * 设置当前线程调用栈的报警阈值，这个配置只会在当前线程、本次监控生效。
     * @param threshold
     */
    public static void setCurrentThreshold(int threshold) {
        currentThreshold.set(threshold);
    }

    /**
     * dump堆栈信息
     *
     */
    private static String dump() {
        if (logs.get() == null) {
            return null;
        }
        TimeProfileElement t = stack.get().pop();
        t.logReleaseTime();
        logs.get().add(t);

        Long timeconsume = 0L;
        for (TimeProfileElement s : logs.get()) {
            if (s.deep == 0) {
                timeconsume += s.getCostTimeMillis();
            }
        }
        if (timeconsume < (long) 0) {
            return null;
        }
        // 输出日志
        StringBuilder sb = new StringBuilder();
        sb.append(timeout).append(timeconsume).append("ms ");
        List<TimeProfileElement> list = new ArrayList<>(logs.get());
        Collections.reverse(list);
        for (TimeProfileElement s : list) {
            Long consume = s.getCostTimeMillis();
            if (consume == 0) {
                continue;
            }
            sb.append("\r\n\t");
            for (int i = 0; i < s.getDeep(); i++) {
                sb.append("  ");
            }
            sb.append("-");
            sb.append(consume * 100 / timeconsume).append("%");
            sb.append("  ").append(consume).append("ms");
            if (s.getTag() != null) {
                sb.append("  ").append(s.getTag());
            }
        }
        clear();

        return sb.toString();
    }

    public static void main(String[] args) throws Exception {
        DBTimeProfile.start();
        try {
            try {
                DBTimeProfile.enter("in");
                Thread.sleep(1000);
                throw new NullPointerException("");
            } finally {
                DBTimeProfile.release();
            }
        }finally {
            DBTimeProfile.end();
            //System.out.println(DBTimeProfile.dump());
        }
    }

    private static class TimeProfileElement {

        //基线时间，为了减小timeMillis的占用空间
        private static final long BASE_TIME = System.currentTimeMillis();

        private int timeMillis;
        private int               deep;
        private String            tag;

        void logEnterTime(){
            timeMillis = Long.valueOf(System.currentTimeMillis() - BASE_TIME).intValue();
        }

        void logReleaseTime(){
            timeMillis = Long.valueOf(System.currentTimeMillis() - BASE_TIME - timeMillis).intValue();
        }

        long getCostTimeMillis(){
            return timeMillis;
        }

        int getDeep() {
            return deep;
        }

        void setDeep(int deep) {
            this.deep = deep;
        }

        String getTag() {
            return tag;
        }

        void setTag(String tag) {
            this.tag = tag;
        }

    }
}
