/**
 * @author huangwq
 */
package cn.com.duiba.wolf.spring.datasource;

import org.apache.commons.lang.StringUtils;
import org.apache.commons.lang.math.RandomUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.jdbc.datasource.lookup.AbstractRoutingDataSource;

import javax.sql.DataSource;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;

/**
 * 读写分离 自动路由 的数据源
 * @author huangwq
 *
 */
public class AutoRoutingDataSource extends AbstractRoutingDataSource {

    private static final Logger log = LoggerFactory.getLogger(AutoRoutingDataSource.class);

    private static final ThreadLocal<Boolean> contextHolder = new ThreadLocal<>();

    private Map<Object, Object>               targetDataSources;

    /**
     * masterDataSource,读写主库
     */
    private DataSource masterDataSource;

    /**
     * master库的key
     */
    private String                            masterKey;

    private String                            weights;
    /**
     * 存储格式如master,master,slave1,slave1,slave1,slave2,slave2
     */
    private List<String>                      keyWeights;
    /**
     * weights的list大小，由于经常调用而且不会改变，设立这样的变量来加快速度,以空间换时间。
     */
    private int                               keyWeightSize;

    @Override
    protected Object determineCurrentLookupKey() {
        Boolean isUseMasterDataSource = contextHolder.get();
        contextHolder.set(null);
        String key = "";
        if (isUseMasterDataSource == null) {
            log.error("warn: isUseMasterDataSource is null,maybe you use sqlMapClient in direct way, or you are not using AutoRoutingDataSourceTransactionManager,it's not suggested. here will use masterDatasource");
            key = masterKey;
        } else if (isUseMasterDataSource) {
            key = masterKey;
        } else {
            // 如果不需要路由到master库，表示是查询操作，根据权重配置负载均衡到各个数据库上
            int randomNum = RandomUtils.nextInt(keyWeightSize);
            key = keyWeights.get(randomNum);
        }
        if (log.isDebugEnabled()) {
            log.debug("use dataSource:{}, threadId:{}", key, Thread.currentThread().getId());
        }
        return key;
    }

    /**
     * 设置是否使用主库（读写库），如果设置false，则使用只读库。
     * @param isUseMasterDataSource
     */
    public void setUseMasterDataSource(boolean isUseMasterDataSource) {
        contextHolder.set(isUseMasterDataSource);
    }

    /**
     * 生成权重字符串，会自动把形如master:2,slave1:3,slave2:2的字符串转换成[master,master,slave1,slave1,slave1,slave2,slave2]数组,后续根据随机算法取得key
     * 
     */
    private void generateKeyWeights() {
        weights = StringUtils.defaultString(weights);

        List<String> weightList = new ArrayList<>();
        List<String> datasourceIds = new ArrayList<>();
        String[] ws = weights.split(",");
        for (String w : ws) {
            w = StringUtils.trim(w);
            if (StringUtils.isEmpty(w)) {
                continue;
            }
            String[] temp = w.split(":");
            int num = Integer.parseInt(temp[1]);
            datasourceIds.add(temp[0]);
            for (int i = 0; i < num; i++) {
                weightList.add(temp[0]);
            }
        }

        for (String key : datasourceIds) {
            if (!targetDataSources.containsKey(key)) {
                throw new IllegalArgumentException("dataSourceKey:" + key + " in weights is not exist");
            }
        }
        for (Map.Entry<Object, Object> entry : this.targetDataSources.entrySet()) {
            String key = (String) entry.getKey();
            if (!datasourceIds.contains(key)) {
                throw new IllegalArgumentException("dataSourceKey:" + key + " is not exist in weights");
            }
        }

        this.keyWeights = weightList;
        keyWeightSize = this.keyWeights.size();
    }

    @Override
    public void setTargetDataSources(Map<Object, Object> targetDataSources) {
        this.targetDataSources = targetDataSources;
        super.setTargetDataSources(targetDataSources);
    }

    @Override
    public void afterPropertiesSet() {
        super.afterPropertiesSet();

        generateKeyWeights();

    }

    public void setMasterKey(String masterKey) {
        this.masterKey = masterKey;
    }

    public void setWeights(String weights) {
        this.weights = weights;
    }

    public void setDefaultTargetDataSource(DataSource defaultTargetDataSource) {
        this.masterDataSource = defaultTargetDataSource;
        super.setDefaultTargetDataSource(defaultTargetDataSource);
    }

    public DataSource getMasterDataSource(){
        return masterDataSource;
    }

    /**
     * 由此类自行决定是否使用主库
     * @param isForceUseMasterDataSource 是否强制使用master库, 当为true时会强制使用master库,当为false时会判断当前是否正在事务中,如果是的话底层仍然会使用master库,如果不是的话会根据配置的规则按权重走主库或slave只读库.建议只在执行增删改sql时传入此参数为true.类似select * for update的语句实际上只在事务中有用,故如果不走事务,这样的语句有可能会被分配到只读库.所以写代码时需要注意for update语句一定要放在事务中.
     */
    public void determineCurrentLookupKeyByItSelf(boolean isForceUseMasterDataSource){
        boolean isCurrentThreadInTransaction = TransactionUtils.isCurrentThreadInTransaction();
        boolean isCurrentDataSourceInTransaction = TransactionUtils.isCurrentDataSourceInTransaction(this);
        if(isCurrentThreadInTransaction){
            //在事务中的时候,在事务开始的时候已经获取了Connection,并且在同一个事务中会重复使用同一个connection(这个connection会和dataSource绑定),所以这里不需要设置标记.
            if(!isCurrentDataSourceInTransaction){//但是如果当前在事务代码块内,但当前数据源操作不属于那个事务,则标记使用主库.
                this.setUseMasterDataSource(true);
            }
        }
        else {
            if(isForceUseMasterDataSource){
                this.setUseMasterDataSource(true);
            }else {
                //如果不在事务中,设置需要随机选择master或slave库的标记.
                this.setUseMasterDataSource(isCurrentThreadInTransaction);
            }
        }
    }
}
