package cn.com.duiba.wolf.utils;

import java.util.Collections;
import java.util.HashMap;
import java.util.Map;

import com.google.common.base.Objects;
import org.apache.commons.lang.StringUtils;

/**
 * Url工具类
 */
public class UrlUtils2 {

    /**
     * 解析出url参数中的键值对,
     * 如 "http://www.baidu.com/index.jsp?Action=del&id=123"，解析出Action:del,id:123存入map中
     *
     * @param url url地址
     * @return url请求参数部分
     */
    public static Map<String, String> extractUrlParamsFromUrl(String url) {
        String strUrlParam = truncateUrlPage(url);
        if (strUrlParam == null) {
            return Collections.emptyMap();
        }
        return extractUrlParams(strUrlParam);
    }

    /**
     * 解析出url参数中的键值对 如 "action=del&id=123"，解析出action:del,id:123存入map中
     * (注意：此方法对类似a=1&a=2的字符串支持不友好)
     */
    public static Map<String, String> extractUrlParams(String paramsStr) {
        if (StringUtils.isBlank(paramsStr)) {
            return Collections.EMPTY_MAP;
        }
        Map<String, String> mapRequest = new HashMap<>();
        String[] arrSplit;
        // 每个键值为一组
        arrSplit = StringUtils.split(paramsStr,'&');
        for (String strSplit : arrSplit) {
            String[] arrSplitEqual = StringUtils.split(strSplit, '=');
            // 解析出键值
            if (arrSplitEqual.length > 1) {
                // 正确解析
                mapRequest.put(arrSplitEqual[0], arrSplitEqual[1]);
            } else {
                if (!arrSplitEqual[0].equals("")) {
                    // 只有参数没有值，加入空字符
                    mapRequest.put(arrSplitEqual[0], "");
                }
            }
        }
        return mapRequest;
    }

    /**
     * 把map组装成类似action=del&id=123的字符串。
     * @param params
     * @return
     */
    public static String buildUrlParams(Map<String, String> params) {
        if (Objects.equal(null, params) || params.isEmpty()) {
            return "";
        }
        StringBuilder sb = new StringBuilder();
        for (Map.Entry<String, String> entry : params.entrySet()) {
            sb.append(entry.getKey()).append("=").append(entry.getValue()).append("&");
        }
        sb.deleteCharAt(sb.length()-1);
        return sb.toString();
    }

    /**
     * 去掉url中的路径，留下请求参数部分
     *
     * @param strURL url地址
     * @return url请求参数部分
     */
    private static String truncateUrlPage(String strURL) {
        String url = StringUtils.trimToEmpty(strURL);

        int idx = url.indexOf('?');
        if(idx == -1) {
            return null;
        }
        return url.substring(idx + 1);
    }

    /**
     * 解析出url请求排除掉GET参数的部分, 比如入参http://www.baidu.com/index?a=1返回http://www.baidu.com/index
     *
     * @param strUrl url地址
     * @return url路径
     */
    public static String extractUrl(String strUrl) {
        if(StringUtils.isBlank(strUrl)){
            return null;
        }
        String url = StringUtils.trimToEmpty(strUrl);

        int idx = url.indexOf('?');
        if(idx == -1) {
            return null;
        }
        return url.substring(0, idx);
    }
    
    /**
     * 往url后面附加GET参数:(url拼参数). <br/>
     * 注意，不会对参数进行urlencode，你需要自行encode
     * @param url
     * @param params
     * @return url
     */
    public static String appendParams(String url, Map<String, String> params){
        StringBuilder sb = new StringBuilder(url);
        if(sb.indexOf("?") != -1){
            sb.append("&");
            sb.append(buildUrlParams(params));
        }else{
            sb.append("?");
            sb.append(buildUrlParams(params));
        }
        return sb.toString();
    }
}
