package cn.com.duiba.youqian.center.api.remoteservice;

import cn.com.duiba.boot.exception.BizException;
import cn.com.duiba.boot.netflix.feign.AdvancedFeignClient;
import cn.com.duiba.youqian.center.api.entity.Employee;
import cn.com.duiba.youqian.center.api.entity.RoleMenu;
import cn.com.duiba.youqian.center.api.enums.EnumPermission;
import cn.com.duiba.youqian.center.api.request.merchant.AddEmployeeRequest;
import cn.com.duiba.youqian.center.api.request.merchant.SearchEmpRequest;
import cn.com.duiba.youqian.center.api.request.merchant.SearchEmployeeRequest;
import cn.com.duiba.youqian.center.api.request.merchant.UpdateEmpRequest;
import cn.com.duiba.youqian.center.api.result.PageResult;
import cn.com.duiba.youqian.center.api.result.merchant.EmployeeVO;
import cn.com.duiba.youqian.center.api.result.merchant.RoleMenuVO;

import java.util.List;

/**
 * @author tengtuox
 * @date 2020/1/6
 * @description 员工/卖家
 */
@AdvancedFeignClient
public interface RemoteEmployeeService {

    /**
     * 获取员工信息
     *
     * @param employeeId employeeId
     * @return Employee
     * @throws BizException 员工不存在
     */
    Employee getEmployee(Long employeeId) throws BizException;

    /**
     * 员工详情
     *
     * @param employeeId employeeId
     * @return EmployeeVO
     * @throws BizException 员工不存在
     */
    EmployeeVO getEmployeeVO(Long employeeId) throws BizException;

    /**
     * 根据account查找
     *
     * @param account account
     * @return Employee
     */
    Employee getByAccount(String account) throws BizException;

    /**
     * 根据UserId查找
     *
     * @param userId userId
     * @return Employee
     */
    Employee getByUserId(Long userId) throws BizException;

    /**
     * 根据商户 + 用户查找员工
     *
     * @param merchantId merchantId
     * @param userId userId
     * @return Employee
     */
    Employee getByMerchantAndUser(Long merchantId, Long userId) throws BizException;

    /**
     * 添加员工
     *
     * @param employee employee
     * @return employeeId
     * @throws BizException 一个账号只能加入一个商户
     */
    Long add(AddEmployeeRequest employee) throws BizException;

    /**
     * 更新员工
     *
     * @param employee employee（简单的更新非空部分）
     * @throws BizException 员工不存在
     */
    void update(Employee employee) throws BizException;

    /**
     * 更新员工
     *
     * @param request request （包含了权限的变更业务逻辑）
     * @throws BizException 员工不存在
     */
    void updateEmp(UpdateEmpRequest request) throws BizException;

    /**
     * 删除员工 -> 退出商户
     *
     * @param employeeId employeeId
     * @throws BizException 员工不存在
     */
    void del(Long employeeId) throws BizException;

    /**
     * 搜索员工
     *
     * @param request request
     * @return EmployeeVO
     * @throws BizException 员工数据异常
     */
    PageResult<EmployeeVO> pageList(SearchEmployeeRequest request) throws BizException;

    /**
     * 搜索Employee
     *
     * @param request request
     * @return List<Employee>
     */
    List<Employee> search(SearchEmpRequest request) throws BizException;

    /**
     * 获取用户菜单
     *
     * @param employeeId employeeId
     * @return List<RoleMenuVO>
     * @throws BizException 用户不存在
     */
    List<RoleMenuVO> getMenuList(Long employeeId) throws BizException;

    /**
     * 获取员工权限
     *
     * @param employeeId employeeId
     * @return List<EnumPermission> 员工权限 = 员工权限 + 员工角色权限
     */
    List<EnumPermission> getEmpPermissionList(Long employeeId) throws BizException;

    /**
     * 判断员工是否拥有权限
     *
     * @param employeeId employeeId
     * @param enumPermission 权限点
     * @return true/false
     */
    Boolean hasEmpPermission(Long employeeId, EnumPermission enumPermission) throws BizException;
}
