package cn.com.duiba.zhongyan.activity.service.api.utils.question;

import cn.com.duiba.boot.exception.BizException;
import cn.com.duiba.cloud.biz.tool.utils.Conditions;
import cn.com.duiba.zhongyan.activity.service.api.enums.CommonStateEnum;
import cn.com.duiba.zhongyan.activity.service.api.enums.JumpTypeEnum;
import cn.com.duiba.zhongyan.activity.service.api.enums.QuestionTypeEnum;
import cn.com.duiba.zhongyan.activity.service.api.param.OptionParam;
import cn.com.duiba.zhongyan.activity.service.api.param.QuestionAnswerParam;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.StringUtils;

import java.util.List;
import java.util.Objects;
import java.util.stream.Collectors;

/**
 * 问题答案校验
 *
 * @author jiangyesheng
 * @version 1.0
 * @date 2022/5/9
 */
public abstract class AbstractQuestionAnswerCheckHandler {

    /**
     * 题目类型
     * @return
     */
    abstract QuestionTypeEnum getType();

    /**
     * 校验问题
     * @param param
     * @throws BizException
     */
    abstract void specificQuestionCheck(QuestionAnswerParam param) throws BizException;

    /**
     * 校验答案
     * @param param
     * @throws BizException
     */
    abstract void specificAnswerCheck(QuestionAnswerParam param) throws BizException;

    /**
     * 校验问题
     * @param param 问题
     * @param allQuestionIds 所有的问题id
     * @param beforeQuestionIds 当前问题之前的问题id
     * @throws BizException
     */
    public void checkQuestion(QuestionAnswerParam param, List<Integer> allQuestionIds, List<Integer> beforeQuestionIds) throws BizException {
        commonCheck(param);
        Integer questionId = param.getQuestionId();
        Conditions.expectTrue(StringUtils.isNotBlank(param.getTitle()) && param.getTitle().length() <= 50, "题目["+questionId+"]:标题错误-不为空且不超过50个字符");
        if (Objects.nonNull(param.getJumpType())) {
            if (param.getJumpType().equals(JumpTypeEnum.OPTION_RULE.getType())) {
                // 多选题、填空题、月份选择题，只有无条件跳转
                Conditions.expectTrue(QuestionTypeEnum.SINGLE_CHOICE.getQuestionType().equals(param.getType())
                        || QuestionTypeEnum.DROP_DOWN_CHOICE.getQuestionType().equals(param.getType()), "题目["+questionId+"]:题目类型错误-题目不支持按选项跳转");
                Conditions.expectTrue(CollectionUtils.isNotEmpty(param.getOption()), "题目["+questionId+"]:选项错误-不能为空");
                // 获取跳转的题目
                List<Integer> jumpQuestionIds = param.getOption().stream().map(OptionParam::getJumpQuestionId).distinct().collect(Collectors.toList());
                Conditions.expectTrue(CollectionUtils.isNotEmpty(jumpQuestionIds), "题目["+questionId+"]:跳转规则错误-跳转题目不能为空");
                // 必须顺序跳
                Conditions.expectTrue(allQuestionIds.containsAll(jumpQuestionIds)
                        && beforeQuestionIds.stream().noneMatch(beforeQuestionIds::contains), "题目["+questionId+"]:跳转规则错误-跳转题目id不正确");
            } else if (param.getJumpType().equals(JumpTypeEnum.QUESTION_RULE.getType())) {
                Conditions.expectNotNull(param.getNoJumpRuleQuestionId(), "题目["+questionId+"]:跳转规则错误-跳转题目不能为空");
                Conditions.expectTrue(allQuestionIds.contains(param.getNoJumpRuleQuestionId())
                        && !beforeQuestionIds.contains(param.getNoJumpRuleQuestionId()), "题目["+questionId+"]:跳转规则错误-跳转题目id不正确");
            }
        }
        specificQuestionCheck(param);
    }

    /**
     * 校验答案 - 不校验跳转规则
     * @param param 答案
     * @throws BizException
     */
    public void checkAnswer(QuestionAnswerParam param) throws BizException {
        commonCheck(param);
        if (Objects.nonNull(param.getQuestionTipStatus())
                && param.getQuestionTipStatus().equals(CommonStateEnum.STATE_ZERO.getState())) {
            // 对答案进行校验
            Conditions.expectTrue(CollectionUtils.isNotEmpty(param.getAnswerList()), "题目["+param.getQuestionId()+"]:答案错误-不能为空");
            specificAnswerCheck(param);
        }
    }

    /**
     * 通用校验
     */
    private void commonCheck(QuestionAnswerParam param) throws BizException {
        if (CommonStateEnum.STATE_ONE.getState().equals(param.getSource())) {
            Conditions.expectNotNull(param.getInfoId(), "题目["+param.getQuestionId()+"]:题目错误-个人题目信息id为空");
        }
    }

    /**
     * 通用的选择题校验
     * @param param
     * @throws BizException
     */
    protected void commonChoiceQuestionCheck(QuestionAnswerParam param, Integer maxOptionSize) throws BizException {
        Integer questionId = param.getQuestionId();
        checkOptionParam(param.getOption(), questionId, maxOptionSize);
        for (OptionParam option : param.getOption()) {
            // 1. 选项内容限制12个字符
            Conditions.expectTrue(StringUtils.isNotBlank(option.getOptionContent()) && option.getOptionContent().length() <= 12, "题目["+questionId+"]:选项内容错误-不能为空且不超过12个字符");
            // 2. 在每个选项后可以选择【在选项后增加填空框】，选择后，需要配置提示语，限制10个字符
            if (CommonStateEnum.STATE_ZERO.getState().equals(option.getAllowWrite())) {
                Conditions.expectTrue(StringUtils.isBlank(option.getOptionTip()) || option.getOptionTip().length() <= 10, "题目["+questionId+"]:选项提示语错误-不能超过10个字符");
            }
            Conditions.expectNotNull(option.getOptionId(), "题目["+questionId+"]:选项错误-选项id不能为空");
        }
    }

    /**
     * 校验选项参数
     */
    protected void checkOptionParam(List<OptionParam> optionList, Integer questionId, Integer maxOptionSize) throws BizException {
        Conditions.expectTrue(CollectionUtils.isNotEmpty(optionList) && optionList.size() <= maxOptionSize, "题目["+questionId+"]:选项错误-不能为空且不超过"+maxOptionSize+"个选项");
        List<String> distinctList = optionList.stream().map(OptionParam::getOptionId).filter(StringUtils::isNotBlank).distinct().collect(Collectors.toList());
        Conditions.expectTrue(distinctList.size() == optionList.size(), "题目["+questionId+"]:选项错误-选项id为空或重复");
    }


}
