package cn.com.duiba.zhongyan.activity.service.api.utils.question;

import cn.com.duiba.boot.exception.BizException;
import cn.com.duiba.cloud.biz.tool.utils.Conditions;
import cn.com.duiba.zhongyan.activity.service.api.param.QuestionAnswerParam;
import cn.hutool.core.util.ClassUtil;
import org.apache.commons.collections4.CollectionUtils;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.function.Function;
import java.util.stream.Collectors;

/**
 * 答题工具
 *
 * @author jiangyesheng
 * @version 1.0
 * @date 2022/5/9
 */
public class QuestionUtil {

    private static Map<Integer, AbstractQuestionAnswerCheckHandler> handlerMap;

    static {
        // 扫描处理器实现类
        Class<AbstractQuestionAnswerCheckHandler> fatherClass = AbstractQuestionAnswerCheckHandler.class;
        String pkgName = ClassUtil.getPackage(fatherClass);
        Set<Class<?>> classes = ClassUtil.scanPackage(pkgName);
        if (CollectionUtils.isEmpty(classes)) {
            handlerMap = Collections.emptyMap();
        }
        List<Class<?>> handlerClassList = classes.stream()
                .filter(sonClass -> ClassUtil.isAssignable(fatherClass, sonClass) && fatherClass != sonClass)
                .collect(Collectors.toList());
        if (CollectionUtils.isEmpty(handlerClassList)) {
            handlerMap = Collections.emptyMap();
        }
        handlerMap = handlerClassList.stream().map(clazz -> {
            try {
                return (AbstractQuestionAnswerCheckHandler) clazz.newInstance();
            } catch (Exception e) {
                // 忽略此处报错
            }
            return null;
        }).filter(Objects::nonNull).collect(Collectors.toMap(o -> o.getType().getQuestionType(), Function.identity()));

    }

    /**
     * 获取处理器
     * @param type
     * @return
     * @throws BizException
     */
    public static AbstractQuestionAnswerCheckHandler getHandler(Integer type) throws BizException {
        Conditions.expectNotNull(type, "题目类型错误-不能为空");
        AbstractQuestionAnswerCheckHandler handler = handlerMap.get(type);
        Conditions.expectNotNull(handler, "不支持的题目类型:"+type);
        return handler;
    }

    /**
     * 校验问题是否符合规则
     * @param questionList
     * @throws BizException
     */
    public static void checkQuestion(List<QuestionAnswerParam> questionList) throws BizException {
        List<Integer> allQuestionIds = getAllQuestionIds(questionList);
        List<Integer> beforeQuestionIds = new ArrayList<>();
        for (QuestionAnswerParam param : questionList) {
            getHandler(param.getType()).checkQuestion(param, allQuestionIds, beforeQuestionIds);
            beforeQuestionIds.add(param.getQuestionId());
        }
    }

    /**
     * 校验答案是否符合规则
     * @param questionList
     * @throws BizException
     */
    public static void checkAnswer(List<QuestionAnswerParam> questionList) throws BizException{
        for (QuestionAnswerParam param : questionList) {
            getHandler(param.getType()).checkAnswer(param);
        }
    }

    /**
     * 获取总问题id列表
     */
    private static List<Integer> getAllQuestionIds(List<QuestionAnswerParam> questionList) throws BizException {
        Conditions.expectTrue(CollectionUtils.isNotEmpty(questionList), "问题列表不能为空");
        List<Integer> allQuestionIds = questionList.stream().map(QuestionAnswerParam::getQuestionId).filter(Objects::nonNull).distinct().collect(Collectors.toList());
        Conditions.expectTrue(questionList.size() == allQuestionIds.size(), "题目id错误-为空或存在重复id");
        return allQuestionIds;
    }

}
