package cn.com.duiba.zhongyan.activity.service.api.utils.question;

import cn.com.duiba.boot.exception.BizException;
import cn.com.duiba.cloud.biz.tool.utils.Conditions;
import cn.com.duiba.zhongyan.activity.service.api.enums.CommonStateEnum;
import cn.com.duiba.zhongyan.activity.service.api.enums.JumpTypeEnum;
import cn.com.duiba.zhongyan.activity.service.api.enums.QuestionTypeEnum;
import cn.com.duiba.zhongyan.activity.service.api.param.AnswerParam;
import cn.com.duiba.zhongyan.activity.service.api.param.OptionParam;
import cn.com.duiba.zhongyan.activity.service.api.param.QuestionAnswerParam;
import cn.hutool.extra.emoji.EmojiUtil;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.lang3.StringUtils;

import java.util.List;
import java.util.Objects;
import java.util.stream.Collectors;

/**
 * 问题答案校验
 *
 * @author jiangyesheng
 * @version 1.0
 * @date 2022/5/9
 */
public abstract class AbstractQuestionAnswerCheckHandler {

    protected final String EMPTY_STRING = StringUtils.EMPTY;

    /**
     * 题目类型
     * @return
     */
    abstract QuestionTypeEnum getType();

    /**
     * 校验问题
     * @param param
     * @throws BizException
     */
    abstract void specificQuestionCheck(QuestionAnswerParam param) throws BizException;

    /**
     * 校验答案
     * @param param
     * @throws BizException
     */
    abstract String specificAnswerCheck(QuestionAnswerParam param) throws BizException;

    /**
     * 校验问题
     * @param param 问题
     * @param allQuestionIds 所有的问题id
     * @param beforeQuestionIds 当前问题之前的问题id
     * @throws BizException
     */
    public void checkQuestion(QuestionAnswerParam param, List<Integer> allQuestionIds, List<Integer> beforeQuestionIds) throws BizException {
        commonCheck(param);
        Integer questionId = param.getQuestionId();
        Conditions.expectTrue(StringUtils.isNotBlank(param.getTitle()) && EmojiUtil.removeAllEmojis(param.getTitle()).length() <= 50, "题目["+questionId+"]:标题错误-不为空且不超过50个字符");
        if (Objects.nonNull(param.getJumpType())) {
            if (param.getJumpType().equals(JumpTypeEnum.OPTION_RULE.getType())) {
                // 多选题、填空题、月份选择题，只有无条件跳转
                Conditions.expectTrue(QuestionTypeEnum.SINGLE_CHOICE.getQuestionType().equals(param.getType())
                        || QuestionTypeEnum.DROP_DOWN_CHOICE.getQuestionType().equals(param.getType()), "题目["+questionId+"]:题目类型错误-题目不支持按选项跳转");
                Conditions.expectTrue(CollectionUtils.isNotEmpty(param.getOption()), "题目["+questionId+"]:选项错误-不能为空");
                // 获取跳转的题目
                List<Integer> jumpQuestionIds = param.getOption().stream().map(OptionParam::getJumpQuestionId).filter(Objects::nonNull).distinct().collect(Collectors.toList());
                Conditions.expectTrue(CollectionUtils.isNotEmpty(jumpQuestionIds), "题目["+questionId+"]:跳转规则错误-跳转题目不能为空");
                // 必须顺序跳
                Conditions.expectTrue(allQuestionIds.containsAll(jumpQuestionIds)
                        && jumpQuestionIds.stream().noneMatch(beforeQuestionIds::contains), "题目["+questionId+"]:跳转规则错误-跳转题目id不正确");
            } else if (param.getJumpType().equals(JumpTypeEnum.QUESTION_RULE.getType())) {
                Conditions.expectNotNull(param.getNoJumpRuleQuestionId(), "题目["+questionId+"]:跳转规则错误-跳转题目不能为空");
                Conditions.expectTrue(allQuestionIds.contains(param.getNoJumpRuleQuestionId())
                        && !beforeQuestionIds.contains(param.getNoJumpRuleQuestionId()), "题目["+questionId+"]:跳转规则错误-跳转题目id不正确");
            }
        }
        specificQuestionCheck(param);
    }

    /**
     * 校验答案
     * @param param 参数
     * @param context 上下文
     * @return 当前用户输入字段 - 用于web层敏感词风控
     * @throws BizException
     */
    public String checkAnswer(QuestionAnswerParam param, AnswerCheckContext context) throws BizException {
        commonCheck(param);
        if (CommonStateEnum.STATE_ONE.getState().equals(param.getSource())) {
            Conditions.expectNotNull(param.getInfoId(), "题目["+param.getQuestionId()+"]:题目错误-个人题目信息id为空");
        }
        Integer jumpId = context.getJumpId();
        if (hasCheckAnswer(param, jumpId)) {
            // 对答案进行校验
            Conditions.expectTrue(CollectionUtils.isNotEmpty(param.getAnswerList()), "题目["+param.getQuestionId()+"]:答案错误-不能为空");
            String input = specificAnswerCheck(param);
            // 当前题目是跳转后的题目，校验完毕后置空
            if (Objects.nonNull(jumpId) && jumpId.equals(param.getQuestionId())) {
                context.setJumpId(null);
            }
            // 获取当前问题的跳转题目
            if (Objects.nonNull(param.getJumpType())) {
                if (JumpTypeEnum.QUESTION_RULE.getType().equals(param.getJumpType())) {
                    context.setJumpId(param.getNoJumpRuleQuestionId());
                } else if (JumpTypeEnum.OPTION_RULE.getType().equals(param.getJumpType())) {
                    // 只有单选题和下拉题目有按照选项跳转
                    context.setJumpId(getOptionJumpQuestionId(param));
                }
            }
            return input;
        }
        return EMPTY_STRING;
    }

    /**
     * 获取按照选项跳转的问题id
     * @param param
     * @return
     */
    private Integer getOptionJumpQuestionId(QuestionAnswerParam param) {
        if (QuestionTypeEnum.SINGLE_CHOICE.getQuestionType().equals(param.getType())
                || QuestionTypeEnum.DROP_DOWN_CHOICE.getQuestionType().equals(param.getType())) {
            AnswerParam answer = param.getAnswerList().get(0);
            // 上面已经校验过了，不会为空的
            OptionParam option = param.getOption().stream().filter(opt -> answer.getAnswer().equals(opt.getOptionId())).findFirst().orElse(null);
            return option.getJumpQuestionId();
        }
        return null;
    }

    /**
     * 是否需要校验答案
     * @param param
     * @return
     */
    private boolean hasCheckAnswer(QuestionAnswerParam param, Integer jumpId) {
        // 如果传入答案，就一定要校验
        if (CollectionUtils.isNotEmpty(param.getAnswerList()) && StringUtils.isNotBlank(param.getAnswerList().get(0).getAnswer())) {
            return true;
        }
        // 如果跳转id不为空，并且当前id不等于跳转id时，不需要校验
        if (Objects.nonNull(jumpId) && !jumpId.equals(param.getQuestionId())) {
            return false;
        }
        // 如果是必填的题目
        return Objects.nonNull(param.getQuestionTipStatus())
                && param.getQuestionTipStatus().equals(CommonStateEnum.STATE_ZERO.getState());
    }

    /**
     * 通用校验
     */
    private void commonCheck(QuestionAnswerParam param) throws BizException {
        // 暂时没得
    }

    /**
     * 通用的选择题校验
     * @param param
     * @throws BizException
     */
    protected void commonChoiceQuestionCheck(QuestionAnswerParam param, Integer maxOptionSize) throws BizException {
        Integer questionId = param.getQuestionId();
        checkOptionParam(param.getOption(), questionId, maxOptionSize);
        for (OptionParam option : param.getOption()) {
            // 1. 选项内容限制12个字符
            Conditions.expectTrue(StringUtils.isNotBlank(option.getOptionContent()) && EmojiUtil.removeAllEmojis(option.getOptionContent()).length() <= 12, "题目["+questionId+"]:选项内容错误-不能为空且不超过12个字符");
            // 2. 在每个选项后可以选择【在选项后增加填空框】，选择后，需要配置提示语，限制10个字符
            if (CommonStateEnum.STATE_ZERO.getState().equals(option.getAllowWrite())) {
                Conditions.expectTrue(StringUtils.isBlank(option.getOptionTip()) || EmojiUtil.removeAllEmojis(option.getOptionTip()).length() <= 10, "题目["+questionId+"]:选项提示语错误-不能超过10个字符");
            }
        }
    }

    /**
     * 校验选项参数
     */
    protected void checkOptionParam(List<OptionParam> optionList, Integer questionId, Integer maxOptionSize) throws BizException {
        Conditions.expectTrue(CollectionUtils.isNotEmpty(optionList) && optionList.size() <= maxOptionSize, "题目["+questionId+"]:选项错误-不能为空且不超过"+maxOptionSize+"个选项");
        List<String> distinctList = optionList.stream().map(OptionParam::getOptionId).filter(StringUtils::isNotBlank).distinct().collect(Collectors.toList());
        Conditions.expectTrue(distinctList.size() == optionList.size(), "题目["+questionId+"]:选项错误-选项id为空或重复");
    }


}
