package cn.com.duiba.user.server.api.request;


import cn.com.duiba.user.server.api.dto.consumer.TimeBasedKeyDTO;
import cn.com.duiba.user.server.api.dto.consumer.TimeBasedRollingKeyDTO;
import cn.com.duiba.user.server.api.param.consumer.RemoteTimeBaseParam;
import cn.com.duiba.user.server.api.remoteservice.RemoteTimeBasedKeyService;
import cn.com.duiba.wolf.utils.DateUtils;
import org.apache.commons.codec.binary.Hex;
import org.joda.time.DateTime;
import org.springframework.stereotype.Service;

import javax.annotation.Resource;
import java.security.SecureRandom;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.Collections;
import java.util.Date;
import java.util.List;

/**
 * @author zhengjianhao
 * @date 2021/9/7
 * @description
 */
@Service
public class KmsClient {

    @Resource
    private RemoteTimeBasedKeyService remoteTimeBasedKeyService;

    private SecureRandom ranGen = new SecureRandom();

    private volatile List<TimeBasedRollingKeyDTO> cachedKeys;
    //当前时间段的密钥
    private volatile TimeBasedRollingKeyDTO currentCachedKey;

    /**
     * 获取按时间滚动的密钥。<br/>
     * 适用场景：一些需要加密数据的非持久化场景;<br/>
     * 场景举例：登录后需要在cookie中留下经过加密的登录凭证，加密前的登录凭证形如：{"appId":1,"consumerId":2,time:1241213213213}, 加密方法为取到time所对应的密钥进行加密，需要在cookie中额外放入一个明文字段用于表示time字段（必须和登录凭证中的time完全一致，以防在两个密钥的切换时间点出现问题）；服务端解密前需要根据明文time取到对应的密钥，然后解密登录凭证。
     *
     * @param timestamp 时间戳，接口会返回此时间戳对应的密钥
     */
    public TimeBasedRollingKeyDTO getCachedTimeBasedRollingKey(long timestamp) {
        long now = System.currentTimeMillis();
        long oneDayAgo = now - 86400000;

        //判断时多加5分钟，以防其他调用端机器跟当前机器有差异导致问题（其他调用端生成cookie时间比当前机器新）
        if (timestamp > now + 300000) {
            throw new IllegalArgumentException("timestamp must not after now");
        }
        if (timestamp < oneDayAgo) {
            throw new IllegalArgumentException("timestamp must not before oneDayAgo");
        }

        //保证cachedKeys中至少包含昨天今天明天的数据
        if (currentCachedKey == null || !currentCachedKey.isMatch(now)) {
            synchronized (this) {
                if (currentCachedKey == null || !currentCachedKey.isMatch(now)) {
                    cachedKeys = this.getTimeBasedRollingKey();
                    for (TimeBasedRollingKeyDTO key : cachedKeys) {
                        if (key.isMatch(now)) {
                            currentCachedKey = key;
                            break;
                        }
                    }
                }
            }
        }

        for (TimeBasedRollingKeyDTO key : cachedKeys) {
            if (key.isMatch(timestamp)) {
                return key;
            }
        }

        throw new IllegalStateException("timeBasedKey not found,timestamp is " + timestamp + ",currentCachedKeys:" + cachedKeys);
    }

    private List<TimeBasedRollingKeyDTO> getTimeBasedRollingKey() {
        long now = System.currentTimeMillis();
        //最多只允许获取一天内的数据，考虑到不同服务器的时间差异，多减五分钟
        Date oneDaysAgo = new DateTime(now).minusDays(1).minusMinutes(5).toDate();
        //同样，考虑到考虑到不同服务器的时间差异，当前时间也多加五分钟
        now += 300000;
        String day = DateUtils.getDayStr(oneDaysAgo);
        Date tomorrow = new DateTime(now).plusDays(1).toDate();
        //从数据库拿出来的是按时间倒序排列的，时间越接近现在的命中可能性越高
        List<TimeBasedKeyDTO> keys = remoteTimeBasedKeyService.findRecently(RemoteTimeBaseParam.builder().day(day).build()).getTimeBasedKeyList();

        Date createStartDay = null;
        //数据库里指定时间开始没有密钥，则创建到明天为止的密钥
        if (keys.isEmpty()) {
            createStartDay = oneDaysAgo;
        } else {
            TimeBasedKeyDTO newestKey = keys.get(0);
            String newestDayInDb = newestKey.getValidDay();
            Date newestDateInDb = DateUtils.getDayDate(newestDayInDb);
            //如果没有今天或明天的数据，则创建
            if (!newestDateInDb.after(new Date(now))) {
                createStartDay = DateUtils.daysAddOrSub(newestDateInDb, 1);
            }
        }
        List<TimeBasedKeyDTO> createdKeys = createKeysFromDayUntilTomorrow(createStartDay, tomorrow);
        createdKeys.addAll(keys);

        List<TimeBasedRollingKeyDTO> dtos = new ArrayList<>(createdKeys.size());
        for (TimeBasedKeyDTO e : createdKeys) {
            TimeBasedRollingKeyDTO dto = new TimeBasedRollingKeyDTO();
            dto.setSecretKey(e.getSecretKey());
            dto.setStartTimeMillis(DateUtils.getDayStartTime(e.getValidDay()).getTime());
            dto.setEndTimeMillis(DateUtils.daysAddOrSub(new Date(dto.getStartTimeMillis()), 1).getTime());
            dtos.add(dto);
        }

        return dtos;
    }

    private List<TimeBasedKeyDTO> createKeysFromDayUntilTomorrow(Date createStartDay, Date tomorrow) {
        List<TimeBasedKeyDTO> list = new ArrayList<>(5);
        if (createStartDay == null) {
            return list;
        }
        Date date = createStartDay;
        Date dayAfterTomorrow = org.apache.commons.lang3.time.DateUtils.truncate(DateUtils.daysAddOrSub(tomorrow, 1), Calendar.DATE);
        while (date.before(dayAfterTomorrow)) {
            TimeBasedKeyDTO e = new TimeBasedKeyDTO();
            String day = DateUtils.getDayStr(date);
            e.setValidDay(day);
            e.setSecretKey(createRandomKey());

            TimeBasedKeyDTO byCondition = remoteTimeBasedKeyService.findByCondition(RemoteTimeBaseParam.builder().day(day).build());
            if (byCondition == null || byCondition.getId() == null) {
                Integer successCount = remoteTimeBasedKeyService.addTimeBasedKey(e).getCount();
                //当前线程插入失败了，那么其他线程肯定插入成功了，从数据库提取之
                if (successCount <= 0) {
                    e = remoteTimeBasedKeyService.findByCondition(RemoteTimeBaseParam.builder().day(day).build());
                    if (e == null) {
                        throw new IllegalStateException();
                    }
                }
            } else {
                e = byCondition;
            }
            list.add(e);
            date = DateUtils.daysAddOrSub(date, 1);
        }
        Collections.reverse(list);
        return list;
    }

    private String createRandomKey() {
        int len = 8;
        // 16 bytes = 128 bits
        byte[] bs = new byte[len];
        ranGen.nextBytes(bs);
        return Hex.encodeHexString(bs);
    }
}
