package cn.lili.modules.goods.client;

import cn.lili.modules.constant.ServiceConstant;
import cn.lili.modules.goods.entity.dos.Goods;
import cn.lili.modules.goods.entity.dos.GoodsSku;
import cn.lili.modules.goods.entity.dos.ReturnAddressDTO;
import cn.lili.modules.goods.entity.dos.Wholesale;
import cn.lili.modules.goods.entity.dto.GoodsSearchParams;
import cn.lili.modules.goods.entity.vos.GoodsVO;
import cn.lili.modules.goods.fallback.GoodsFallback;
import cn.lili.modules.store.entity.dos.Store;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import org.springframework.cloud.openfeign.FeignClient;
import org.springframework.web.bind.annotation.*;

import java.util.List;

/**
 * 商品服务client
 *
 * @author Chopper
 * @version v1.0 2021-11-08 09:52
 */
@FeignClient(name = ServiceConstant.GOODS_SERVICE, contextId = "goods", fallback = GoodsFallback.class)
public interface GoodsClient {

    /**
     * 查询商品VO
     *
     * @param goodsId 商品id
     * @return 商品VO
     */
    @GetMapping("/feign/goods/vo/{goodsId}")
    GoodsVO getGoodsVO(@PathVariable String goodsId);

    /**
     * 从缓存中获取商品
     *
     * @param skuId skuid
     * @return 商品详情
     */
    @GetMapping("/feign/goods/sku/{skuId}/cache")
    GoodsSku getGoodsSkuByIdFromCache(@PathVariable String skuId);

    /**
     * 从缓存中获取商品
     *
     * @param ids skuid
     * @return 商品详情
     */
    @PostMapping("/feign/goods/sku/list/cache")
    List<GoodsSku> getGoodsSkuByIdFromCache(@RequestBody List<String> ids);

    /**
     * 从缓存中获取可参与促销商品
     *
     * @param skuId skuid
     * @return 商品详情
     */
    @GetMapping("/feign/goods/sku/promotion/{skuId}/cache")
    GoodsSku getCanPromotionGoodsSkuByIdFromCache(@PathVariable String skuId);

    /**
     * 批量修改商品信息 以及 sku信息
     *
     * @param store 店铺信息
     */
    @PutMapping("/feign/goods/update/store")
    void updateStoreDetail(@RequestBody Store store);

    /**
     * 批量下架所有店铺商品
     *
     * @param id 店铺id
     */
    @PutMapping("/feign/goods/under/store/{id}")
    void underStoreGoods(@PathVariable String id);

    @PutMapping("/feign/goods/del/store/{id}")
    void delStoreGoods(@PathVariable String id);

    /**
     * 条件统计商品
     *
     * @param storeId 店铺ID
     * @return 总数
     */
    @GetMapping("/feign/goods/{storeId}/count")
    Long count(@PathVariable String storeId);

    /**
     * 获取sku库存
     *
     * @param skuId skuId
     * @return 库存
     */
    @GetMapping("/feign/goods/sku/{skuId}/stock")
    Integer getStock(@PathVariable String skuId);

    /**
     * 添加商品评价数量
     *
     * @param commentNum 评价数量
     * @param goodsId    商品ID
     */
    @GetMapping("/feign/goods/comment/{goodsId}/num")
    void addGoodsCommentNum(@RequestParam Integer commentNum, @PathVariable String goodsId);

    @GetMapping("/feign/goods/goods/{goodsId}/getById")
    Goods getById(@PathVariable String goodsId);


    @PostMapping("/feign/goods/queryListByParams")
    List<Goods> queryListByParams(@RequestBody GoodsSearchParams searchParams);


    @GetMapping("/feign/goods/{goodsId}/getSkuIdsByGoodsId")
    List<String> getSkuIdsByGoodsId(@PathVariable String goodsId);

    @PostMapping("/feign/goods/sku/list")
    List<GoodsSku> getGoodsSkuByList(@RequestBody GoodsSearchParams searchParams);

    @PutMapping("/feign/goods/buy/{goodsId}/count")
    void updateGoodsBuyCount(@PathVariable String goodsId, @RequestParam int buyCount);

    @GetMapping("/feign/goods/sku/{goodsId}")
    GoodsSku getGoodsSkuById(@PathVariable String goodsId);

    @PutMapping("/feign/goods/goodsSku/updateGoodsSku")
    void updateGoodsSku(@RequestBody GoodsSku goodsSku);

    @PostMapping("/feign/goods/goods/getGoodsByParams")
    Goods getGoodsByParams(@RequestBody GoodsSearchParams searchParams);

    /**
     * 匹配批发规则
     *
     * @param goodsId 商品规则
     * @param num     数量
     * @return 批发规则
     */
    @PostMapping("/feign/goods/wholesale/{goodsId}/match")
    Wholesale getMatchWholesale(@PathVariable String goodsId, @RequestParam Integer num);

    /**
     * 统计sku总数
     *
     * @param storeId 店铺id
     * @return sku总数
     */
    @GetMapping("/feign/goods/sku/countSkuNum/{storeId}")
    Long countSkuNum(@PathVariable String storeId);


    /**
     * 分页查询商品sku信息
     *
     * @param searchParams 查询参数
     * @return 商品sku信息
     */
    @PostMapping("/feign/goods/sku/getGoodsSkuByPage")
    Page<GoodsSku> getGoodsSkuByPage(@RequestBody GoodsSearchParams searchParams);

    /**
     * 根据分类下架商品信息
     * @param categoryId   分类ID
     * @param marketEnable 上架状态
     * @param underMessage 下架原因
     * @return 是否成功
     */
    @GetMapping("/feign/goods/updateGoodsMarketEnable")
    void updateGoodsMarketEnable(@RequestParam String categoryId, @RequestParam String marketEnable,@RequestParam String underMessage);

    /**
     * 同步商品分类名称
     * @param categoryId  分类ID
     */
    @GetMapping("/feign/goods/categoryGoodsName")
    void categoryGoodsName(@RequestParam String categoryId);

    @GetMapping("/feign/goods/getReturnAddress")
    ReturnAddressDTO getReturnAddress(@RequestParam Long goodsId);
}
