package cn.lili.modules.goods.client;

import cn.lili.cache.CachePrefix;
import cn.lili.modules.constant.ServiceConstant;
import cn.lili.modules.goods.entity.dos.GoodsSku;
import cn.lili.modules.goods.fallback.GoodsSkuFallback;
import org.springframework.cloud.openfeign.FeignClient;
import org.springframework.web.bind.annotation.*;

import java.util.List;

/**
 * @author: ftyy
 * @date: 2022-01-12 10:52
 * @description: goodsSku Client
 */
@FeignClient(
        name = ServiceConstant.GOODS_SERVICE,
        contextId = "goods-sku",
        fallback = GoodsSkuFallback.class)
public interface GoodsSkuClient {

    /**
     * 获取商品SKU库存缓存ID
     *
     * @param id SkuId
     * @return 商品SKU缓存ID
     */
    @GetMapping("/feign/goods/goods-sku/getStockCacheKey")
    static String getStockCacheKey(@RequestParam String id) {
        return CachePrefix.SKU_STOCK.getPrefix() + id;
    }

    /**
     * 从redis缓存中获取商品SKU信息
     *
     * @param id SkuId
     * @return 商品SKU信息
     */
    @GetMapping("/feign/goods/goods-sku/getGoodsSkuByIdFromCache")
    GoodsSku getGoodsSkuByIdFromCache(@RequestParam String id);

    /**
     * 批量从redis中获取商品SKU信息
     *
     * @param ids SkuId集合
     * @return 商品SKU信息集合
     */
    @PostMapping("/feign/goods/goods-sku/getGoodsSkuByIdFromCache")
    List<GoodsSku> getGoodsSkuByIdFromCache(@RequestBody List<String> ids);

    /**
     * 获取商品sku库存
     *
     * @param skuId 商品skuId
     * @return 库存数量
     */
    @GetMapping("/feign/goods/goods-sku/getStock")
    Integer getStock(@RequestParam String skuId);

    /**
     * 修改商品库存字段
     *
     * @param goodsSkus
     */
    @PutMapping("/feign/goods/goods-sku/updateGoodsStuck")
    void updateGoodsStuck(@RequestBody List<GoodsSku> goodsSkus);

    /**
     * 更新SKU评价数量
     *
     * @param skuId SKUId
     */
    @PutMapping("/feign/goods/goods-sku/updateGoodsSkuCommentNum")
    void updateGoodsSkuCommentNum(@RequestParam String skuId);

    @PutMapping("/feign/goods/goods-sku/updateBatchById")
    void updateBatchById(@RequestBody List<GoodsSku> goodsSkus);

    /**
     * 获取商品SKU信息列表
     *
     * @param stockWarning 库存
     * @param storeId      店铺ID
     * @return 商品SKU信息列表
     */
    @GetMapping("/feign/goods/goods-sku/getStockWarningGoods")
    List<GoodsSku> getStockWarningGoods(@RequestParam Integer stockWarning, @RequestParam String storeId);
}
