package cn.lili.modules.order.order.client;

import cn.lili.modules.constant.ServiceConstant;
import cn.lili.modules.order.fallback.OrderFallback;
import cn.lili.modules.order.order.entity.dos.Order;
import cn.lili.modules.order.order.entity.dto.OrderSearchParams;
import cn.lili.modules.order.order.entity.dto.PaymentLogSearchParams;
import cn.lili.modules.order.order.entity.vo.OrderDetailVO;
import cn.lili.modules.order.order.entity.vo.PaymentLog;
import cn.lili.modules.store.entity.dos.Store;
import com.baomidou.mybatisplus.core.conditions.Wrapper;
import com.baomidou.mybatisplus.extension.plugins.pagination.Page;
import org.springframework.cloud.openfeign.FeignClient;
import org.springframework.web.bind.annotation.*;

import java.util.List;

/**
 * 订单服务client
 *
 * @author Chopper
 * @version v1.0 2021-11-08 09:52
 */
@FeignClient(
        name = ServiceConstant.ORDER_SERVICE,
        contextId = "order",
        fallback = OrderFallback.class)
public interface OrderClient {

    /**
     * 订单编号查询
     *
     * @param sn 订单编号
     * @return 订单模型
     */
    @GetMapping("/feign/order/query/{sn}")
    Order getBySn(@PathVariable String sn);

    /**
     * 支付订单
     *
     * @param sn            订单编号
     * @param paymentMethod 支付方式
     * @param receivableNo  流水好吗
     */
    @GetMapping("/feign/order/pay")
    void payOrder(
            @RequestParam("sn") String sn,
            @RequestParam("payment_method") String paymentMethod,
            @RequestParam("receivable_no") String receivableNo,
            @RequestParam("pay_order_no") String payOrderNo);

    /**
     * 系统取消订单
     *
     * @param orderSn 订单编号
     * @param reason  错误原因
     */
    @GetMapping("/feign/order/system/cancel")
    void systemCancel(@RequestParam String orderSn, @RequestParam String reason);

    /**
     * 获取实际支付金额
     *
     * @param orderSn 订单编号
     * @return 支付金额
     */
    @GetMapping("/feign/order/query/{orderSn}/payment-total")
    Double getPaymentTotal(@PathVariable String orderSn);

    /**
     * 获取订单详情
     *
     * @param sn 订单编号
     * @return
     */
    @GetMapping("/feign/order/query/{sn}/detail-vo")
    OrderDetailVO queryDetail(@PathVariable String sn);

    /**
     * 根据交易sn，获取订单列表
     *
     * @param sn
     * @return
     */
    @GetMapping("/feign/order/query/{sn}/by-trade-sn")
    List<Order> getByTradeSn(@PathVariable String sn);

    /**
     * 订单信息
     *
     * @param orderSearchParams 查询参数
     * @return 订单信息
     */
    @PostMapping("/feign/order/query/list")
    List<Order> queryListByParams(@RequestBody OrderSearchParams orderSearchParams);

    /**
     * 根据促销查询订单
     *
     * @param orderPromotionType 订单类型
     * @param payStatus          支付状态
     * @param parentOrderSn      依赖订单编号
     * @param orderSn            订单编号
     * @return 订单信息
     */
    @GetMapping("/feign/order/query/promotions/list")
    List<Order> queryListByPromotion(
            @RequestParam String orderPromotionType,
            @RequestParam String payStatus,
            @RequestParam String parentOrderSn,
            @RequestParam String orderSn);

    /**
     * 根据促销查询订单
     *
     * @param orderPromotionType 订单类型
     * @param payStatus          支付状态
     * @param parentOrderSn      依赖订单编号
     * @param orderSn            订单编号
     * @return 订单信息
     */
    @GetMapping("/feign/order/query/promotions/count")
    long queryCountByPromotion(
            @RequestParam String orderPromotionType,
            @RequestParam String payStatus,
            @RequestParam String parentOrderSn,
            @RequestParam String orderSn);

    /**
     * 检查是否开始虚拟成团
     *
     * @param pintuanId   拼团活动id
     * @param requiredNum 成团人数
     * @param fictitious  是否开启成团
     * @return 是否成功
     */
    @GetMapping("/feign/order/fictitious/pintuan/{pintuanId}")
    boolean checkFictitiousOrder(
            @PathVariable String pintuanId,
            @RequestParam Integer requiredNum,
            @RequestParam Boolean fictitious);

    @PostMapping("/feign/order/fictitious/save")
    void save(@RequestBody Order order);

    @GetMapping("/feign/order/{orderSn}/fictitious/afterOrderConfirm")
    void afterOrderConfirm(@PathVariable String orderSn);

    /**
     * 系统定时完成订单
     *
     * @param orderSn 订单编号
     */
    @GetMapping("/feign/order/{orderSn}/system/complete")
    void systemComplete(@PathVariable String orderSn);

    /**
     * 自动成团订单处理
     *
     * @param pintuanId     拼团活动id
     * @param parentOrderSn 拼团订单sn
     */
    @GetMapping("/feign/order/agglomerate/pintuan/{pintuanId}")
    void agglomeratePintuanOrder(@PathVariable String pintuanId, @RequestParam String parentOrderSn);

    @PutMapping("/feign/order/fictitious/update")
    void update(@RequestBody Order order);

    @PostMapping("/feign/order/fictitious/getOne")
    Order getOne(@RequestBody Order order);

    /**
     * 根据交易SN或者订单列表
     *
     * @param sn 交易SN
     * @return
     */
    @PostMapping("/feign/order/getOrderListByTrade")
    List<Order> getOrderListByTrade(@RequestParam String sn);

    @PostMapping("/feign/order/queryPaymentLogs")
    Page<PaymentLog> queryPaymentLogs(@RequestBody PaymentLogSearchParams paymentLogSearchParams);
    @PostMapping("/feign/order/queryPaymentLogsList")
    List<PaymentLog> queryPaymentLogsList(@RequestBody PaymentLogSearchParams paymentLogSearchParams);
    @PutMapping("/feign/order/queryPaymentLogsList")
    void editPaymentLogsList(@RequestBody List<PaymentLog> paymentLogList);

    /**
     * 修改订单中店铺名称
     * @param store  店铺信息
     */
    @PutMapping("/feign/order/updateStoreName")
    void updateStoreName(@RequestBody Store store);

    @PostMapping("/feign/order/afterSaleCallBack")
    void afterSaleCallBack(@RequestParam Boolean success,@RequestParam String orderSn);

    @PutMapping("/feign/order/updateMchInfo")
    void updateMchInfo(@RequestBody Order order);
}
