package cn.tuia.mango.thirdparty.duiba.api;

import cn.tuia.mango.core.domain.api.Result;
import cn.tuia.mango.core.domain.api.Results;
import cn.tuia.mango.thirdparty.duiba.api.model.DuibaAppConfig;
import cn.tuia.mango.thirdparty.duiba.api.model.UserInfo;
import cn.tuia.mango.thirdparty.duiba.sdk.CreditConsumeParams;
import cn.tuia.mango.thirdparty.duiba.sdk.CreditConsumeResult;
import cn.tuia.mango.thirdparty.duiba.sdk.CreditNotifyParams;
import cn.tuia.mango.thirdparty.duiba.sdk.CreditTool;
import com.alibaba.fastjson.JSON;
import io.swagger.annotations.ApiOperation;
import lombok.extern.slf4j.Slf4j;
import org.springframework.util.StringUtils;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.ResponseBody;

import javax.annotation.Resource;
import javax.servlet.http.HttpServletRequest;
import java.util.HashMap;
import java.util.Map;

import static cn.tuia.mango.thirdparty.duiba.support.DefaultDuibaCoreServiceImpl.AUTO_LOGIN_URL;

/**
 * 兑吧积分Controller
 *
 * <p>可以直接继承此类，然后实现DuibaCreditsOrderService中的方法</p>
 * @author lijicong
 * @since 2020-04-13
 */
@Slf4j
public class DuibaCreditsOrderController {

    private DuibaCreditsOrderService creditsOrderService;

    @Resource
    public void setCreditsOrderService(DuibaCreditsOrderService creditsOrderService) {
        this.creditsOrderService = creditsOrderService;
    }

    @GetMapping(value = "/getAutologinUrl")
    @ApiOperation("获取免密url")
    @ResponseBody
    public Result<String> getAutologinUrl(HttpServletRequest request) {
        DuibaAppConfig appConfig = creditsOrderService.getAppConfig();
        CreditTool tool = new CreditTool(appConfig.getAppKey(), appConfig.getAppSecret());
        UserInfo userInfo = creditsOrderService.getUserInfo(request);
        if (userInfo == null || !StringUtils.hasText(userInfo.getUid())) {
            return Results.success();
        }
        Map<String, String> params = new HashMap<>();
        params.put("uid", userInfo.getUid());
        params.put("credits", userInfo.getCredits().toString());
        if (StringUtils.hasText(userInfo.getRedirect())) {
            params.put("redirect", userInfo.getRedirect());
        }
        if (StringUtils.hasText(userInfo.getDcustom())) {
            params.put("dcustom", userInfo.getDcustom());
        }
        if (StringUtils.hasText(userInfo.getTransfer())) {
            params.put("transfer", userInfo.getTransfer());
        }
        if (StringUtils.hasText(userInfo.getSingKeys())) {
            params.put("signKeys", userInfo.getSingKeys());
        }
        String url = tool.buildUrlWithSign(AUTO_LOGIN_URL, params);
        return Results.success(url);
    }

    @GetMapping(value = "/consume")
    @ApiOperation("扣积分")
    public CreditConsumeResult consume(HttpServletRequest request) {
        DuibaAppConfig appConfig = creditsOrderService.getAppConfig();
        CreditTool tool = new CreditTool(appConfig.getAppKey(), appConfig.getAppSecret());
        CreditConsumeResult result = null;
        try {
            CreditConsumeParams params = tool.parseCreditConsume(request);
            if (log.isDebugEnabled()) {
                log.debug("兑吧-扣积分接口 request={}", JSON.toJSONString(params));
            }
            result = creditsOrderService.consumeCredits(params);
        } catch (Exception e) {
            result = new CreditConsumeResult(false);
            result.setErrorMessage(e.getMessage());
        }
        return result;
        //return result.toString(); // 返回扣积分结果json信息
    }

    @GetMapping(value = "/consumeOrder")
    @ApiOperation("扣积分")
    public String consumeOrder(HttpServletRequest request) {
        return consume(request).toString(); // 返回扣积分结果json信息
    }

    @GetMapping(value = "/notify")
    @ApiOperation("兑换结果通知")
    public String notifyResult(HttpServletRequest request) {
        /*
         * 兑换订单的结果通知请求的解析方法
         * 当兑换订单成功时，兑吧会发送请求通知开发者，兑换订单的结果为成功或者失败，如果为失败，开发者需要将积分返还给用户
         */
        DuibaAppConfig appConfig = creditsOrderService.getAppConfig();
        CreditTool tool = new CreditTool(appConfig.getAppKey(), appConfig.getAppSecret());
        try {
            CreditNotifyParams params = tool.parseCreditNotify(request);//利用tool来解析这个请求
            if (log.isDebugEnabled()) {
                log.debug("兑吧-兑换结果通知接口 request={}", JSON.toJSONString(params));
            }
            String orderNum = params.getOrderNum();
            if (params.isSuccess()) {
                // 兑换成功
                creditsOrderService.confirmConsumeCredits(orderNum);
            } else {
                // 兑换失败，根据orderNum，对用户的金币进行返还，回滚操作
                creditsOrderService.cancleConsumeCredits(orderNum, params.getErrorMessage());
            }
        } catch (Exception e) {
            e.printStackTrace();
            log.warn("兑吧-兑换结果通知接口 异常 e={}", e.getMessage());
        }
        return "ok";
    }


}
