package cn.tuia.tools.deviceid;

import org.apache.commons.lang.StringUtils;
import org.springframework.util.Assert;

/**
 * 【组合模式】
 * 全局deviceId生成器
 * CF文档：http://cf.dui88.com/pages/viewpage.action?pageId=63937519
 * （1）deviceId生成规则：imei==>idfa==oaid==>uuid
 * （2）tcid生成：uuid会根据输入的 deviceType，生成前缀
 * （3）返回合法的deviceId（32位md5），不合法的32位md5会置为null，改成正确的返回
 */
public class DeviceIdUtils {

    /**
     * 根据入参的DeviceDTO对象进行处理,自动MD5、自动校验、自动识别错误MD5
     * DeviceDTO deviceDTO = DeviceDTO.builder().imei("351710058880864").deviceId("11").build();
     * DeviceIdUtilService.checkAndGenerateDeviceId(deviceDTO);
     */
    public static void checkAndGenerateDeviceId(DeviceDTO deviceDTO) {
        DeviceIdUtilService.checkAndGenerateDeviceId(deviceDTO);
    }

    /**
     * 生成带时间错的uuid，tcid也采用当前策略
     * 1.时间戳
     * 2.带标志信息：设备类型
     * IMEI(1, "1010"),
     * IDFA(2, "1020"),
     * OAID(3, "1011"),
     * 合法的deviceId是md5小写32位的
     * DEVICE_ID(4, "1030"),
     * UUID(5, "1040"),
     * UNKNOWN(6, "1050");
     *
     * @return
     */
    public static String getUUIDAndMd5(String desc) {
        return DeviceIdUtilService.getUUIDAndMd5(desc);
    }

    /**
     * 合法性校验
     *
     * @param str
     * @return
     */
    public static boolean checkLegalMd5(String str, DeviceTypeEnum typeEnum) {
        return LegalMd5Factory.checkLegalMd5(str, typeEnum);
    }

    /**
     * 是否小写32位MD5加密字符串
     *
     * @param str
     * @return
     */
    public static boolean isLowerCase32Md5(String str) {
        return LegalMd5Factory.isLowerCase32Md5(str);
    }

    /**
     * 判断前缀
     * @param desc
     * @param deviceUUID
     * @return
     */
    public static Boolean startWithDesc(String desc, String deviceUUID) {
        if (StringUtils.isEmpty(desc) || desc.length() > 4) {
            return false;
        }
        if (StringUtils.isBlank(deviceUUID)) {
            return false;
        }
        return deviceUUID.startsWith(desc);
    }


    /**
     * 测试的脚本------------------------------------------------------------
     * d41d8cd98f00b204e9800998ecf8427e
     *
     * @param args
     */
    public static void main(String[] args) {
        System.out.println("【测试】 imei 合法： 351710058880864");
        DeviceDTO deviceDTO = DeviceDTO.builder().imei("351710058880864").deviceId("11").build();
        DeviceIdUtils.checkAndGenerateDeviceId(deviceDTO);
        printResult(deviceDTO);

        System.out.println("【测试】 idfa 合法: CCD6E1CD-8C4B-40CB-8A62-4BBC7AFE07D6");
        deviceDTO = DeviceDTO.builder().idfa("CCD6E1CD-8C4B-40CB-8A62-4BBC7AFE07D6").deviceId("22").build();
        DeviceIdUtils.checkAndGenerateDeviceId(deviceDTO);
        printResult(deviceDTO);

        System.out.println("【测试】 oaid 合法: 351710058880864");
        deviceDTO = DeviceDTO.builder().oaid("351710058880864").build();
        DeviceIdUtils.checkAndGenerateDeviceId(deviceDTO);
        printResult(deviceDTO);

        System.out.println("【测试】 imei Md5 合法: d41d8cd98f00b204e9800998ecf84271");
        deviceDTO = DeviceDTO.builder().imei_md5("d41d8cd98f00b204e9800998ecf84271").build();
        DeviceIdUtils.checkAndGenerateDeviceId(deviceDTO);
        printResult(deviceDTO);

        System.out.println("【测试】 idfa Md5 合法");
        deviceDTO = DeviceDTO.builder().idfa_md5("d41d8cd98f00b204e9800998ecf84272").build();
        DeviceIdUtils.checkAndGenerateDeviceId(deviceDTO);
        printResult(deviceDTO);

        System.out.println("【测试】 oaid Md5 合法");
        deviceDTO = DeviceDTO.builder().oaid_md5("d41d8cd98f00b204e9800998ecf84273").build();
        DeviceIdUtils.checkAndGenerateDeviceId(deviceDTO);
        printResult(deviceDTO);

        System.out.println("【测试】 deviceId 合法： d41d8cd98f00b204e9800998ecf84274");
        deviceDTO = DeviceDTO.builder().deviceId("d41d8cd98f00b204e9800998ecf84274").build();
        DeviceIdUtils.checkAndGenerateDeviceId(deviceDTO);
        printResult(deviceDTO);

        System.out.println("【测试】 imei 传错了传了MD5： d41d8cd98f00b204e9800998ecf84275");
        deviceDTO = DeviceDTO.builder().imei("d41d8cd98f00b204e9800998ecf84275").build();
        DeviceIdUtils.checkAndGenerateDeviceId(deviceDTO);
        printResult(deviceDTO);

        System.out.println("【测试】 imeiMd5 不合法： cfcd208495d565ef66e7dff9f98764da");
        deviceDTO = DeviceDTO.builder().imei_md5(DeviceMd5BlackEnum.ERROR_1.getDeviceMd5())
                .deviceType(DeviceTypeEnum.IMEI.getCode()).build();
        DeviceIdUtils.checkAndGenerateDeviceId(deviceDTO);
        printResult(deviceDTO);

        System.out.println("【测试】 deviceId 不合法，oaidMd5 合法： d41d8cd98f00b204e9800998ecf84273");
        deviceDTO = DeviceDTO.builder().deviceId(DeviceMd5BlackEnum.ERROR_1.getDeviceMd5())
                .oaid_md5("d41d8cd98f00b204e9800998ecf84273").build();
        DeviceIdUtils.checkAndGenerateDeviceId(deviceDTO);
        printResult(deviceDTO);

        System.out.println("【测试】 deviceId 不合法，imei 传错成md5： d41d8cd98f00b204e9800998ecf84273");
        deviceDTO = DeviceDTO.builder().deviceId(DeviceMd5BlackEnum.ERROR_1.getDeviceMd5())
                .imei("d41d8cd98f00b204e9800998ecf84273").build();
        DeviceIdUtils.checkAndGenerateDeviceId(deviceDTO);
        printResult(deviceDTO);

        System.out.println("【测试】 采集imei，imei传错： cfcd208495d565ef66e7dff9f98764da");
        deviceDTO = DeviceDTO.builder().imei_md5(DeviceMd5BlackEnum.ERROR_1.getDeviceMd5())
                .deviceType(DeviceTypeEnum.IMEI.getCode()).build();
        DeviceIdUtils.checkAndGenerateDeviceId(deviceDTO);
        printResult(deviceDTO);

        System.out.println("【测试】 采集 IDFA，IDFA 传错： cfcd208495d565ef66e7dff9f98764da");
        deviceDTO = DeviceDTO.builder().idfa("cfcd208495d565ef66e7dff9f98764da")
                .deviceType(DeviceTypeEnum.IDFA.getCode()).build();
        DeviceIdUtils.checkAndGenerateDeviceId(deviceDTO);
        printResult(deviceDTO);

        String uuidAndMd5 = DeviceIdUtils.getUUIDAndMd5(DeviceTypeEnum.THIRD_DEVICE.getDesc());
        System.out.println("tcid随机串：" + uuidAndMd5);
    }

    private static void printResult(DeviceDTO deviceDTO) {
        System.out.println(deviceDTO);
        Assert.isTrue(LegalMd5Factory.isLowerCase32Md5(deviceDTO.getDeviceId()), "deviceId是否合法");
    }
}
