package cn.tuia.tools.deviceid;


import org.apache.commons.codec.digest.DigestUtils;
import org.apache.commons.lang.StringUtils;

import java.util.UUID;

/**
 * 全局deviceId生成器
 * CF文档：http://cf.dui88.com/pages/viewpage.action?pageId=63937519
 * （1）deviceId生成规则：imei==>idfa==oaid==>uuid
 * （2）tcid生成：uuid会根据输入的 deviceType，生成前缀
 * （3）返回合法的deviceId（32位md5），不合法的32位md5会置为null，改成正确的返回
 */
public class DeviceIdUtilService {

    /**
     * 根据入参的DeviceDTO对象进行处理,自动MD5、自动校验、自动识别错误MD5
     * DeviceDTO deviceDTO = DeviceDTO.builder().imei("351710058880864").deviceId("11").build();
     * DeviceIdUtilService.checkAndGenerateDeviceId(deviceDTO);
     */
    public static void checkAndGenerateDeviceId(DeviceDTO deviceDTO) {

        // 1. 封装成md5===》imei==>idfa==oaid==>uuid
        transferMd5DTO(deviceDTO);

        // 2. deviceId不为空，且合法，就直接返回
        String deviceId = deviceDTO.getDeviceId();
        if (StringUtils.isNotEmpty(deviceId)) {
            if (LegalMd5Factory.checkLegalMd5(deviceId, DeviceTypeEnum.DEVICE_ID)) {
                deviceDTO.setLegalFlag(true);
                deviceDTO.setDeviceType(DeviceTypeEnum.DEVICE_ID.getCode());
                return;
            }
        }

        // 3. 获取设备号，md5的imei==>idfa==oaid==>uuid
        Integer deviceType = DeviceTypeEnum.UNKNOWN.getCode();
        if (StringUtils.isNotEmpty(deviceDTO.getImei_md5())) {
            deviceType = DeviceTypeEnum.IMEI.getCode();
            deviceId = deviceDTO.getImei_md5();
        } else if (StringUtils.isNotEmpty(deviceDTO.getIdfa_md5())) {
            deviceType = DeviceTypeEnum.IDFA.getCode();
            deviceId = deviceDTO.getIdfa_md5();
        } else if (StringUtils.isNotEmpty(deviceDTO.getOaid_md5())) {
            deviceType = DeviceTypeEnum.OAID.getCode();
            deviceId = deviceDTO.getOaid_md5();
        }

        // 4. 如果已经命中（不是unknown了）设备就直接返回
        if (!DeviceTypeEnum.UNKNOWN.getCode().equals(deviceType)) {
            deviceDTO.setLegalFlag(true);
            deviceDTO.setDeviceId(deviceId);
            deviceDTO.setDeviceType(deviceType);
            return;
        }

        // 5. 未命中走uuid策略，根据【入参的deviceType】进行拼接
        deviceType = deviceDTO.getDeviceType();
        //    设备号类型获取拼接的设备
        DeviceTypeEnum deviceTypeEnum = DeviceTypeEnum.seekValue(deviceType);
        String desc = deviceTypeEnum.getDesc();
        deviceId = getUUIDAndMd5(desc);

        deviceDTO.setLegalFlag(false);
        deviceDTO.setDeviceType(deviceTypeEnum.getCode());
        deviceDTO.setDeviceId(deviceId);
    }

    /**
     * 将设备信息封装成md5
     * * IMEI：351710058880864
     * * IDFA: CCD6E1CD-8C4B-40CB-8A62-4BBC7AFE07D6
     * * oaid：351710058880864
     *
     * @param deviceDTO
     */
    private static void transferMd5DTO(DeviceDTO deviceDTO) {
        // 1. imei号处理
        TempObjectMd5 objectMd51 = new TempObjectMd5(deviceDTO.getImei(), deviceDTO.getImei_md5());
        checkLegalMd5(objectMd51, DeviceTypeEnum.IMEI);
        deviceDTO.setImei(objectMd51.getStr());
        deviceDTO.setImei_md5(objectMd51.getStrMd5());

        // 2. idfa处理
        TempObjectMd5 objectMd52 = new TempObjectMd5(deviceDTO.getIdfa(), deviceDTO.getIdfa_md5());
        checkLegalMd5(objectMd52, DeviceTypeEnum.IDFA);
        deviceDTO.setIdfa(objectMd52.getStr());
        deviceDTO.setIdfa_md5(objectMd52.getStrMd5());

        // 3. oaid 处理
        TempObjectMd5 objectMd53 = new TempObjectMd5(deviceDTO.getOaid(), deviceDTO.getOaid_md5());
        checkLegalMd5(objectMd53, DeviceTypeEnum.OAID);
        deviceDTO.setOaid(objectMd53.getStr());
        deviceDTO.setOaid_md5(objectMd53.getStrMd5());

    }

    /**
     * imei合法，设置md5（imei）到 【imeiMd5】
     * imei不合法，如果是md5的，就设置到【imeiMd5Temp】==>设置【imei】为空
     * <p>
     * imeiMd5合法，设置到【imeiMd5】（不动）
     * imeiMd5不合法，设置imeiMd5Temp到【imeiMd5】
     */
    private static void checkLegalMd5(TempObjectMd5 objectMd5, DeviceTypeEnum typeEnum) {
        String str = objectMd5.getStr();
        // 1. imei合法，设置md5（imei）到 【imeiMd5】
        if (LegalMd5Factory.checkLegalMd5(str, typeEnum)) {
            objectMd5.setStrMd5(DigestUtils.md5Hex(str));
            return;
        }
        objectMd5.setStr("");

        // 2. 不合法的imei，有可能是传错了成MD5的imei
        String imeiMd5Temp = "";
        if (LegalMd5Factory.isLowerCase32Md5(str)) {
            imeiMd5Temp = str;
        }

        // 3. imeiMd5不合法，设置imeiMd5Temp到【imeiMd5】
        String strMd5 = objectMd5.getStrMd5();
        if (StringUtils.isEmpty(strMd5)) {
            return;
        }
        if (!LegalMd5Factory.isLowerCase32Md5(strMd5)) {
            objectMd5.setStrMd5(imeiMd5Temp);
        }

        // 4. imeiMd5设置成了imei
        if (LegalMd5Factory.checkLegalMd5(strMd5, typeEnum)) {
            objectMd5.setStr(strMd5);
            objectMd5.setStrMd5(DigestUtils.md5Hex(strMd5));
        }


    }

    /**
     * 生成带时间错的uuid，tcid也采用当前策略
     * 1.时间戳
     * 2.带标志信息：设备类型
     * IMEI(1, "1010"),
     * IDFA(2, "1020"),
     * OAID(3, "1011"),
     * 合法的deviceId是md5小写32位的
     * DEVICE_ID(4, "1030"),
     * UUID(5, "1040"),
     * UNKNOWN(6, "1050");
     *
     * @return
     */
    public static String getUUIDAndMd5(String desc) {
        if (StringUtils.isEmpty(desc) || desc.length() > 4) {
            desc = DeviceTypeEnum.UNKNOWN.getDesc();
        }
        int length = desc.length();
        String uuid = UUID.randomUUID().toString() + System.currentTimeMillis();
        uuid = DigestUtils.md5Hex(uuid);

        // 替换后的uuid
        uuid = desc + uuid.substring(length, uuid.length());
        return uuid;
    }

}
