/**
 * Copyright (c) 2019, duiba.com.cn All Rights Reserved.
 */
package cn.tuia.tools.deviceid.v2;

import lombok.ToString;
import org.apache.commons.codec.digest.DigestUtils;
import org.apache.commons.collections4.MapUtils;
import org.apache.commons.lang3.StringUtils;

import java.util.Map;

import static cn.tuia.tools.deviceid.v2.ImeiAndIdfaConstant.*;

/**
 * 描述: 设备唯一信息对象
 * <p>
 * 提供有关imei、idfa相关服务
 * <p>
 * 这个类提供的功能：根据入参(imei/idfa/oaid)，经过严格的真实设备号格式校验，最终得到真正符合设备号格式入参的MD5值。
 * <p>
 * 你该如何使用这个功能？
 * 1.调用构造方法 new DeviceInfo("imei", "idfa", "oaid");
 *      1.1.尽量保证对应的值，传入对应的字段。
 *          1.1.1.如果你不能确定到底是哪个，请每一个都传入，例如new DeviceInfo("deviceId", "deviceId", "deviceId");
 *          最终按照imei > idfa > oaid 的优先级来使用。
 *      1.2.每个字段既可接受明文，也可接收密文。
 * 2.调用build(isEncrypt)方法。
 *      2.1.如果确定是密文，请传密文标识；如果确定是明文，请传明文标识；如果不确定，传入default。
 * 3.调用get方法，获取你想使用的值
 *      3.1.imeiMd5/idfaMd5/oaidMd5目前给全链路字段用的
 *      3.2.deviceType/deviceUniqueId最终作为了真实链路中的deviceId使用的
 *
 * <p>
 * 如果你懒得看上面的描述或者看不明白，直接去open-web看如何使用也行。
 *
 * @author guopengfei@duiba.com.cn
 * @version v1.0
 * @date 2019/11/8 19:10
 */
@ToString
public class DeviceInfo {

    private String imei;
    private String idfa;
    private String oaid;

    //****以下下都是生成的值****

    /* 原始设备号是否加密 */
    private Boolean isEncrypt;

    /* 设备类型，参考枚举 */
    private DeviceType deviceType;

    /* 真实MD5后设备号 */
    private String deviceUniqueId;

    private String imeiMd5;
    private String idfaMd5;
    private String oaidMd5;

    public DeviceInfo() {
    }

    public DeviceInfo(String imei, String idfa, String oaid) {
        this.imei = imei;
        this.idfa = idfa;
        this.oaid = oaid;

        // 命中黑名单规则，重置初始值
        resetInitValue();
    }

    public DeviceInfo(String imei, String idfa, String oaid, Map<String,String> extBlack) {
        this.imei = imei;
        this.idfa = idfa;
        this.oaid = oaid;

        // 命中黑名单规则，重置初始值
        resetInitValue(extBlack);
    }

    public String getImei() {
        return imei;
    }

    public String getIdfa() {
        return idfa;
    }

    public String getOaid() {
        return oaid;
    }

    public Boolean getEncrypt() {
        return isEncrypt;
    }

    public DeviceType getDeviceType() {
        return deviceType;
    }

    public String getDeviceUniqueId() {
        return deviceUniqueId;
    }

    public String getImeiMd5() {
        return imeiMd5;
    }

    public String getIdfaMd5() {
        return idfaMd5;
    }

    public String getOaidMd5() {
        return oaidMd5;
    }

    /**
     * 以下三个方法为内部使用private
     **/

    private DeviceInfo setDeviceUniqueId(String deviceUniqueId) {
        this.deviceUniqueId = deviceUniqueId;
        return this;
    }

    private DeviceInfo setDeviceType(DeviceType deviceType) {
        this.deviceType = deviceType;
        return this;
    }

    private DeviceInfo setEncrypt(Boolean encrypt) {
        isEncrypt = encrypt;
        return this;
    }

    private DeviceInfo setImeiMd5(String imeiMd5) {
        this.imeiMd5 = imeiMd5;
        return this;
    }

    private DeviceInfo setIdfaMd5(String idfaMd5) {
        this.idfaMd5 = idfaMd5;
        return this;
    }

    private DeviceInfo setOaidMd5(String oaidMd5) {
        this.oaidMd5 = oaidMd5;
        return this;
    }

    /** 以上三个方法为内部使用private **/

    /**
     * 对象内容是否为空
     *
     * @return
     */
    public boolean isNull() {
        return StringUtils.isBlank(imei) && StringUtils.isBlank(idfa) && StringUtils.isBlank(oaid);
    }

    /**
     * 构建后无真实设备号
     *
     * @return
     */
    public boolean isNullAfterBuild() {
        return deviceType == null;
    }

    /**
     * 根据API加密参数构建对象
     *
     * @param isEncrypt
     * @return
     */
    public DeviceInfo build(String isEncrypt) {
        if (StringUtils.isBlank(imei) && StringUtils.isBlank(idfa) && StringUtils.isBlank(oaid)) {
            return this;
        }
        if (StringUtils.isBlank(isEncrypt)) {
            isEncrypt = ENCRYPT_DEFAULT;
        }
        switch (isEncrypt) {
            case ENCRYPT_TRUE:
                setIfEncryptTrue();
                break;
            case ENCRYPT_FALSE:
                setIfEncryptFalse();
                break;
            default:
                setIfEncryptDefault();
        }

        return this;
    }

    /**
     * 设置值，If isEncrypt == false
     */
    private void setIfEncryptFalse() {
        // 按照oaid,idfa,imei的顺序设置，可以保证DeviceUniqueId是优先使用imei的
        if (ImeiAndIdfaUtils.isOaid(oaid)) {
            String oaidMd5 = DigestUtils.md5Hex(oaid);
            this.setDeviceType(DeviceType.OAID).setOaidMd5(oaidMd5)
                    .setDeviceUniqueId(oaidMd5).setEncrypt(Boolean.FALSE);
        }

        if (ImeiAndIdfaUtils.isIdfa(idfa)) {
            String idfaMd5 = DigestUtils.md5Hex(idfa);
            this.setDeviceType(DeviceType.IDFA).setIdfaMd5(idfaMd5)
                    .setDeviceUniqueId(idfaMd5).setEncrypt(Boolean.FALSE);
        }

        if (ImeiAndIdfaUtils.isImei(imei)) {
            String imeiMd5 = DigestUtils.md5Hex(imei);
            this.setDeviceType(DeviceType.IMEI).setImeiMd5(imeiMd5)
                    .setDeviceUniqueId(imeiMd5).setEncrypt(Boolean.FALSE);
        }
    }

    /**
     * 设置值，If isEncrypt == true
     */
    private void setIfEncryptTrue() {
        // 按照oaid,idfa,imei的顺序设置，可以保证DeviceUniqueId是优先使用imei的
        if (ImeiAndIdfaUtils.isLowerCase32Md5(oaid)) {
            this.setDeviceType(DeviceType.OAID).setOaidMd5(oaid)
                    .setDeviceUniqueId(oaid).setEncrypt(Boolean.TRUE);
        } else if (ImeiAndIdfaUtils.isUpperCase32Md5(oaid)) {
            String lowerCaseOaid = StringUtils.lowerCase(oaid);
            this.setDeviceType(DeviceType.OAID).setOaidMd5(lowerCaseOaid)
                    .setDeviceUniqueId(lowerCaseOaid).setEncrypt(Boolean.TRUE);
        }

        if (ImeiAndIdfaUtils.isLowerCase32Md5(idfa)) {
            this.setDeviceType(DeviceType.IDFA).setIdfaMd5(idfa)
                    .setDeviceUniqueId(idfa).setEncrypt(Boolean.TRUE);
        } else if (ImeiAndIdfaUtils.isUpperCase32Md5(idfa)) {
            String lowerCaseIdfa = StringUtils.lowerCase(idfa);
            this.setDeviceType(DeviceType.IDFA).setIdfaMd5(lowerCaseIdfa)
                    .setDeviceUniqueId(lowerCaseIdfa).setEncrypt(Boolean.TRUE);
        }

        if (ImeiAndIdfaUtils.isLowerCase32Md5(imei)) {
            this.setDeviceType(DeviceType.IMEI).setImeiMd5(imei)
                    .setDeviceUniqueId(imei).setEncrypt(Boolean.TRUE);
        } else if (ImeiAndIdfaUtils.isUpperCase32Md5(imei)) {
            String lowerCaseImei = StringUtils.lowerCase(imei);
            this.setDeviceType(DeviceType.IMEI).setImeiMd5(lowerCaseImei)
                    .setDeviceUniqueId(lowerCaseImei).setEncrypt(Boolean.TRUE);
        }
    }

    /**
     * 设置值，If isEncrypt == 未知
     */
    private void setIfEncryptDefault() {
        // 按照oaid,idfa,imei的顺序设置，可以保证DeviceUniqueId是优先使用imei的

        // oaid<由于oaid是没有规则的，oaid必须先设置明文，后设置密文>
        if (ImeiAndIdfaUtils.isOaid(oaid)) {
            String oaidMd5 = DigestUtils.md5Hex(oaid);
            this.setDeviceType(DeviceType.OAID).setOaidMd5(oaidMd5)
                    .setDeviceUniqueId(oaidMd5).setEncrypt(Boolean.FALSE);
        }
        if (ImeiAndIdfaUtils.isLowerCase32Md5(oaid)) {
            this.setDeviceType(DeviceType.OAID).setOaidMd5(oaid)
                    .setDeviceUniqueId(oaid).setEncrypt(Boolean.TRUE);
        } else if (ImeiAndIdfaUtils.isUpperCase32Md5(oaid)) {
            String lowerCaseOaid = StringUtils.lowerCase(oaid);
            this.setDeviceType(DeviceType.OAID).setOaidMd5(lowerCaseOaid)
                    .setDeviceUniqueId(lowerCaseOaid).setEncrypt(Boolean.TRUE);
        }

        // idfa
        if (ImeiAndIdfaUtils.isLowerCase32Md5(idfa)) {
            this.setDeviceType(DeviceType.IDFA).setIdfaMd5(idfa)
                    .setDeviceUniqueId(idfa).setEncrypt(Boolean.TRUE);
        } else if (ImeiAndIdfaUtils.isUpperCase32Md5(idfa)) {
            String lowerCaseIdfa = StringUtils.lowerCase(idfa);
            this.setDeviceType(DeviceType.IDFA).setIdfaMd5(lowerCaseIdfa)
                    .setDeviceUniqueId(lowerCaseIdfa).setEncrypt(Boolean.TRUE);
        }
        if (ImeiAndIdfaUtils.isIdfa(idfa)) {
            String idfaMd5 = DigestUtils.md5Hex(idfa);
            this.setDeviceType(DeviceType.IDFA).setIdfaMd5(idfaMd5)
                    .setDeviceUniqueId(idfaMd5).setEncrypt(Boolean.FALSE);
        }

        // imei
        if (ImeiAndIdfaUtils.isLowerCase32Md5(imei)) {
            this.setDeviceType(DeviceType.IMEI).setImeiMd5(imei)
                    .setDeviceUniqueId(imei).setEncrypt(Boolean.TRUE);
        } else if (ImeiAndIdfaUtils.isUpperCase32Md5(imei)) {
            String lowerCaseImei = StringUtils.lowerCase(imei);
            this.setDeviceType(DeviceType.IMEI).setImeiMd5(lowerCaseImei)
                    .setDeviceUniqueId(lowerCaseImei).setEncrypt(Boolean.TRUE);
        }
        if (ImeiAndIdfaUtils.isImei(imei)) {
            String imeiMd5 = DigestUtils.md5Hex(imei);
            this.setDeviceType(DeviceType.IMEI).setImeiMd5(imeiMd5)
                    .setDeviceUniqueId(imeiMd5).setEncrypt(Boolean.FALSE);
        }
    }

    /**
     * 命中黑名单之后，重置传入的值
     */
    private void resetInitValue() {
        if (ImeiAndIdfaUtils.isBlackDevice(imei)) {
            imei = null;
        }
        if (ImeiAndIdfaUtils.isBlackDevice(idfa)) {
            idfa = null;
        }
        if (ImeiAndIdfaUtils.isBlackDevice(oaid)) {
            oaid = null;
        }
    }

    private void resetInitValue(Map<String,String> extBlack) {
        if(MapUtils.isEmpty(extBlack)) {
            resetInitValue();
            return;
        }

        if (ImeiAndIdfaUtils.isBlackDevice(imei) || ImeiAndIdfaUtils.isBlackDeviceExt(imei,extBlack)) {
            imei = null;
        }
        if (ImeiAndIdfaUtils.isBlackDevice(idfa) || ImeiAndIdfaUtils.isBlackDeviceExt(idfa,extBlack)) {
            idfa = null;
        }
        if (ImeiAndIdfaUtils.isBlackDevice(oaid) || ImeiAndIdfaUtils.isBlackDeviceExt(oaid,extBlack)) {
            oaid = null;
        }
    }

    /**
     * 获取原始的imei或idfa或oaid
     *
     * @return
     */
    public String getOriginDeviceUniqueId() {
        if (StringUtils.isNotBlank(imei)) {
            return imei;
        }
        if (StringUtils.isNotBlank(idfa)) {
            return idfa;
        }
        if (StringUtils.isNotBlank(oaid)) {
            return oaid;
        }
        return "";
    }
}
