package cn.yoofans.knowledge.center.api.param.flow;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Iterator;
import java.util.List;
import java.util.Objects;

import org.springframework.data.domain.Page;
import org.springframework.data.domain.PageImpl;
import org.springframework.data.domain.Pageable;
import org.springframework.data.domain.Sort;

/**
 * Created by zhangshun on 2018/5/24,08:50:56.
 */
public class FlowPageImpl<T> implements Page<T>, Serializable {

    private static final long serialVersionUID = 867755909294344406L;

    private final List<T> content = new ArrayList<T>();

    private final Pageable pageable;

    private final long total;

    private final long last;

    /**
     * Constructor of {@code PageImpl}.
     *
     * @param content  the content of this page
     * @param pageable the paging information
     * @param total    the total amount of items available
     */
    public FlowPageImpl(List<T> content, Pageable pageable, long total, long last) {

        if (null == content) {
            throw new IllegalArgumentException("Content must not be null!");
        }

        this.content.addAll(content);
        this.pageable = pageable;
        this.total = total;
        this.last = last;
    }

    /**
     * Creates a new {@link PageImpl} with the given content. This will result in the created {@link Page} being identical
     * to the entire {@link List}.
     *
     * @param content
     */
    public FlowPageImpl(List<T> content) {
        this(content, null, (null == content) ? 0 : content.size(), 0);
    }

    public int getNumber() {
        return pageable == null ? 0 : pageable.getPageNumber();
    }

    public int getSize() {
        return pageable == null ? 0 : pageable.getPageSize();
    }

    public int getTotalPages() {
        return getSize() == 0 ? 0 : (int) Math.ceil((double) total / (double) getSize());
    }

    public int getNumberOfElements() {
        return content.size();
    }

    public long getTotalElements() {
        return total;
    }

    public boolean hasPreviousPage() {
        return getNumber() > 0;
    }

    public boolean isFirstPage() {
        return !hasPreviousPage();
    }

    public boolean hasNextPage() {
        return ((getNumber() + 1) * getSize()) < total;
    }

    public boolean isLastPage() {
        return !hasNextPage();
    }

    public Iterator<T> iterator() {
        return content.iterator();
    }

    public List<T> getContent() {
        return Collections.unmodifiableList(content);
    }

    public boolean hasContent() {
        return !content.isEmpty();
    }

    public Sort getSort() {
        return pageable == null ? null : pageable.getSort();
    }

    public Pageable getPageable() {
        return pageable;
    }

    public long getTotal() {
        return total;
    }

    public long getLast() {
        return last;
    }

    @Override
    public String toString() {
        String contentType = "UNKNOWN";
        if (content.size() > 0) {
            contentType = content.get(0).getClass().getName();
        }
        return String
            .format("Page %s of %d last %s containing %s instances", getNumber(), getTotalPages(), last, contentType);
    }

    @Override
    public boolean equals(Object o) {
        if (this == o)
            return true;
        if (o == null || getClass() != o.getClass())
            return false;
        FlowPageImpl<?> flowPage = (FlowPageImpl<?>) o;
        return total == flowPage.total && last == flowPage.last && Objects.equals(content, flowPage.content) && Objects
            .equals(pageable, flowPage.pageable);
    }

    @Override
    public int hashCode() {
        return Objects.hash(content, pageable, total, last);
    }
}
