/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.flink.table.temptable;

import org.apache.flink.runtime.io.network.partition.ResultPartitionID;
import org.apache.flink.runtime.io.network.partition.external.ExternalBlockShuffleServiceConfiguration;
import org.apache.flink.runtime.io.network.partition.external.ExternalBlockShuffleServiceOptions;
import org.apache.flink.runtime.io.network.partition.external.LocalResultPartitionResolver;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.File;
import java.io.IOException;
import java.util.Collections;
import java.util.Set;

/**
 * A default external result partition resolver for default shuffle service.
 * This class is uesd for locate cache data generated by {@link TableStorage}.
 */
public class DefaultExternalResultPartitionResolver extends LocalResultPartitionResolver {

	private static final Logger LOG = LoggerFactory.getLogger(DefaultExternalResultPartitionResolver.class);

	private final String rootPath;

	public DefaultExternalResultPartitionResolver(ExternalBlockShuffleServiceConfiguration shuffleServiceConfiguration) {
		super(shuffleServiceConfiguration);
		this.rootPath = shuffleServiceConfiguration.getConfiguration()
			.getString(ExternalBlockShuffleServiceOptions.LOCAL_DIRS);
	}

	@Override
	public ResultPartitionFileInfo getResultPartitionDir(ResultPartitionID resultPartitionID) throws IOException {
		long tableHashId = resultPartitionID.getPartitionId().getLowerPart();
		long tablePartition = resultPartitionID.getPartitionId().getUpperPart();
		return new DefaultResultPartitionFileInfo(
			rootPath,
			rootPath + tableHashId + File.separator + tablePartition + File.separator);
	}

	@Override
	public void recycleResultPartition(ResultPartitionID resultPartitionID) {
		// do nothing for now.
		LOG.info("DefaultExternalResultPartitionResolver recycleResultPartition");
	}

	@Override
	public void initializeApplication(String user, String appId) {
		LOG.info("DefaultExternalResultPartitionResolver initializeApplication");
	}

	@Override
	public Set<ResultPartitionID> stopApplication(String appId) {
		LOG.info("DefaultExternalResultPartitionResolver initializeApplication");
		return Collections.emptySet();
	}

	@Override
	public void stop() {
		// do nothing for now.
		LOG.info("DefaultExternalResultPartitionResolver stops.");
	}

	/**
	 * DefaultResultPartitionFileInfo for default shuffle service.
	 */
	public static class DefaultResultPartitionFileInfo implements ResultPartitionFileInfo {

		private final String rootDir;

		private final String partitionDir;

		public DefaultResultPartitionFileInfo(String rootDir, String partitionDir) {
			this.rootDir = rootDir;
			this.partitionDir = partitionDir;
		}

		@Override
		public String getRootDir() {
			return rootDir;
		}

		@Override
		public String getPartitionDir() {
			return partitionDir;
		}

		@Override
		public long getConsumedPartitionTTL() {
			return Long.MAX_VALUE;
		}

		@Override
		public long getPartialConsumedPartitionTTL() {
			return Long.MAX_VALUE;
		}
	}
}
