/*
 * Decompiled with CFR 0.152.
 */
package org.apache.flink.table.dataformat;

import java.math.BigDecimal;
import java.math.BigInteger;
import java.math.MathContext;
import java.math.RoundingMode;
import org.apache.flink.table.types.DecimalType;
import org.apache.flink.table.types.InternalType;
import org.apache.flink.table.types.Types;
import org.apache.flink.util.Preconditions;

public final class Decimal
implements Comparable<Decimal> {
    public static final int MAX_PS = 38;
    private static final MathContext MC_DIVIDE = new MathContext(38, RoundingMode.HALF_UP);
    public static final int MAX_COMPACT_PRECISION = 18;
    public static final int MAX_INT_DIGITS = 9;
    public static final int MAX_LONG_DIGITS = 18;
    public static final long[] POW10 = new long[19];
    private final int precision;
    private final int scale;
    private final long longVal;
    private BigDecimal decimalVal;

    private Decimal(int precision, int scale, long longVal, BigDecimal decimalVal) {
        this.precision = precision;
        this.scale = scale;
        this.longVal = longVal;
        this.decimalVal = decimalVal;
    }

    public boolean isCompact() {
        return Decimal.isCompact(this.precision);
    }

    public static boolean isCompact(int precision) {
        return precision <= 18;
    }

    public BigDecimal toBigDecimal() {
        BigDecimal bd = this.decimalVal;
        if (bd == null) {
            this.decimalVal = bd = BigDecimal.valueOf(this.longVal, this.scale);
        }
        return bd;
    }

    public int hashCode() {
        return this.toBigDecimal().hashCode();
    }

    @Override
    public int compareTo(Decimal that) {
        if (this.isCompact() && that.isCompact() && this.scale == that.scale) {
            return Long.compare(this.longVal, that.longVal);
        }
        return this.toBigDecimal().compareTo(that.toBigDecimal());
    }

    public boolean equals(Object o) {
        if (!(o instanceof Decimal)) {
            return false;
        }
        Decimal that = (Decimal)o;
        return this.compareTo(that) == 0;
    }

    public String toString() {
        return this.toBigDecimal().toPlainString();
    }

    public int signum() {
        if (this.isCompact()) {
            return Long.signum(this.longVal);
        }
        return this.decimalVal.signum();
    }

    private static ArithmeticException overflowException(String typeName, Object value) {
        return new ArithmeticException("numeric value out of range, type=" + typeName + ", value=" + value);
    }

    public static Decimal fromLong(long l, int precision, int scale) {
        Preconditions.checkArgument(precision > 0 && precision <= 18);
        Preconditions.checkArgument((l >= 0L ? l : -l) < POW10[precision]);
        return new Decimal(precision, scale, l, null);
    }

    public static Decimal fromBigDecimal(BigDecimal bd, int precision, int scale) {
        if ((bd = bd.setScale(scale, RoundingMode.HALF_UP)).precision() > precision) {
            return null;
        }
        long longVal = -1L;
        if (precision <= 18) {
            longVal = bd.movePointRight(scale).longValueExact();
        }
        return new Decimal(precision, scale, longVal, bd);
    }

    public static Decimal zero(int precision, int scale) {
        if (precision <= 18) {
            return new Decimal(precision, scale, 0L, null);
        }
        return Decimal.fromBigDecimal(BigDecimal.ZERO, precision, scale);
    }

    public Decimal copy() {
        return new Decimal(this.precision, this.scale, this.longVal, this.decimalVal);
    }

    public long toUnscaledLong() {
        assert (this.isCompact());
        return this.longVal;
    }

    public static Decimal fromUnscaledLong(int precision, int scale, long longVal) {
        assert (Decimal.isCompact(precision));
        return new Decimal(precision, scale, longVal, null);
    }

    public byte[] toUnscaledBytes() {
        if (!this.isCompact()) {
            return this.toBigDecimal().unscaledValue().toByteArray();
        }
        byte[] bytes = new byte[8];
        long l = this.longVal;
        for (int i = 0; i < 8; ++i) {
            bytes[7 - i] = (byte)l;
            l >>>= 8;
        }
        return bytes;
    }

    public static Decimal fromUnscaledBytes(int precision, int scale, byte[] bytes) {
        if (precision > 18) {
            BigDecimal bd = new BigDecimal(new BigInteger(bytes), scale);
            return new Decimal(precision, scale, -1L, bd);
        }
        assert (bytes.length == 8);
        long l = 0L;
        for (int i = 0; i < 8; ++i) {
            l <<= 8;
            l |= (long)(bytes[i] & 0xFF);
        }
        return new Decimal(precision, scale, l, null);
    }

    public double doubleValue() {
        if (this.isCompact()) {
            return (double)this.longVal / (double)POW10[this.scale];
        }
        return this.decimalVal.doubleValue();
    }

    public Decimal negate() {
        if (this.isCompact()) {
            return new Decimal(this.precision, this.scale, -this.longVal, null);
        }
        return new Decimal(this.precision, this.scale, -1L, this.decimalVal.negate());
    }

    public Decimal abs() {
        if (this.isCompact()) {
            if (this.longVal >= 0L) {
                return this;
            }
            return new Decimal(this.precision, this.scale, -this.longVal, null);
        }
        if (this.decimalVal.signum() >= 0) {
            return this;
        }
        return new Decimal(this.precision, this.scale, -1L, this.decimalVal.negate());
    }

    public Decimal floor() {
        BigDecimal bd = this.toBigDecimal().setScale(0, RoundingMode.FLOOR);
        return Decimal.fromBigDecimal(bd, bd.precision(), 0);
    }

    public Decimal ceil() {
        BigDecimal bd = this.toBigDecimal().setScale(0, RoundingMode.CEILING);
        return Decimal.fromBigDecimal(bd, bd.precision(), 0);
    }

    public int getPrecision() {
        return this.precision;
    }

    public int getScale() {
        return this.scale;
    }

    public static Decimal add(Decimal v1, Decimal v2, int precision, int scale) {
        if (v1.isCompact() && v2.isCompact() && v1.scale == v2.scale) {
            assert (scale == v1.scale);
            try {
                long ls = Math.addExact(v1.longVal, v2.longVal);
                return new Decimal(precision, scale, ls, null);
            }
            catch (ArithmeticException arithmeticException) {
                // empty catch block
            }
        }
        BigDecimal bd = v1.toBigDecimal().add(v2.toBigDecimal());
        return Decimal.fromBigDecimal(bd, precision, scale);
    }

    public static Decimal subtract(Decimal v1, Decimal v2, int precision, int scale) {
        if (v1.isCompact() && v2.isCompact() && v1.scale == v2.scale) {
            assert (scale == v1.scale);
            try {
                long ls = Math.subtractExact(v1.longVal, v2.longVal);
                return new Decimal(precision, scale, ls, null);
            }
            catch (ArithmeticException arithmeticException) {
                // empty catch block
            }
        }
        BigDecimal bd = v1.toBigDecimal().subtract(v2.toBigDecimal());
        return Decimal.fromBigDecimal(bd, precision, scale);
    }

    public static Decimal multiply(Decimal v1, Decimal v2, int precision, int scale) {
        BigDecimal bd = v1.toBigDecimal().multiply(v2.toBigDecimal());
        return Decimal.fromBigDecimal(bd, precision, scale);
    }

    public static Decimal divide(Decimal v1, Decimal v2, int precision, int scale) {
        BigDecimal bd = v1.toBigDecimal().divide(v2.toBigDecimal(), MC_DIVIDE);
        return Decimal.fromBigDecimal(bd, precision, scale);
    }

    public static Decimal mod(Decimal v1, Decimal v2, int precision, int scale) {
        BigDecimal bd = v1.toBigDecimal().remainder(v2.toBigDecimal(), MC_DIVIDE);
        return Decimal.fromBigDecimal(bd, precision, scale);
    }

    public static Decimal divideToIntegralValue(Decimal v1, Decimal v2, int precision, int scale) {
        BigDecimal bd = v1.toBigDecimal().divideToIntegralValue(v2.toBigDecimal());
        return Decimal.fromBigDecimal(bd, precision, scale);
    }

    private static long castToIntegral(Decimal dec, int numBits, String typeName) {
        BigDecimal bd = dec.toBigDecimal();
        bd = bd.setScale(0, RoundingMode.DOWN);
        return bd.longValue();
    }

    private static boolean withinRange(long r, int numBits) {
        return (r >>= numBits) == -1L || r == 0L;
    }

    public static long castToLong(Decimal dec) {
        return Decimal.castToIntegral(dec, 63, "LONG");
    }

    public static int castToInteger(Decimal dec) {
        return (int)Decimal.castToIntegral(dec, 31, "INT");
    }

    public static short castToShort(Decimal dec) {
        return (short)Decimal.castToIntegral(dec, 15, "SHORT");
    }

    public static byte castToByte(Decimal dec) {
        return (byte)Decimal.castToIntegral(dec, 7, "BYTE");
    }

    public static float castToFloat(Decimal dec) {
        return (float)dec.doubleValue();
    }

    public static double castToDouble(Decimal dec) {
        return dec.doubleValue();
    }

    public static Decimal castToDecimal(Decimal dec, int precision, int scale) {
        return Decimal.fromBigDecimal(dec.toBigDecimal(), precision, scale);
    }

    public static boolean castToBoolean(Decimal dec) {
        return dec.toBigDecimal().compareTo(BigDecimal.ZERO) != 0;
    }

    public static long castToTimestamp(Decimal dec) {
        return (long)(dec.doubleValue() * 1000.0);
    }

    public static Decimal castFrom(Decimal dec, int precision, int scale) {
        return Decimal.fromBigDecimal(dec.toBigDecimal(), precision, scale);
    }

    public static Decimal castFrom(String string, int precision, int scale) {
        return Decimal.fromBigDecimal(new BigDecimal(string), precision, scale);
    }

    public static Decimal castFrom(double val, int p, int s) {
        return Decimal.fromBigDecimal(BigDecimal.valueOf(val), p, s);
    }

    public static Decimal castFrom(long val, int p, int s) {
        return Decimal.fromBigDecimal(BigDecimal.valueOf(val), p, s);
    }

    public static Decimal castFrom(boolean val, int p, int s) {
        return Decimal.fromBigDecimal(BigDecimal.valueOf(val ? 1 : 0), p, s);
    }

    public static Decimal sign(Decimal b0) {
        if (b0.isCompact()) {
            return new Decimal(b0.precision, b0.scale, (long)b0.signum() * POW10[b0.scale], null);
        }
        return Decimal.fromBigDecimal(BigDecimal.valueOf(b0.signum()), b0.precision, b0.scale);
    }

    public static int compare(Decimal b1, Decimal b2) {
        return b1.compareTo(b2);
    }

    public static int compare(Decimal b1, long n2) {
        if (!b1.isCompact()) {
            return b1.decimalVal.compareTo(BigDecimal.valueOf(n2));
        }
        if (b1.scale == 0) {
            return Long.compare(b1.longVal, n2);
        }
        long i1 = b1.longVal / POW10[b1.scale];
        if (i1 == n2) {
            long l2 = n2 * POW10[b1.scale];
            return Long.compare(b1.longVal, l2);
        }
        return i1 > n2 ? 1 : -1;
    }

    public static int compare(Decimal b1, double n2) {
        return Double.compare(b1.doubleValue(), n2);
    }

    public static int compare(long n1, Decimal b2) {
        return -Decimal.compare(b2, n1);
    }

    public static int compare(double n1, Decimal b2) {
        return -Decimal.compare(b2, n1);
    }

    public static DecimalType inferDivisionType(int p1, int s1, int p2, int s2) {
        int s = Math.max(6, s1 + p2 + 1);
        int p = p1 - s1 + s2 + s;
        if (p > 38) {
            s = Math.max(6, 38 - (p - s));
            p = 38;
        }
        return new DecimalType(p, s);
    }

    public static DecimalType inferIntDivType(int p1, int s1, int p2, int s2) {
        int p = Math.min(38, p1 - s1 + s2);
        return new DecimalType(p, 0);
    }

    public static DecimalType inferAggSumType(int p, int s) {
        return new DecimalType(38, s);
    }

    public static DecimalType inferAggAvgType(int p, int s) {
        return Decimal.inferDivisionType(38, s, 20, 0);
    }

    public static DecimalType inferRoundType(int p, int s, int r) {
        if (r >= s) {
            return new DecimalType(p, s);
        }
        if (r < 0) {
            return new DecimalType(Math.min(38, 1 + p - s), 0);
        }
        return new DecimalType(1 + p - s + r, r);
    }

    public static Decimal sround(Decimal b0, int r) {
        if (r >= b0.scale) {
            return b0;
        }
        BigDecimal b2 = b0.toBigDecimal().movePointRight(r).setScale(0, RoundingMode.HALF_UP).movePointLeft(r);
        int p = b0.precision;
        int s = b0.scale;
        if (r < 0) {
            return Decimal.fromBigDecimal(b2, Math.min(38, 1 + p - s), 0);
        }
        return Decimal.fromBigDecimal(b2, 1 + p - s + r, r);
    }

    public static boolean is32BitDecimal(int precision) {
        return precision <= 9;
    }

    public static boolean is64BitDecimal(int precision) {
        return precision <= 18 && precision > 9;
    }

    public static boolean isByteArrayDecimal(int precision) {
        return precision > 18;
    }

    static {
        Decimal.POW10[0] = 1L;
        for (int i = 1; i < POW10.length; ++i) {
            Decimal.POW10[i] = 10L * POW10[i - 1];
        }
    }

    public static class Ref {
        private static String fullName(String methodName) {
            return Decimal.class.getCanonicalName() + "." + methodName;
        }

        public static String operator(String operator) {
            return Ref.fullName(Ref.opToName(operator));
        }

        private static String opToName(String op) {
            switch (op) {
                case "+": {
                    return "add";
                }
                case "-": {
                    return "subtract";
                }
                case "*": {
                    return "multiply";
                }
                case "/": {
                    return "divide";
                }
                case "%": {
                    return "mod";
                }
                case "DIV": {
                    return "divideToIntegralValue";
                }
            }
            throw new RuntimeException("Unsupported decimal arithmetic operator: " + op);
        }

        public static String castTo(Class<?> type) {
            return Ref.fullName("castTo" + type.getSimpleName());
        }

        public static String castTo(InternalType type) {
            String name = null;
            if (type.equals(Types.INT)) {
                name = "Integer";
            } else if (type.equals(Types.LONG)) {
                name = "Long";
            } else if (type.equals(Types.SHORT)) {
                name = "Short";
            } else if (type.equals(Types.BYTE)) {
                name = "Byte";
            } else if (type.equals(Types.FLOAT)) {
                name = "Float";
            } else if (type.equals(Types.DOUBLE)) {
                name = "Double";
            } else if (type instanceof DecimalType) {
                name = "Decimal";
            } else if (type.equals(Types.BOOLEAN)) {
                name = "Boolean";
            } else if (type.equals(Types.TIMESTAMP)) {
                name = "Timestamp";
            }
            return Ref.fullName("castTo" + name);
        }

        public static String castFrom() {
            return Ref.fullName("castFrom");
        }

        public static String compare() {
            return Ref.fullName("compare");
        }
    }
}

