package com.atlassian.mail.config;

import com.atlassian.core.util.ClassLoaderUtils;
import com.atlassian.core.util.XMLUtils;
import com.atlassian.mail.Settings;
import com.atlassian.mail.server.MailServerManager;
import org.apache.log4j.Logger;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.NodeList;

import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import java.io.IOException;
import java.io.InputStream;
import java.util.HashMap;
import java.util.Map;

import static com.atlassian.core.util.ClassLoaderUtils.loadClass;

public class ConfigLoader
{
    private static final Logger log = Logger.getLogger(ConfigLoader.class);
    private static final String DEFAULT_CONFIG_FILE = "mail-config.xml";

    private MailServerManager loadedManager;
    private Settings loadedSettings;

    public ConfigLoader(final String file)
    {
        //This is a bit of a hack for JBOSS
        final InputStream configurationFileAsStream = ClassLoaderUtils.getResourceAsStream(file, ConfigLoader.class);
        try
        {
            final DocumentBuilder db = DocumentBuilderFactory.newInstance().newDocumentBuilder();
            final Document xmlDoc = db.parse(configurationFileAsStream);
            final Element root = xmlDoc.getDocumentElement();
            final Element manager = (Element) root.getElementsByTagName("manager").item(0);
            final Class aClass = loadClass(manager.getAttribute("class"), this.getClass());
            final MailServerManager mailServerManager = (MailServerManager) aClass.newInstance();

            final Map<String, String> params = new HashMap<String, String>();
            final NodeList properties = manager.getElementsByTagName("property");
            if (properties.getLength() > 0)
            {
                for (int i = 0; i < properties.getLength(); i++)
                {
                    final Element property = (Element) properties.item(i);
                    final String name = XMLUtils.getContainedText(property, "name");
                    final String value = XMLUtils.getContainedText(property, "value");
                    params.put(name, value);
                }
            }
            mailServerManager.init(params);
            setLoadedManager(mailServerManager);

            final Element settingsNode = (Element) root.getElementsByTagName("settings").item(0);
            if (settingsNode == null)
            {
                setLoadedSettings(new Settings.Default());
            }
            else
            {
                final Class settingsClass = loadClass(settingsNode.getAttribute("class"), ConfigLoader.class);
                final Settings settingsInstance = (Settings) settingsClass.newInstance();
                setLoadedSettings(settingsInstance);
            }
        }
        catch (Exception e)
        {
            log.fatal(e, e);
            throw new RuntimeException("Error in mail configuration: " + e.getMessage(), e);
        }
        finally
        {
            try
            {
                configurationFileAsStream.close();
            }
            catch (IOException e2)
            {
                log.error(e2);
            }
        }
    }

    public static MailServerManager getServerManager()
    {
        return getServerManager(DEFAULT_CONFIG_FILE);
    }

    public static MailServerManager getServerManager(final String file)
    {
        final ConfigLoader configLoader = new ConfigLoader(file);
        return configLoader.getLoadedManager();
    }

    public static Settings getSettings(final String file)
    {
        final ConfigLoader configLoader = new ConfigLoader(file);
        return configLoader.getLoadedSettings();
    }

    public static Settings getSettings()
    {
        return getSettings(DEFAULT_CONFIG_FILE);
    }

    public MailServerManager getLoadedManager()
    {
        return loadedManager;
    }

    public void setLoadedManager(final MailServerManager loadedManager)
    {
        this.loadedManager = loadedManager;
    }

    public void setLoadedSettings(final Settings loadedSettings)
    {
        this.loadedSettings = loadedSettings;
    }

    public Settings getLoadedSettings()
    {
        return loadedSettings;
    }
}
