package com.dianping.cat.consumer.all.config.transform;

import static com.dianping.cat.consumer.all.config.Constants.ATTR_ID;
import static com.dianping.cat.consumer.all.config.Constants.ENTITY_ALL_CONFIG;
import static com.dianping.cat.consumer.all.config.Constants.ENTITY_NAME;
import static com.dianping.cat.consumer.all.config.Constants.ENTITY_REPORT;
import static com.dianping.cat.consumer.all.config.Constants.ENTITY_TYPE;

import com.dianping.cat.consumer.all.config.IEntity;
import com.dianping.cat.consumer.all.config.IVisitor;
import com.dianping.cat.consumer.all.config.entity.AllConfig;
import com.dianping.cat.consumer.all.config.entity.Name;
import com.dianping.cat.consumer.all.config.entity.Report;
import com.dianping.cat.consumer.all.config.entity.Type;

public class DefaultXmlBuilder implements IVisitor {

   private IVisitor m_visitor = this;

   private int m_level;

   private StringBuilder m_sb;

   private boolean m_compact;

   public DefaultXmlBuilder() {
      this(false);
   }

   public DefaultXmlBuilder(boolean compact) {
      this(compact, new StringBuilder(4096));
   }

   public DefaultXmlBuilder(boolean compact, StringBuilder sb) {
      m_compact = compact;
      m_sb = sb;
      m_sb.append("<?xml version=\"1.0\" encoding=\"utf-8\"?>\r\n");
   }

   public String buildXml(IEntity<?> entity) {
      entity.accept(m_visitor);
      return m_sb.toString();
   }

   protected void endTag(String name) {
      m_level--;

      indent();
      m_sb.append("</").append(name).append(">\r\n");
   }

   protected String escape(Object value) {
      return escape(value, false);
   }
   
   protected String escape(Object value, boolean text) {
      if (value == null) {
         return null;
      }

      String str = value.toString();
      int len = str.length();
      StringBuilder sb = new StringBuilder(len + 16);

      for (int i = 0; i < len; i++) {
         final char ch = str.charAt(i);

         switch (ch) {
         case '<':
            sb.append("&lt;");
            break;
         case '>':
            sb.append("&gt;");
            break;
         case '&':
            sb.append("&amp;");
            break;
         case '"':
            if (!text) {
               sb.append("&quot;");
               break;
            }
         default:
            sb.append(ch);
            break;
         }
      }

      return sb.toString();
   }
   
   protected void indent() {
      if (!m_compact) {
         for (int i = m_level - 1; i >= 0; i--) {
            m_sb.append("   ");
         }
      }
   }

   protected void startTag(String name) {
      startTag(name, false, null);
   }
   
   protected void startTag(String name, boolean closed, java.util.Map<String, String> dynamicAttributes, Object... nameValues) {
      startTag(name, null, closed, dynamicAttributes, nameValues);
   }

   protected void startTag(String name, java.util.Map<String, String> dynamicAttributes, Object... nameValues) {
      startTag(name, null, false, dynamicAttributes, nameValues);
   }

   protected void startTag(String name, Object text, boolean closed, java.util.Map<String, String> dynamicAttributes, Object... nameValues) {
      indent();

      m_sb.append('<').append(name);

      int len = nameValues.length;

      for (int i = 0; i + 1 < len; i += 2) {
         Object attrName = nameValues[i];
         Object attrValue = nameValues[i + 1];

         if (attrValue != null) {
            m_sb.append(' ').append(attrName).append("=\"").append(escape(attrValue)).append('"');
         }
      }

      if (dynamicAttributes != null) {
         for (java.util.Map.Entry<String, String> e : dynamicAttributes.entrySet()) {
            m_sb.append(' ').append(e.getKey()).append("=\"").append(escape(e.getValue())).append('"');
         }
      }

      if (text != null && closed) {
         m_sb.append('>');
         m_sb.append(escape(text, true));
         m_sb.append("</").append(name).append(">\r\n");
      } else {
         if (closed) {
            m_sb.append('/');
         } else {
            m_level++;
         }
   
         m_sb.append(">\r\n");
      }
   }

   @Override
   public void visitAllConfig(AllConfig allConfig) {
      startTag(ENTITY_ALL_CONFIG, null);

      if (!allConfig.getReports().isEmpty()) {
         for (Report report : allConfig.getReports().values()) {
            report.accept(m_visitor);
         }
      }

      endTag(ENTITY_ALL_CONFIG);
   }

   @Override
   public void visitName(Name name) {
      startTag(ENTITY_NAME, true, null, ATTR_ID, name.getId());
   }

   @Override
   public void visitReport(Report report) {
      startTag(ENTITY_REPORT, null, ATTR_ID, report.getId());

      if (!report.getTypes().isEmpty()) {
         for (Type type : report.getTypes().values()) {
            type.accept(m_visitor);
         }
      }

      endTag(ENTITY_REPORT);
   }

   @Override
   public void visitType(Type type) {
      startTag(ENTITY_TYPE, null, ATTR_ID, type.getId());

      if (!type.getNameList().isEmpty()) {
         for (Name name : type.getNameList()) {
            name.accept(m_visitor);
         }
      }

      endTag(ENTITY_TYPE);
   }
}
