package com.duiba.tuia.abtest.api.sdk.core;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import com.duiba.tuia.abtest.api.dto.*;
import com.duiba.tuia.abtest.api.remoteservice.RemoteABTestService;
import com.duiba.tuia.abtest.api.sdk.ABTest;
import com.google.common.cache.CacheBuilder;
import com.google.common.cache.CacheLoader;
import com.google.common.cache.LoadingCache;
import com.google.common.util.concurrent.ListenableFuture;
import com.google.common.util.concurrent.ListenableFutureTask;
import org.apache.commons.codec.digest.DigestUtils;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Service;

import javax.annotation.PostConstruct;
import javax.annotation.Resource;
import java.util.*;
import java.util.concurrent.*;
import java.util.stream.Collectors;

//@Service
public class ABTestV1 implements ABTest {

    /**
     * @Description 不提供热刷新
     * @Date 11:41 上午 2021/4/26
     * @Param
     * @return
     **/
    @Value("${abtest.log.time.milliseconds:100}")
    private Integer LOG_TIME;

    @Value("${abtest.cache.refresh.seconds:5}")
    private Integer CACHE_REFRESH_MILLS;

    @Value("${abtest.cache.expire.seconds:60}")
    private Integer CACHE_EXPIRE_MILLS;

    private static Random random = new Random();

    protected Logger logger = LoggerFactory.getLogger(getClass());


    //使用默认线程池
    @Resource
    private ExecutorService executorService;

    private LoadingCache<Long, TestSlotDTO> testPlanCache;

    private LoadingCache<Long, TestSlotDTO> advertTestPlanCache;

    private LoadingCache<String, TestSlotDTO> advertAlgoPlanCache;


    @Resource
    private RemoteABTestService remoteABTestService;

    /**
     * 初始化本地缓存
     */
    @PostConstruct
    public void init() {

        /**
         * 本地缓存用户存储ABTest配置信息
         * 最大缓存：5000 个
         * 每 1 分钟异步刷新一次缓存
         */
        testPlanCache = CacheBuilder.newBuilder()
                .initialCapacity(100)
                .maximumSize(5000)
                .refreshAfterWrite(CACHE_REFRESH_MILLS, TimeUnit.SECONDS)
                .expireAfterWrite(CACHE_EXPIRE_MILLS, TimeUnit.SECONDS)
                .build(cacheLoader);


        /**
         * 本地缓存用户存储ABTest配置信息
         * 最大缓存：5000 个
         * 每 1 分钟异步刷新一次缓存
         */
        advertTestPlanCache = CacheBuilder.newBuilder()
                .initialCapacity(100)
                .maximumSize(5000)
                .refreshAfterWrite(CACHE_REFRESH_MILLS, TimeUnit.MILLISECONDS)
                .expireAfterWrite(CACHE_EXPIRE_MILLS, TimeUnit.SECONDS)
                .build(advertCacheLoader);

        /**
         * 本地缓存用户存储ABTest配置信息
         * 最大缓存：5000 个
         * 每 1 小时异步刷新一次缓存
         */
        advertAlgoPlanCache = CacheBuilder.newBuilder()
                .initialCapacity(100)
                .maximumSize(5000)
                .refreshAfterWrite(CACHE_REFRESH_MILLS, TimeUnit.MILLISECONDS)
                .expireAfterWrite(CACHE_EXPIRE_MILLS, TimeUnit.SECONDS)
                .build(advertAlgoCacheLoader);


        //定时执行的线程池，每隔100毫秒执行一次(间隔时间可以由业务决定)，把所有堆积的请求
        ScheduledExecutorService executorService = Executors.newScheduledThreadPool(1);
        executorService.scheduleAtFixedRate(() -> {
            //在这里具体执行批量查询逻辑
            int size = queue.size();
            if (size == 0) {
                //若没有请求堆积，直接返回，等100毫秒再执行一次
                return;
            }
            //若有请求堆积把所有请求都拿出来
            List<JSONObject> requestTests = new ArrayList<>();
            for (int i = 0; i < size; i++) {
                //把请求拿出来
                JSONObject poll = queue.poll();
                requestTests.add(poll);
            }
            //至此请求已经被合并了
            logger.info("日志合并了" + requestTests.size() + "条！");
            //调用批量打日志接口
            //todo:上线前加回来
//            remoteABTestService.batchLog(requestTests);

        }, 0, LOG_TIME, TimeUnit.MILLISECONDS);

    }


    /**
     * 缓存加载相关实现，异步过程中返回旧的值，避免堵塞
     */
    private final CacheLoader cacheLoader = new CacheLoader<Long, TestSlotDTO>() {

        @Override
        public TestSlotDTO load(Long slotId) throws Exception {
            return remoteABTestService.getSlotCache(slotId);
        }

        @Override
        public ListenableFuture<TestSlotDTO> reload(Long key, TestSlotDTO oldValue) throws Exception {
            ListenableFutureTask<TestSlotDTO> task = ListenableFutureTask.create(() -> {
                try {
                    return load(key);
                } catch (Exception e) {
//                    log.error("reload abTest config error: key=" + key, e);
                    return oldValue;
                }
            });
            executorService.submit(task);
            return task;
        }
    };

    /**
     * 缓存加载相关实现，异步过程中返回旧的值，避免堵塞
     */
    private final CacheLoader advertCacheLoader = new CacheLoader<Long, TestSlotDTO>() {

        @Override
        public TestSlotDTO load(Long advertId) throws Exception {
            ABCacheRequestDto cacheRequestDto = new ABCacheRequestDto();
            cacheRequestDto.setAdvertId(advertId);
            return remoteABTestService.getAdvertCache(cacheRequestDto);
        }

        @Override
        public ListenableFuture<TestSlotDTO> reload(Long key, TestSlotDTO oldValue) throws Exception {
            ListenableFutureTask<TestSlotDTO> task = ListenableFutureTask.create(() -> {
                try {
                    return load(key);
                } catch (Exception e) {
//                    log.error("reload abTest config error: key=" + key, e);
                    return oldValue;
                }
            });
            executorService.submit(task);
            return task;
        }
    };

    /**
     * 缓存加载相关实现，异步过程中返回旧的值，避免堵塞
     */
    private final CacheLoader advertAlgoCacheLoader = new CacheLoader<String, TestSlotDTO>() {

        @Override
        public TestSlotDTO load(String layerCode) throws Exception {
            return remoteABTestService.getAdvertAlgoCache(layerCode);
        }

        @Override
        public ListenableFuture<TestSlotDTO> reload(String key, TestSlotDTO oldValue) throws Exception {
            ListenableFutureTask<TestSlotDTO> task = ListenableFutureTask.create(() -> {
                try {
                    return load(key);
                } catch (Exception e) {
//                    log.error("reload abTest config error: key=" + key, e);
                    return oldValue;
                }
            });
            executorService.submit(task);
            return task;
        }
    };


    @Override
    @Deprecated
    public ABResponseDto run(ABRequestDto request) {
        //1.获取测试计划,缓存中获取
        TestSlotDTO slotPlan = null;
        try {
            slotPlan = testPlanCache.get(request.getSlotId());
        } catch (ExecutionException e) {
            slotPlan = NullObject(request.getSlotId());
            logger.warn("异常返回空对象:[{}]", e);
        }
        List<TestSlotPlanDTO> testPlans = filter(request.getExtra(), request.getLayerCode(), slotPlan);
        if (testPlans.isEmpty()) {
            setInnerLog(request);
            //没有实验
            return new ABResponseDto(false, "没有实验");
        }
        //extra:遍历广告计划中的用户白名单,假如当前deviceId有命中,直接记录日志并返回
        UserWhiteListDO userWhiteListDO = filterWithWhiteList(testPlans, request.getDeviceId());
        if (Objects.nonNull(userWhiteListDO)) {
            //遍历得到当前白名单对应的实验计划
            TestSlotPlanDTO testPlan = new TestSlotPlanDTO();
            for (TestSlotPlanDTO testSlotPlanDTO : testPlans) {
                if (Objects.equals(testSlotPlanDTO.getPlanId(), userWhiteListDO.getPlanId())) {
                    testPlan = testSlotPlanDTO;
                }
            }
            //遍历得到当前白名单对应的实验组
            TestPlanGroupDTO current = new TestPlanGroupDTO();
            List<TestPlanGroupDTO> testPlanGroupDTOS = testPlan.getTestPlan().getTestPlanGroups().get(request.getLayerCode());
            for (TestPlanGroupDTO testPlanGroupDTO : testPlanGroupDTOS) {
                if (Objects.equals(testPlanGroupDTO.getGroupId(), userWhiteListDO.getGroupId())) {
                    current = testPlanGroupDTO;
                }
            }
            ABResultDto result;
            if (current != null) {
                result = new ABResultDto(true);
                result.setLayerCode(current.getTestLayerCode());
                result.setGroupId(current.getGroupId());
                result.setGroupName(current.getGroupName());
                result.setTestType(current.getTestType());
                result.setTestValue(current.getTestValue());
                result.setPlanId(testPlan.getPlanId());
            } else {
                result = new ABResultDto(false, request.getLayerCode(), "没有分流到测试组");
            }
            List<ABResultDto> results = new ArrayList<>();
            results.add(result);
            setInnerLog(testPlan, results, request);
            return new ABResponseDto(true, testPlan.getPlanId(), results);
        }
        //2.域分流 (同一层的实验计划之间分流)
        TestSlotPlanDTO plan = domainHash(testPlans, request, false, 0);
        if (plan == null) {
            setInnerLog(request);
            //没有命中实验
            return new ABResponseDto(false, "没有分流到实验");
        }
        //3.实验组之间分流  (增加广告算法定制逻辑,以及PV分流支持)
        List<ABResultDto> layerResult = layerHash(plan, request, false, false);
        setInnerLog(plan, layerResult, request);
        return new ABResponseDto(true, plan.getPlanId(), layerResult);
    }


    /**
     * @param request
     * @return com.duiba.tuia.abtest.api.dto.ABAdvertResponseDto
     * @Description 执行实验(广告定制逻辑)
     * 对全局流量生效,支持批量传层编号
     * @Param [request]
     */
    @Override
    @Deprecated
    public ABAdvertResponseDto advertRun(ABRequestDto request) {
        //逗号分隔的实验层编号转换为
        List<String> layerCodes = Arrays.asList(request.getLayerCode().split(","));
        //总的返回结果
        Map<String, ABResultDto> total = new HashMap<>();

        layerCodes.forEach(layerCode -> {
            //判断是否走域,默认不走
            Boolean isDomain = false;
            Integer domainPercent = 0;
            request.setLayerCode(layerCode);
            //根据传参中是否带有广告计划ID判断流量维度,是广告计划还是全局
            List<TestSlotPlanDTO> testPlans = new ArrayList<>();
            if (Objects.nonNull(request.getAdvertId())) {
                //根据deviceId分流,判断是否走域
                TestSlotDTO slotPlan = null;
                try {
                    slotPlan = advertTestPlanCache.get(request.getAdvertId());
                } catch (ExecutionException e) {
                    slotPlan = NullObject(request.getSlotId());
                    logger.warn("异常返回空对象:[{}]", e);
                }
                testPlans = filter(request.getExtra(), request.getLayerCode(), slotPlan);

                isDomain = isDomainHash(testPlans, request);
                //计算域实验占用的流量,如果存在域实验,在实验计划分流的时候,要去掉该部分流量(hash取模时)
                domainPercent = getDomainPercent(testPlans);
                //不走域的情况下,过滤掉所有域实验,防止在下一步正交分流的时候分流到该实验
                if (!isDomain) {
                    testPlans = filterNotDomainPlans(testPlans);
                }

            } else {
                //1.获取测试计划
                // 先尝试根据层编号获取广告算法实验计划
                TestSlotDTO slotPlan = null;
                try {
                    slotPlan = advertAlgoPlanCache.get(request.getLayerCode());
                } catch (ExecutionException e) {
                    slotPlan = NullObject(request.getLayerCode());
                    logger.warn("异常返回空对象:[{}]", e);
                }
                testPlans = filter(request.getExtra(), request.getLayerCode(), slotPlan);
            }
            if (testPlans.isEmpty()) {
                setInnerLog(request);
                //为空跳出本次循环
                return;
            }

            //2.域分流 (同一层的实验计划之间分流)
            TestSlotPlanDTO plan = domainHash(testPlans, request, isDomain, domainPercent);
            if (plan == null) {
                setInnerLog(request);
                //没有命中实验
                return;
            }

            //3.实验组之间分流  (增加广告算法定制逻辑,以及PV分流支持)
            List<ABResultDto> layerResult = layerHash(plan, request, true, isDomain);
            //4.获取实验组失败,打日志跳出本次循环
            if (!layerResult.get(0).isSuccess()) {
                setInnerLog(plan, layerResult, request);
                return;
            }
            setInnerLog(plan, layerResult, request);

            //5. 整理为 层编号 - layerResult 的 map,便于广告线同学获取
            total.put(layerResult.get(0).getLayerCode(), layerResult.get(0));
        });

        //判断返回result是否为空
        if (total.isEmpty()) {
            //为空返回查询失败
            return new ABAdvertResponseDto(false, "没有试验");
        }
        return new ABAdvertResponseDto(true, total);
    }

    /**
     * @param request
     * @return com.duiba.tuia.abtest.api.dto.ABAdvertResponseDto
     * @Description 广告定制run接口(支持批量传层编号)
     * @Param [request]
     */
    @Override
    @Deprecated
    public ABAdvertResponseDto advertRunWithRemote(ABRequestDto request) {
        return remoteABTestService.advertRun(request);
    }


    /**
     * @param requests
     * @return com.duiba.tuia.abtest.api.dto.ABAdvertResponseDto
     * @Description 广告实验分流对接接口, 批量广告计划
     * @Param [request]
     */
    @Override
    @Deprecated
    public ABAdvertResponseBatchDto advertRunBatch(List<ABRequestDto> requests) {
        List<ABResultDto> total = new ArrayList<>();

        requests.forEach(request -> {
            //根据传参中是否带有广告计划ID判断流量维度,是广告计划还是全局
            TestSlotDTO slotPlan = null;
            try {
                slotPlan = advertTestPlanCache.get(request.getAdvertId());
            } catch (ExecutionException e) {
                slotPlan = NullObject(request.getAdvertId());
                logger.warn("异常返回空对象:[{}]", e);
            }
            List<TestSlotPlanDTO> testPlans = filter(request.getExtra(), request.getLayerCode(), slotPlan);
            if (testPlans.isEmpty()) {
                setInnerLog(request);
                //为空跳出本次循环
                return;
            }
            //仅根据用户判断是否走域实验
            Boolean isDomain = isDomainHash(testPlans, request);
            //计算域实验占用的流量,如果存在域实验,在实验计划分流的时候,要去掉该部分流量(hash取模时)
            Integer domainPercent = getDomainPercent(testPlans);
            //不走域的情况下,过滤掉所有域实验,防止在下一步正交分流的时候分流到该实验
            if (!isDomain) {
                testPlans = filterNotDomainPlans(testPlans);
            }
            //2.域分流 (同一层的实验计划之间分流)
            TestSlotPlanDTO plan = domainHash(testPlans, request, isDomain, domainPercent);
            if (plan == null) {
                setInnerLog(request);
                //没有命中实验
                return;
            }

            //3.实验组之间分流  (增加广告算法定制逻辑,以及PV分流支持)
            List<ABResultDto> layerResult = layerHash(plan, request, true, isDomain);
            //4.获取实验组失败,打日志跳出本次循环
            if (!layerResult.get(0).isSuccess()) {
                setInnerLog(plan, layerResult, request);
                return;
            }
            setInnerLog(plan, layerResult, request);
            //extra参数返回
            layerResult.get(0).setExtra(request.getExtra());
            //5. 放到list,返回
            total.add(layerResult.get(0));
        });

        //判断返回result是否为空
        if (CollectionUtils.isEmpty(total)) {
            //为空返回查询失败
            return new ABAdvertResponseBatchDto(false, "没有试验");
        }
//            logger.info("最终返回数据:[{}]",JSON.toJSONString(total));
        return new ABAdvertResponseBatchDto(true, total);
    }


    /**
     * 根据域划分流量，选择一个实验计划
     *
     * @param testPlans
     * @param request
     * @param isDomain
     * @param domainPercent
     * @return
     */
    private TestSlotPlanDTO domainHash(List<TestSlotPlanDTO> testPlans, ABRequestDto request, Boolean isDomain, Integer domainPercent) {
        //域分流
        //增加PV分流支持
        int domainHash = 0;
        if (isDomain) {
            //域逻辑:仅用deviceId,不加盐,保证每层得到结果都一样
            domainHash = hash(request.getDeviceId()) % 100;
        } else if (Objects.equals(testPlans.get(0).getTestPlan().getSplitType(), 1)) {
            domainHash = random.nextInt(100 - domainPercent);
        } else {
            domainHash = hash(request.getDeviceId() + request.getSlotId() + request.getLayerCode()) % (100 - domainPercent);
        }
        TestSlotPlanDTO current = null;
        for (TestSlotPlanDTO slotPlan : testPlans) {
            if (slotPlan.getPercentDistrictSet().contains(domainHash)) {
                current = slotPlan;
                break;
            }
        }
        return current;
    }

    /**
     * 实验组分流,广告算法实验的场景下,不使用广告位ID参与分流
     *
     * @param planDTO
     * @param isAdvert
     * @param isDomain
     */
    private List<ABResultDto> layerHash(TestSlotPlanDTO planDTO, ABRequestDto request, boolean isAdvert, Boolean isDomain) {
        //层分流 (deviceId + slotId + layerCode + planId)
        int layerHash = 0;
        if (isDomain) {
            //保证同一个实验计划同一个人在每一层路由到同一个组
            layerHash = hash(request.getDeviceId() + planDTO.getPlanId()) % 100;
        } else if (Objects.equals(planDTO.getTestPlan().getSplitType(), 1)) {
            layerHash = random.nextInt(100);
        } else {
            //区分是否为广告算法场景,是的话就不让广告位ID参与Hash
            layerHash = isAdvert ? hash(request.getDeviceId() + request.getLayerCode() + planDTO.getPlanId()) % 100
                    : hash(request.getDeviceId() + request.getSlotId() + request.getLayerCode() + planDTO.getPlanId()) % 100;
        }
        List<TestPlanGroupDTO> groups = planDTO.getTestPlan().getTestPlanGroups().get(request.getLayerCode());
        TestPlanGroupDTO current = null;
        //遍历实验组,选中包含当前hash值的对象(0-99)
        for (TestPlanGroupDTO group : groups) {
            if (group.getPercentDistrictSet().contains(layerHash)) {
                current = group;
                break;
            }
        }
        ABResultDto result;
        if (current != null) {
            result = new ABResultDto(true);
            result.setLayerCode(current.getTestLayerCode());
            result.setGroupId(current.getGroupId());
            result.setGroupName(current.getGroupName());
            result.setTestType(current.getTestType());
            result.setTestValue(current.getTestValue());
            result.setPlanId(planDTO.getPlanId());
            result.setDomainStatus(isDomain ? 1 : 0);
        } else {
            result = new ABResultDto(false, request.getLayerCode(), "没有分流到测试组 hash:" + layerHash);
        }
        List<ABResultDto> results = new ArrayList<>();
        results.add(result);
        return results;
    }

    private int hash(String str) {
        String HashStr = DigestUtils.md5Hex(str);
        int hash = HashStr.length();
        for (int i = 0; i < HashStr.length(); i++) {
            hash = ((hash << 5) ^ (hash >> 27)) ^ HashStr.charAt(i);
        }
        int hashCode = (hash & 0x7FFFFFFF);
        return hashCode < 0 ? -hashCode : hashCode;
    }

    /**
     * @return com.duiba.tuia.abtest.api.dto.UserWhiteListDTO
     * @Author zhangliwei
     * @Description 遍历实验计划List, 判断当前设备号是否有命中
     * @Date 17:21 2020-12-23
     * @Param [testPlans]
     **/
    private UserWhiteListDO filterWithWhiteList(List<TestSlotPlanDTO> testPlans, String deviceId) {
        //遍历所有实验计划
        for (TestSlotPlanDTO testPlan : testPlans) {
            if (Objects.isNull(testPlan.getTestPlan()) || CollectionUtils.isEmpty(testPlan.getTestPlan().getWhiteLists())) {
                continue;
            }
            //遍历每个实验计划中的所有用户白名单,和当前能对上就直接返回
            List<UserWhiteListDO> whiteLists = testPlan.getTestPlan().getWhiteLists();
            for (UserWhiteListDO whiteList : whiteLists) {
                if (Objects.equals(whiteList.getDeviceId(), deviceId)) {
                    return whiteList;
                }
            }
        }
        return null;
    }

    LinkedBlockingDeque<JSONObject> queue = new LinkedBlockingDeque<>();

    private void setInnerLog(ABRequestDto request) {
        JSONObject json = new JSONObject();
        json.put("device_id", request.getDeviceId());
        json.put("slot_id", request.getSlotId());
        json.put("advert_id", request.getAdvertId());
        if (StringUtils.isNotBlank(request.getRid())) {
            json.put("rid", request.getRid());
        }

        List<PlanLogDTO> list = new ArrayList<>();
        PlanLogDTO planLog = new PlanLogDTO();
        planLog.setLayer_code(request.getLayerCode());
        planLog.setIs_hit(0);
        list.add(planLog);

        json.put("plan", JSON.toJSONString(list));

        queue.add(json);
    }

    /**
     * 设置日志,统计请求结果
     */
    private void setInnerLog(TestSlotPlanDTO plan, List<ABResultDto> layerResult, ABRequestDto request) {
        JSONObject json = new JSONObject();
        json.put("device_id", request.getDeviceId());
        json.put("slot_id", request.getSlotId());
        json.put("advert_id", request.getAdvertId());
        json.put("plan_id", plan.getPlanId());

        List<PlanLogDTO> list = new ArrayList<>();
        layerResult.forEach(layer -> {
            PlanLogDTO planLog = new PlanLogDTO();
            planLog.setLayer_code(layer.getLayerCode());
            if (Objects.equals(layer.isSuccess(), true)) {
                planLog.setIs_hit(1);
                planLog.setGroup_id(layer.getGroupId());
            } else {
                planLog.setIs_hit(0);
            }
            list.add(planLog);
        });

        json.put("plan", JSON.toJSONString(list));

        queue.add(json);
    }


    private List<TestSlotPlanDTO> filter(String extra, String layerCode, TestSlotDTO dto) {
        List<TestSlotPlanDTO> slotPlans = dto.getTestPlans();
        List<TestSlotPlanDTO> plans = new ArrayList<>();
        Date now = new Date();
        for (TestSlotPlanDTO slotPlan : slotPlans) {
            TestPlanDTO planDTO = slotPlan.getTestPlan();
            if (Objects.isNull(planDTO)) {
                logger.warn("实验计划为空!,layerCode:[{}],extra:[{}]", layerCode, extra);
            }
            if (now.before(planDTO.getStartTime())) {
                continue;
            }
            if (now.after(planDTO.getEndTime())) {
                continue;
            }
            if (planDTO.getTestPlanGroups().get(layerCode) == null) {
                continue;
            }
            List<TestConditionDTO> conditionValues = planDTO.getConditionValues();

            //无受众条件实验符合要求
            if (CollectionUtils.isEmpty(conditionValues)) {
                plans.add(slotPlan);
                continue;
            }

            if (StringUtils.isBlank(extra)) {
                continue;
            }

            //受众条件过滤
            Map<String, String> conditionValueMap = JSONObject.parseObject(extra)
                    .entrySet()
                    .stream()
                    .collect(Collectors.toMap(Map.Entry::getKey, entry -> entry.getValue().toString()));

            boolean flag = false;
            for (TestConditionDTO conditionValue : conditionValues) {
                String s = conditionValueMap.get(conditionValue.getConditionField());
                //无对应受众
                if (StringUtils.isBlank(s)) {
                    flag = false;
                    break;
                }
                //有对应受众
                if (StringUtils.isNotBlank(s)) {
                    List<String> conditionValueList = conditionValue.getConditionValueList();
                    //受众值包含
                    if (conditionValueList.contains(s)) {
                        flag = true;
                    } else {
                        flag = false;
                        break;
                    }
                }
            }
            if (flag) {
                plans.add(slotPlan);
            }

        }
        return plans;
    }


    /**
     * @return java.lang.Integer
     * @Description 获取当前实验中, 域实验占用的百分比
     * @Date 4:46 下午 2021/4/6
     * @Param [testPlans]
     **/
    private Integer getDomainPercent(List<TestSlotPlanDTO> testPlans) {
        //
        if (CollectionUtils.isEmpty(testPlans)) {
            return 0;
        }
        Integer domainPercent = 0;
        for (TestSlotPlanDTO testPlan : testPlans) {
            if (testPlan.getTestPlan().getDomainType() == 1) {
                domainPercent += testPlan.getPercent();
            }
        }
        return domainPercent;
    }

    /**
     * @return java.util.List<com.duiba.tuia.abtest.api.dto.TestSlotPlanDTO>
     * @Author zhangliwei
     * @Description 不走域的情况下, 过滤掉所有域实验
     * @Date 10:52 上午 2021/3/22
     * @Param [testPlans]
     **/
    private List<TestSlotPlanDTO> filterNotDomainPlans(List<TestSlotPlanDTO> testPlans) {
        testPlans = testPlans.stream().filter(o -> o.getTestPlan().getDomainType() == 0).collect(Collectors.toList());
        return testPlans;
    }

    /**
     * @return java.lang.Boolean
     * @Description 广告实验域, 不加盐分流, 根据用户维度判断是否分流到域实验
     * @Date 8:43 下午 2021/3/10
     * @Param [request]
     **/
    private Boolean isDomainHash(List<TestSlotPlanDTO> testPlans, ABRequestDto request) {
        //域分流
        //增加PV分流支持
        int domainHash = 0;
        domainHash = hash(request.getDeviceId()) % 100;

        TestSlotPlanDTO current = null;
        for (TestSlotPlanDTO slotPlan : testPlans) {
            //分流到域,如果没分流到域,就把域相关的实验都过滤掉
            if (slotPlan.getPercentDistrictSet().contains(domainHash) && slotPlan.getTestPlan().getDomainType() == 1) {
                current = slotPlan;
                break;
            }
        }
        //没有分流到域的情况,过滤掉域实验
        if (current == null) {
            return false;
        }
        return true;
    }


    /**
     * @return com.duiba.tuia.abtest.api.dto.TestSlotDTO
     * @Description 出异常时返回空对象
     * @Date 2:03 下午 2021/4/23
     * @Param [slotId]
     **/
    private TestSlotDTO NullObject(Long slotId) {
        TestSlotDTO dto = new TestSlotDTO();
        dto.setSlotId(slotId);
        dto.setId(-1L);
        dto.setTestPlans(new ArrayList<>());
        return dto;
    }


    private TestSlotDTO NullObject(String layerCode) {
        TestSlotDTO dto = new TestSlotDTO();
        dto.setSlotId(-1l);
        dto.setTestLayerCode(layerCode);
        dto.setId(-1L);
        dto.setTestPlans(new ArrayList<>());
        return dto;
    }


}
