package com.github.gquintana.metrics.sql;

/*
 * #%L
 * Metrics SQL
 * %%
 * Copyright (C) 2014 Open-Source
 * %%
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *      http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 * #L%
 */

import com.github.gquintana.metrics.proxy.ProxyFactory;
import com.github.gquintana.metrics.proxy.ReflectProxyFactory;

import javax.sql.DataSource;
import java.sql.Connection;

/**
 * Metrics SQL initiazing class
 */
public class MetricsSql {
    /**
     * Builder of {@link JdbcProxyFactory}
     */
    public static class Builder {
        private final MetricNamingStrategy namingStrategy;
        private ProxyFactory proxyFactory = new ReflectProxyFactory();
        private JdbcProxyFactory jdbcProxyFactory;

        public Builder(MetricNamingStrategy namingStrategy) {
            this.namingStrategy = namingStrategy;
        }

        /**
         * Select factory of proxies
         */
        public Builder withProxyFactory(ProxyFactory proxyFactory) {
            this.proxyFactory = proxyFactory;
            return this;
        }

        public JdbcProxyFactory build() {
            if (jdbcProxyFactory == null) {
                jdbcProxyFactory = new JdbcProxyFactory(namingStrategy, proxyFactory);
            }
            return jdbcProxyFactory;
        }

        /**
         * Wrap an existing {@link DataSource} to add metrics
         *
         * @param dataSource   The data source
         * @param databaseName The name of the datavaser.
         * @return the new {@link DataSource}.
         */
        public DataSource wrap(String databaseName, DataSource dataSource) {
            return build().wrapDataSource(databaseName, dataSource);
        }

        /**
         * Wrap an existing {@link Connection} to add connection
         *
         * @param databaseName The name of the database
         * @param connection   The connection to wrap
         * @return the new {@link Connection}
         */
        public Connection wrap(String databaseName, Connection connection) {
            return build().wrapConnection(databaseName, connection);
        }
    }

    /**
     * Select Metric naming strategy and Metric registry
     *
     * @param namingStrategy The naming strategy
     * @return the builder for the naming strategy
     */
    public static Builder withMetricNamingStrategy(MetricNamingStrategy namingStrategy) {
        return new Builder(namingStrategy);
    }
}
