/*******************************************************************************
 * 	Copyright 2016 ContainX and OpenStack4j                                          
 * 	                                                                                 
 * 	Licensed under the Apache License, Version 2.0 (the "License"); you may not      
 * 	use this file except in compliance with the License. You may obtain a copy of    
 * 	the License at                                                                   
 * 	                                                                                 
 * 	    http://www.apache.org/licenses/LICENSE-2.0                                   
 * 	                                                                                 
 * 	Unless required by applicable law or agreed to in writing, software              
 * 	distributed under the License is distributed on an "AS IS" BASIS, WITHOUT        
 * 	WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the         
 * 	License for the specific language governing permissions and limitations under    
 * 	the License.                                                                     
 *******************************************************************************/
package com.huawei.openstack4j.openstack.heat.domain;

import com.fasterxml.jackson.annotation.JsonProperty;
import com.huawei.openstack4j.model.heat.StackUpdate;
import com.huawei.openstack4j.model.heat.builder.StackUpdateBuilder;
import com.huawei.openstack4j.openstack.heat.utils.Environment;
import com.huawei.openstack4j.openstack.heat.utils.Template;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * Model Entity used for updating a Stack
 *
 * @author Jeremy Unruh
 */
public class HeatStackUpdate implements StackUpdate {

    private static final long serialVersionUID = 1L;
    private static final Logger LOG = LoggerFactory.getLogger(HeatStackUpdate.class);

    @JsonProperty("template")
    private String template;
    @JsonProperty("template_url")
    private String templateURL;
    @JsonProperty("parameters")
    private Map<String, String> parameters;
    @JsonProperty("timeout_mins")
    private Long timeoutMins;
    @JsonProperty("environment")
    private String environment;
    @JsonProperty("files")
    private Map<String, String> files = new HashMap<String, String>();
    @JsonProperty("tags")
    private String tags;

    public static StackUpdateBuilder builder() {
        return new HeatStackUpdateConcreteBuilder();
    }

    @Override
    public Map<String, String> getParameters() {
        return parameters;
    }

    @Override
    public String getTemplate() {
        return template;
    }

    public String getTempateURL() {
        return templateURL;
    }

    public String getEnvironment(){
        return environment;
    }

    public Map<String, String> getFiles() {
        return files;
    }

    public String getTags() {
        return tags;
    }

    @Override
    public StackUpdateBuilder toBuilder() {
        return new HeatStackUpdateConcreteBuilder(this);
    }

    public static class HeatStackUpdateConcreteBuilder implements StackUpdateBuilder {

        private HeatStackUpdate model;

        public HeatStackUpdateConcreteBuilder() {
            this(new HeatStackUpdate());
        }

        public HeatStackUpdateConcreteBuilder(HeatStackUpdate model) {
            this.model = model;
        }

        @Override
        public StackUpdate build() {
            return model;
        }

        @Override
        public StackUpdateBuilder from(StackUpdate in) {
            model = (HeatStackUpdate) in;
            return this;
        }

        @Override
        public StackUpdateBuilder template(String template) {
            model.template = template;
            return this;
        }

        @Override
        public StackUpdateBuilder templateFromFile(String tplFile) {
            try {
                Template tpl = new Template(tplFile);
                model.template = tpl.getTplContent();
                model.files.putAll(tpl.getFiles());
            } catch (Exception e) {
                LOG.error(e.getMessage(), e);
            }
            return this;
        }

        @Override
        public StackUpdateBuilder templateURL(String templateURL) {
            model.templateURL = templateURL;
            return this;
        }

        @Override
        public StackUpdateBuilder parameters(Map<String, String> parameters) {
            model.parameters = parameters;
            return this;
        }

        @Override
        public StackUpdateBuilder timeoutMins(Long timeoutMins) {
            model.timeoutMins = timeoutMins;
            return this;
        }

        @Override
        public StackUpdateBuilder environment(String environment){
            model.environment = environment;
            return this;
        }

        @Override
        public StackUpdateBuilder environmentFromFile(String envFile){
            try {
                Environment env = new Environment(envFile);
                model.environment = env.getEnvContent();
                model.files.putAll(env.getFiles());
            } catch (Exception e) {
                LOG.error(e.getMessage(), e);
            }
            return this;
        }

        @Override
        public StackUpdateBuilder files(Map<String, String> files) {
            model.files = files;
            return this;
        }

        @Override
        public StackUpdateBuilder tags(String tags) {
            model.tags = tags;
            return this;
        }

    }
}
