/*
 * Decompiled with CFR 0.152.
 */
package com.netflix.spinnaker.cats.dynomite.cache;

import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.google.common.collect.Lists;
import com.google.common.hash.Hashing;
import com.netflix.dyno.connectionpool.exception.DynoException;
import com.netflix.dyno.jedis.DynoJedisPipeline;
import com.netflix.spinnaker.cats.cache.CacheData;
import com.netflix.spinnaker.cats.cache.DefaultCacheData;
import com.netflix.spinnaker.cats.compression.CompressionStrategy;
import com.netflix.spinnaker.cats.compression.NoopCompression;
import com.netflix.spinnaker.cats.dynomite.DynomiteUtils;
import com.netflix.spinnaker.cats.dynomite.ExcessiveDynoFailureRetries;
import com.netflix.spinnaker.cats.redis.cache.AbstractRedisCache;
import com.netflix.spinnaker.cats.redis.cache.RedisCacheOptions;
import com.netflix.spinnaker.kork.dynomite.DynomiteClientDelegate;
import com.netflix.spinnaker.kork.jedis.RedisClientDelegate;
import java.io.IOException;
import java.nio.charset.StandardCharsets;
import java.time.Duration;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashMap;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.ThreadLocalRandom;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.stream.Collectors;
import net.jodah.failsafe.Failsafe;
import net.jodah.failsafe.RetryPolicy;
import net.jodah.failsafe.SyncFailsafe;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import redis.clients.jedis.Response;
import redis.clients.jedis.exceptions.JedisException;

public class DynomiteCache
extends AbstractRedisCache {
    private final Logger log = LoggerFactory.getLogger(((Object)((Object)this)).getClass());
    private static final RetryPolicy REDIS_RETRY_POLICY = DynomiteUtils.greedyRetryPolicy(500L);
    private final CacheMetrics cacheMetrics;
    private final CompressionStrategy compressionStrategy;

    public DynomiteCache(String prefix, DynomiteClientDelegate dynomiteClientDelegate, ObjectMapper objectMapper, RedisCacheOptions options, CacheMetrics cacheMetrics, CompressionStrategy compressionStrategy) {
        super(prefix, (RedisClientDelegate)dynomiteClientDelegate, objectMapper, options);
        this.cacheMetrics = cacheMetrics == null ? new CacheMetrics.NOOP() : cacheMetrics;
        this.compressionStrategy = compressionStrategy == null ? new NoopCompression() : compressionStrategy;
    }

    public void mergeItems(String type, Collection<CacheData> items) {
        if (items.isEmpty()) {
            return;
        }
        AtomicInteger relationships = new AtomicInteger();
        AtomicInteger hmsetOperations = new AtomicInteger();
        AtomicInteger saddOperations = new AtomicInteger();
        AtomicInteger expireOperations = new AtomicInteger();
        AtomicInteger delOperations = new AtomicInteger();
        AtomicInteger skippedWrites = new AtomicInteger();
        AtomicInteger hashesUpdated = new AtomicInteger();
        Map<CacheData, Map<String, String>> allHashes = this.getAllHashes(type, items);
        ((SyncFailsafe)Failsafe.with((RetryPolicy)REDIS_RETRY_POLICY).onRetriesExceeded(failure -> {
            this.log.error("Encountered repeated failures while caching {}:{}, attempting cleanup", new Object[]{this.prefix, type, failure});
            try {
                this.redisClientDelegate.withPipeline(pipeline -> {
                    DynoJedisPipeline p = (DynoJedisPipeline)pipeline;
                    for (CacheData item : items) {
                        p.del(this.itemHashesId(type, item.getId()));
                        delOperations.incrementAndGet();
                    }
                    p.sync();
                });
            }
            catch (DynoException | JedisException e) {
                this.log.error("Failed cleaning up hashes in failure handler in {}:{}", new Object[]{this.prefix, type, e});
            }
            throw new ExcessiveDynoFailureRetries(String.format("Running cache agent %s:%s", this.prefix, type), (Throwable)failure);
        })).run(() -> this.redisClientDelegate.withPipeline(pipeline -> {
            DynoJedisPipeline p = (DynoJedisPipeline)pipeline;
            boolean pipelineHasOps = false;
            for (CacheData item : items) {
                MergeOp op = this.buildHashedMergeOp(type, item, (Map)allHashes.get(item));
                skippedWrites.addAndGet(op.skippedWrites);
                if (op.valuesToSet.isEmpty()) continue;
                pipelineHasOps = true;
                p.hmset(this.itemId(type, item.getId()), op.valuesToSet);
                hmsetOperations.incrementAndGet();
                if (!op.relNames.isEmpty()) {
                    p.sadd(this.allRelationshipsId(type), op.relNames.toArray(new String[op.relNames.size()]));
                    saddOperations.incrementAndGet();
                    relationships.addAndGet(op.relNames.size());
                }
                if (item.getTtlSeconds() > 0) {
                    p.expire(this.itemId(type, item.getId()), item.getTtlSeconds());
                    expireOperations.incrementAndGet();
                }
                p.sadd(this.allOfTypeId(type), new String[]{item.getId()});
                saddOperations.incrementAndGet();
                if (op.hashesToSet.isEmpty()) continue;
                p.hmset(this.itemHashesId(type, item.getId()), op.hashesToSet);
                hmsetOperations.incrementAndGet();
                p.expire(this.itemHashesId(type, item.getId()), this.getHashExpiry());
                expireOperations.incrementAndGet();
                hashesUpdated.addAndGet(op.hashesToSet.size());
            }
            if (pipelineHasOps) {
                p.sync();
            }
        }));
        this.cacheMetrics.merge(this.prefix, type, items.size(), relationships.get(), skippedWrites.get(), hashesUpdated.get(), saddOperations.get(), hmsetOperations.get(), expireOperations.get(), delOperations.get());
    }

    protected void evictItems(String type, List<String> identifiers, Collection<String> allRelationships) {
        AtomicInteger delOperations = new AtomicInteger();
        AtomicInteger sremOperations = new AtomicInteger();
        ((SyncFailsafe)Failsafe.with((RetryPolicy)REDIS_RETRY_POLICY).onRetriesExceeded(failure -> {
            throw new ExcessiveDynoFailureRetries(String.format("Evicting items for %s:%s", this.prefix, type), (Throwable)failure);
        })).run(() -> this.redisClientDelegate.withPipeline(pipeline -> {
            DynoJedisPipeline p = (DynoJedisPipeline)pipeline;
            for (List idPartition : Lists.partition((List)identifiers, (int)this.options.getMaxDelSize())) {
                String[] ids = idPartition.toArray(new String[idPartition.size()]);
                pipeline.srem(this.allOfTypeId(type), ids);
                sremOperations.incrementAndGet();
            }
            for (String id : identifiers) {
                pipeline.del(this.itemId(type, id));
                delOperations.incrementAndGet();
                pipeline.del(this.itemHashesId(type, id));
                delOperations.incrementAndGet();
            }
            if (!identifiers.isEmpty()) {
                p.sync();
            }
        }));
        this.cacheMetrics.evict(this.prefix, type, identifiers.size(), delOperations.get(), sremOperations.get());
    }

    protected Collection<CacheData> getItems(String type, List<String> ids, List<String> knownRels) {
        if (ids.isEmpty()) {
            return new ArrayList<CacheData>();
        }
        AtomicInteger hmgetAllOperations = new AtomicInteger();
        Map rawItems = (Map)((SyncFailsafe)Failsafe.with((RetryPolicy)REDIS_RETRY_POLICY).onRetriesExceeded(failure -> {
            throw new ExcessiveDynoFailureRetries(String.format("Getting items for %s:%s", this.prefix, type), (Throwable)failure);
        })).get(() -> (Map)this.redisClientDelegate.withPipeline(pipeline -> {
            DynoJedisPipeline p = (DynoJedisPipeline)pipeline;
            HashMap<String, Response> responses = new HashMap<String, Response>();
            for (String id : ids) {
                responses.put(id, pipeline.hgetAll(this.itemId(type, id)));
                hmgetAllOperations.incrementAndGet();
            }
            p.sync();
            return responses.entrySet().stream().filter(e -> !((Map)((Response)e.getValue()).get()).isEmpty()).collect(Collectors.toMap(Map.Entry::getKey, it -> (Map)((Response)it.getValue()).get()));
        }));
        ArrayList<CacheData> results = new ArrayList<CacheData>(ids.size());
        for (Map.Entry rawItem : rawItems.entrySet()) {
            CacheData item = this.extractHashedItem(type, (String)rawItem.getKey(), (Map)rawItem.getValue(), knownRels);
            if (item == null) continue;
            results.add(item);
        }
        this.cacheMetrics.get(this.prefix, type, results.size(), ids.size(), knownRels.size(), hmgetAllOperations.get());
        return results;
    }

    private CacheData extractHashedItem(String type, String id, Map<String, String> values, List<String> knownRels) {
        if (values == null) {
            return null;
        }
        try {
            Map attributes = values.get("attributes") != null ? (Map)this.objectMapper.readValue(this.compressionStrategy.decompress(values.get("attributes")), ATTRIBUTES) : null;
            HashMap<String, Collection> relationships = new HashMap<String, Collection>();
            for (Map.Entry<String, String> value : values.entrySet()) {
                Collection deserializedRel;
                if (value.getKey().equals("attributes") || value.getKey().equals("id") || !knownRels.contains(value.getKey())) continue;
                try {
                    deserializedRel = (Collection)this.objectMapper.readValue(this.compressionStrategy.decompress(value.getValue()), this.getRelationshipsTypeReference());
                }
                catch (JsonProcessingException e) {
                    this.log.warn("Failed processing property '{}' on item '{}'", (Object)value.getKey(), (Object)this.itemId(type, id));
                    continue;
                }
                relationships.put(value.getKey(), deserializedRel);
            }
            return new DefaultCacheData(id, attributes, relationships);
        }
        catch (IOException deserializationException) {
            throw new RuntimeException("Deserialization failed", deserializationException);
        }
    }

    protected Set<String> scanMembers(String setKey, Optional<String> glob) {
        return (Set)Failsafe.with((RetryPolicy)REDIS_RETRY_POLICY).get(() -> super.scanMembers(setKey, glob));
    }

    private boolean hashCheck(Map<String, String> hashes, String id, String serializedValue, Map<String, String> updatedHashes, boolean hasTtl) {
        if (this.options.isHashingEnabled() && !hasTtl) {
            String existingHash;
            String hash = Hashing.sha1().newHasher().putString((CharSequence)serializedValue, StandardCharsets.UTF_8).hash().toString();
            if (hash.equals(existingHash = hashes.get(id))) {
                return true;
            }
            updatedHashes.put(id, hash);
        }
        return false;
    }

    private MergeOp buildHashedMergeOp(String type, CacheData cacheData, Map<String, String> hashes) {
        String serializedAttributes;
        int skippedWrites = 0;
        boolean hasTtl = cacheData.getTtlSeconds() > 0;
        try {
            serializedAttributes = cacheData.getAttributes().isEmpty() ? null : this.objectMapper.writeValueAsString((Object)cacheData.getAttributes());
        }
        catch (JsonProcessingException serializationException) {
            throw new RuntimeException("Attribute serialization failed", serializationException);
        }
        HashMap<String, String> hashesToSet = new HashMap<String, String>();
        HashMap<String, String> valuesToSet = new HashMap<String, String>();
        if (serializedAttributes != null && this.hashCheck(hashes, this.attributesId(type, cacheData.getId()), serializedAttributes, hashesToSet, hasTtl)) {
            ++skippedWrites;
        } else if (serializedAttributes != null) {
            valuesToSet.put("attributes", this.compressionStrategy.compress(serializedAttributes));
        }
        if (!cacheData.getRelationships().isEmpty()) {
            for (Map.Entry relationship : cacheData.getRelationships().entrySet()) {
                String relationshipValue;
                try {
                    relationshipValue = this.objectMapper.writeValueAsString(new LinkedHashSet((Collection)relationship.getValue()));
                }
                catch (JsonProcessingException serializationException) {
                    throw new RuntimeException("Relationship serialization failed", serializationException);
                }
                if (this.hashCheck(hashes, this.relationshipId(type, cacheData.getId(), (String)relationship.getKey()), relationshipValue, hashesToSet, hasTtl)) {
                    ++skippedWrites;
                    continue;
                }
                valuesToSet.put((String)relationship.getKey(), this.compressionStrategy.compress(relationshipValue));
            }
        }
        return new MergeOp(cacheData.getRelationships().keySet(), valuesToSet, hashesToSet, skippedWrites);
    }

    private Map<CacheData, Map<String, String>> getAllHashes(String type, Collection<CacheData> items) {
        if (this.isHashingDisabled(type)) {
            return new HashMap<CacheData, Map<String, String>>();
        }
        return (Map)((SyncFailsafe)Failsafe.with((RetryPolicy)REDIS_RETRY_POLICY).onRetriesExceeded(failure -> {
            throw new ExcessiveDynoFailureRetries(String.format("Getting all requested hashes for %s:%s", this.prefix, type), (Throwable)failure);
        })).get(() -> (Map)this.redisClientDelegate.withPipeline(pipeline -> {
            DynoJedisPipeline p = (DynoJedisPipeline)pipeline;
            HashMap<CacheData, Response> responses = new HashMap<CacheData, Response>();
            for (CacheData item : items) {
                responses.put(item, p.hgetAll(this.itemHashesId(type, item.getId())));
            }
            p.sync();
            return responses.entrySet().stream().collect(Collectors.toMap(Map.Entry::getKey, it -> (Map)((Response)it.getValue()).get()));
        }));
    }

    protected boolean isHashingDisabled(String type) {
        return (Boolean)((SyncFailsafe)Failsafe.with((RetryPolicy)REDIS_RETRY_POLICY).onRetriesExceeded(failure -> {
            throw new ExcessiveDynoFailureRetries(String.format("Getting hashing flag for %s:%s", this.prefix, type), (Throwable)failure);
        })).get(() -> super.isHashingDisabled(type));
    }

    private int getHashExpiry() {
        return (int)Duration.ofMinutes(ThreadLocalRandom.current().nextInt(60, 240)).getSeconds();
    }

    private String itemId(String type, String id) {
        return String.format("{%s:%s}:%s", this.prefix, type, id);
    }

    private String itemHashesId(String type, String id) {
        return String.format("{%s:%s}:hashes:%s", this.prefix, type, id);
    }

    protected String attributesId(String type, String id) {
        return String.format("{%s:%s}:attributes:%s", this.prefix, type, id);
    }

    protected String relationshipId(String type, String id, String relationship) {
        return String.format("{%s:%s}:relationships:%s:%s", this.prefix, type, id, relationship);
    }

    protected String allRelationshipsId(String type) {
        return String.format("{%s:%s}:relationships", this.prefix, type);
    }

    protected String allOfTypeId(String type) {
        return String.format("{%s:%s}:members", this.prefix, type);
    }

    private static class MergeOp {
        final Set<String> relNames;
        final Map<String, String> valuesToSet;
        final Map<String, String> hashesToSet;
        final int skippedWrites;

        public MergeOp(Set<String> relNames, Map<String, String> valuesToSet, Map<String, String> hashesToSet, int skippedWrites) {
            this.relNames = relNames;
            this.valuesToSet = valuesToSet;
            this.hashesToSet = hashesToSet;
            this.skippedWrites = skippedWrites;
        }
    }

    public static interface CacheMetrics {
        default public void merge(String prefix, String type, int itemCount, int relationshipCount, int hashMatches, int hashUpdates, int saddOperations, int hmsetOperations, int expireOperations, int delOperations) {
        }

        default public void evict(String prefix, String type, int itemCount, int delOperations, int sremOperations) {
        }

        default public void get(String prefix, String type, int itemCount, int requestedSize, int relationshipsRequested, int hmgetAllOperations) {
        }

        public static class NOOP
        implements CacheMetrics {
        }
    }
}

