/**
 * Project Name:qiho-center-api
 * File Name:RemoteOrderService.java
 * Package Name:com.qiho.center.api.remoteservice
 * Date:2017年6月2日下午7:03:02
 * Copyright (c) 2017, duiba.com.cn All Rights Reserved.
 */

package com.qiho.center.api.remoteservice.order;

import java.util.List;

import com.qiho.center.api.dto.ChannelInfoDto;
import com.qiho.center.api.dto.ItemDetailDto;
import com.qiho.center.api.dto.OrderDto;
import com.qiho.center.api.dto.OrderSnapshotDto;
import com.qiho.center.api.dto.PagenationDto;
import com.qiho.center.api.params.OrderAuditParams;
import com.qiho.center.api.params.OrderConsumerParam;
import com.qiho.center.api.params.OrderGainParams;
import com.qiho.center.api.params.OrderItemParam;
import com.qiho.center.api.params.OrderPageParams;
import com.qiho.center.api.params.YunPianPushParams;

import cn.com.duiba.boot.exception.BizException;
import cn.com.duiba.boot.netflix.feign.AdvancedFeignClient;
import cn.com.duiba.wolf.dubbo.DubboResult;

/**
 * ClassName:RemoteOrderService <br/>
 * 订单服务. <br/>
 * Date:     2017年6月2日 下午7:03:02 <br/>
 *
 * @author zhanglihui
 * @see
 * @since JDK 1.6
 */
@AdvancedFeignClient
public interface RemoteOrderService {

    /**
     * 提交订单,同步返回订单ID，并同时异步更新订单快照信息，扫描订单防作弊规则
     *
     * @param order       订单内容
     * @param channelInfo 订单渠道信息
     * @param payType     付款方式
     *
     * @return 订单ID
     *
     * @author zhanglihui
     */
    DubboResult<String> submit(OrderDto order, ChannelInfoDto channelInfo, String payType);

    /**
     * 发送短信验证码
     *
     * @param mobile  发送的手机号码
     * @param smsCode 发送的短信验证码
     *
     * @return DubboResult
     *
     * @author zhanglihui
     */
    DubboResult<Boolean> sendSmsCode(String mobile, String smsCode);

    /**
     * 查询全部订单的分页
     *
     * @param params
     *
     * @return
     */
    DubboResult<PagenationDto<OrderSnapshotDto>> findAllPage(OrderPageParams params);

    /**
     * 获取订单分页总条数
     *
     * @param params
     *
     * @return
     */
    DubboResult<Integer> findAllPageCount(OrderPageParams params);

    /**
     * 导出订单的分页查询
     *
     * @param params
     *
     * @return
     */
    DubboResult<List<OrderSnapshotDto>> exportAllOrderPage(OrderPageParams params);

    /**
     * 查询导出订单的总数(覆盖索引),值会远远大于真实值
     *
     * @param params
     *
     * @return
     */
    DubboResult<Integer> exportOrderPageCount(OrderPageParams params);

    /**
     * 批量审核订单
     *
     * @param progressKey 审核进度查询Key,用于向缓存中写进度,调用batchAuditOrderInit获取
     * @param paramList
     *
     * @return 提交的审核条数
     */
    DubboResult<Integer> batchAuditOrder(String progressKey, List<OrderAuditParams> paramList);

    /**
     * 审核订单
     *
     * @param params
     *
     * @return
     */
    DubboResult<Boolean> auditOrder(OrderAuditParams params);

    /**
     * 根据订单ID查询订单信息
     *
     * @param orderId
     *
     * @return OrderDto
     *
     * @author zhanglihui
     */
    DubboResult<OrderDto> findByOrderId(String orderId);

    /**
     * 查询订单快照
     *
     * @param orderId
     *
     * @return OrderSnapshotDto
     *
     * @author zhanglihui
     */
    DubboResult<OrderSnapshotDto> findOrderSnapshot(String orderId);

    /**
     * 更新订单商品详情
     *
     * @param param
     *
     * @return DubboResult
     *
     * @author zhanglihui
     */
    DubboResult<Void> updateOrderItem(OrderItemParam param);

    /**
     * 更新订单买家信息
     *
     * @param param
     *
     * @return DubboResult
     *
     * @author zhanglihui
     */
    DubboResult<Void> updateOrderConsumer(OrderConsumerParam param);

    /**
     * 更新订单状态
     *
     * @param orderId
     * @param orderStatus
     *
     * @return DubboResult
     */
    DubboResult<Void> updateOrderStatus(String orderId, String orderStatus);

    /**
     * 根据手机号查询订单列表
     *
     * @param mobile     手机号
     * @param statusList 订单状态
     *
     * @return List<OrderSnapshotDto>
     *
     * @author zhanglihui
     */
    DubboResult<List<OrderSnapshotDto>> findByMobile(String mobile, List<String> statusList);

    /**
     * 批量处理订单状态流转及处理方式变更
     */
    DubboResult<Void> updateOrderWithProcessWay(List<OrderGainParams> orderGainParams);

    /**
     * 根据手机号查询最新的一单订单
     */
    DubboResult<Boolean> notePushProcessByYunPain(YunPianPushParams params);

    /**
     * 通过订单id,批量查询订单
     */
    List<OrderSnapshotDto> queryOrderByOrderIds(List<String> orderIds);

    /**
     * 查询指定数量最新订单
     *
     * @param pageParams
     *
     * @return
     */
    List<OrderSnapshotDto> queryOrderAsNewMost(OrderPageParams pageParams);

    ItemDetailDto getOrderItemSnapshot(String orderId, Long itemId);

    /**
     * 获取用户最新一笔&状态成功的订单
     *
     * @param mobile
     *
     * @return
     */
    OrderDto getLastOrderByMoblie(String mobile);

    /**
     * 根据手机号和商品ID判断是否存在订单
     *
     * @param mobile 手机号
     * @param itemId 商品ID
     *
     * @return boolean
     */
    Boolean isExistOrderByMobileAndItem(String mobile, Long itemId);

    /**
     * 批量审核订单
     *
     * @param paramList
     *
     * @return
     */
    Boolean batchAudit(List<OrderAuditParams> paramList) throws BizException;

}

