/**
 * Project Name:qiho-center-api
 * File Name:RemoteOrderService.java
 * Package Name:com.qiho.center.api.remoteservice
 * Date:2017年6月2日下午7:03:02
 * Copyright (c) 2017, duiba.com.cn All Rights Reserved.
 */

package com.qiho.center.api.remoteservice.order;

import cn.com.duiba.boot.exception.BizException;
import cn.com.duiba.boot.netflix.feign.AdvancedFeignClient;
import cn.com.duiba.wolf.dubbo.DubboResult;
import com.qiho.center.api.dto.*;
import com.qiho.center.api.dto.es.OrderSnapshotESDto;
import com.qiho.center.api.dto.order.*;
import com.qiho.center.api.dto.task.BatchOrderTaskDto;
import com.qiho.center.api.params.*;

import java.util.List;
import java.util.Map;

/**
 * ClassName:RemoteOrderService <br/>
 * 订单服务. <br/>
 * Date:     2017年6月2日 下午7:03:02 <br/>
 *
 * @author zhanglihui
 * @see
 * @since JDK 1.6
 */
@AdvancedFeignClient
public interface RemoteOrderService {

    /**
     * 提交订单,同步返回订单ID，并同时异步更新订单快照信息，扫描订单防作弊规则
     *
     * @param order       订单内容
     * @param channelInfo 订单渠道信息
     * @param payType     付款方式
     *
     * update note：订单流程改造后，此接口不再使用
     *
     * @return 订单ID
     *
     * @author zhanglihui
     */
    @Deprecated
    DubboResult<String> submit(OrderDto order, ChannelInfoDto channelInfo, String payType);

    /**
     * 发送短信验证码
     *
     * @param mobile  发送的手机号码
     * @param smsCode 发送的短信验证码
     *
     * @return DubboResult
     *
     * @author zhanglihui
     */
    DubboResult<Boolean> sendSmsCode(String mobile, String smsCode);

    /**
     * 查询全部订单的分页
     *
     * @param params
     *
     * @return
     */
    DubboResult<PagenationDto<OrderSnapshotDto>> findAllPage(OrderPageParams params);

    /**
     * 获取订单分页总条数
     *
     * @param params
     *
     * @return
     */
    DubboResult<Integer> findAllPageCount(OrderPageParams params);

    /**
     * 导出订单的分页查询
     *
     * @param params
     *
     * @return
     */
    OrderSnapshotESDto exportAllOrderPage(OrderPageParams params);

    /**
     * 查询导出订单的总数(覆盖索引),值会远远大于真实值
     *
     * @param params
     *
     * @return
     */
    DubboResult<Integer> exportOrderPageCount(OrderPageParams params);

    /**
     * 批量审核订单(和批量取消订单保持同样的实现方式，保存文件校验的详情，用于用户下载)
     *
     * @param batchOrderDto
     *
     * @return 提交的审核条数
     */
    DubboResult<Integer> batchAuditOrder(BatchOrderDto batchOrderDto);

    /**
     * 审核订单
     *
     * @param params
     *
     * @return
     */
    DubboResult<Boolean> auditOrder(OrderAuditParams params);

    /**
     * 根据订单ID查询订单信息
     *
     * @param orderId
     *
     * @return OrderDto
     *
     * @author zhanglihui
     */
    DubboResult<OrderDto> findByOrderId(String orderId);

    /**
     * 查询订单快照
     *
     * @param orderId
     *
     * @return OrderSnapshotDto
     *
     * @author zhanglihui
     */
    DubboResult<OrderSnapshotDto> findOrderSnapshot(String orderId);

    /**
     * 更新订单商品详情
     *
     * @param param
     *
     * @return DubboResult
     *
     * @author zhanglihui
     */
    DubboResult<Void> updateOrderItem(OrderItemParam param);

    /**
     * 更新订单商品信息_不校验限购数量
     * @author zhanglihui
     * @param param
     */
    DubboResult<Void> updateOrderItemInfo(OrderItemParam param);

    /**
     * 更新订单买家信息
     *
     * @param param
     *
     * @return DubboResult
     *
     * @author zhanglihui
     */
    DubboResult<Void> updateOrderConsumer(OrderConsumerParam param);

    /**
     * 更新订单买家信息_不校验地址
     *
     * @param param
     *
     * @return DubboResult
     *
     * @author jiafeng.geng
     */
    DubboResult<Void> updateOrderConsumerInfo(OrderConsumerParam param);

    /**
     * 更新订单状态
     *
     * @param orderId
     * @param orderStatus
     *
     * @return DubboResult
     */
    DubboResult<Void> updateOrderStatus(String orderId, String orderStatus);

    /**
     * 根据orderId更新订单信息
     *
     * @param orderDto
     * @return DubboResult
     */
    DubboResult<Void> updateByOrderId(OrderDto orderDto);

    /**
     * 根据手机号查询订单列表
     *
     * @param mobile     手机号
     * @param statusList 订单状态
     *
     * @return List<OrderSnapshotDto>
     *
     * @author zhanglihui
     */
    DubboResult<List<OrderSnapshotDto>> findByMobile(String mobile, List<String> statusList);

    /**
     * 批量处理订单状态流转及处理方式变更
     */
    DubboResult<Void> updateOrderWithProcessWay(List<OrderGainParams> orderGainParams);

    /**
     * 根据手机号查询最新的一单订单
     */
    DubboResult<Boolean> notePushProcessByYunPain(YunPianPushParams params);

    /**
     * 通过订单id,批量查询订单
     */
    List<OrderSnapshotDto> queryOrderByOrderIds(List<String> orderIds);

    /**
     * 查询指定数量最新订单
     *
     * @param pageParams
     *
     * @return
     */
    List<OrderSnapshotDto> queryOrderAsNewMost(OrderPageParams pageParams);

    ItemDetailDto getOrderItemSnapshot(String orderId, Long itemId);

    /**
     * 根据手机号获取用户最新一笔&状态成功的订单
     *
     * @param mobile
     *
     * @return
     */
    OrderDto getLastOrderByMoblie(String mobile);


    /**
     * 根据acid获取用户最新的一笔订单
     * @param acid
     * @return
     */
    OrderDto getLastOrderByAcid(String acid);


    /**
     * 根据手机号和商品ID判断是否存在订单
     *
     * @param mobile 手机号
     * @param itemId 商品ID
     *
     * @return boolean
     */
    Boolean isExistOrderByMobileAndItem(String mobile, Long itemId);

    /**
     * 批量审核订单
     *
     * @param paramList
     *
     * @return
     */
    Boolean batchAudit(List<OrderAuditParams> paramList) throws BizException;

    /**
     * 根据手机号查询最新的一单订单
     */
    DubboResult<Boolean> notePushProcessByMeiLian(MeiLianPushParams params);


    /**
     * 单个确认发货
     * @param orderConfirmDto
     * @return
     */
    Boolean orderConfirm(OrderConfirmDto orderConfirmDto) throws BizException;


    /**
     * 单个确认发货
     * @param orderConfirmDto
     * @return
     */
    Boolean orderUpdate(OrderConfirmDto orderConfirmDto) throws BizException;

    /**
     * 单个取消发货
     * @param orderCancelDto
     */
    Boolean orderCancel(OrderCancelDto orderCancelDto) throws BizException;

    /**
     * 批量处理
     * 主订单发货
     * @param batchOrderConfirmDto
     * @return
     */
    BatchOrderTaskDto batchOrderConfirm(BatchOrderConfirmDto batchOrderConfirmDto);

    /**
     * 批量更新
     * @param batchOrderConfirmDto
     * @return
     */
    BatchOrderTaskDto batchOrderUpdate(BatchOrderConfirmDto batchOrderConfirmDto);

    /**
     * 批量处理
     * 主订单取消发货
     * @param batchOrderCancelDto
     * @return
     */
    BatchOrderTaskDto batchOrderCancel(BatchOrderCancelDto batchOrderCancelDto);

    /**
     * 对外商家批量确认发货
     * @param confirmDto
     * @return
     */
    BatchOrderTaskDto openBatchOrderConfirm(OpenBatchOrderConfirmDto confirmDto);

    /**
     * 对外商家批量取消发货
     * @param cancelDto
     * @return
     */
    BatchOrderTaskDto openBatchOrderCancel(OpenBatchOrderCancelDto cancelDto);
    /**
     * 创蓝根据手机号查询最新的一单订单
     */
    DubboResult<Boolean> notePushProcessByChuangLan(Map<String,String> params);

    /**
     * 用户评价通过 orderId和mobile校验订单
     * @param orderId
     * @param mobile
     * @return
     */
    OrderSnapshotDto findOrderByIdAndMobile(String orderId,String mobile);

    /**
     * ES上的订单数量
     * @param params
     * @return
     */
    Integer exportOrderESCount(OrderPageParams params);


    Boolean sendUpdateAndForgetPWDCodeByCL(String phoneNum, String originPwd, String vcode);

    /**
     * 手动订单状态回传
     *
     * @param batchOrderProcessDto
     * @param
     */
    void batchProcessOrder(BatchOrderProcessDto batchOrderProcessDto);
    /**
     * 根据CID分页查询订单信息
     */
    PagenationDto<OrderSnapshotDto> findOrderPageByCid(OrderPageParams params);

    /**
     * 记录订单相关日志
     *
     * @param orderSubTypeParam
     */
    void  recordLog(OrderSubTypeParam orderSubTypeParam);

    /**
     * 获取订单拓展信息
     */
    List<BatchOrderExtDto> getOrderExt(List<BatchOrderExtDto> orderExtDtos);

    /**
     * 保存订单拓展信息
     */
    void batchSaveOrderExt(List<BatchOrderExtDto> orderExtDtos);

    /**
     * 更新订单操作人相关信息
     */
    void updateOrderOperateInfo(List<BatchOrderExtDto> orderExtDtos);

    void updateOperateStatus(String orderId, Long operateOwnerId);

    void updateCancelOperateStatus(String cancelOperator, String orderId, Integer cancelOperatorType);

//    void cancelOrder(String orderId, String remark);
}

