package com.qiho.center.biz.bo;

import java.util.List;
import java.util.concurrent.ExecutorService;

import javax.annotation.Resource;

import org.apache.commons.lang.StringUtils;
import org.springframework.beans.BeansException;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.ApplicationContext;
import org.springframework.context.ApplicationContextAware;
import org.springframework.stereotype.Service;

import com.google.common.collect.Lists;
import com.qiho.center.api.dto.OrderDto;
import com.qiho.center.api.dto.blackList.BlackListDto;
import com.qiho.center.api.enums.QueryBlackListTypeEnum;
import com.qiho.center.api.params.BlackListOperationParams;
import com.qiho.center.api.params.ExpressBlackListOperationParams;
import com.qiho.center.biz.runnable.BlackListRunnable;
import com.qiho.center.biz.runnable.ExpressBlackListRunnable;
import com.qiho.center.biz.service.BlackListService;
import com.qiho.center.biz.service.order.AnticheateService;
import com.qiho.center.biz.service.order.OrderService;
import com.qiho.center.common.entity.order.QihoAnticheateEntity;

/**
 * ClassName: BlackListBo <br/>
 * Function: 黑名单处理. <br/>
 * date: 2017年6月26日 下午8:05:35 <br/>
 *
 * @author chencheng
 * @version
 */
@Service
public class BlackListBo implements ApplicationContextAware {

    private ApplicationContext context;

    @Resource
    private ExecutorService    executorService;

    @Autowired
    private BlackListService   blackListService;

    @Autowired
    private OrderService       orderService;

    @Resource
    private AnticheateService  anticheateService;

    /**
     * 批量黑名单处理
     *
     * @author chencheng
     * @param progressKey
     * @param paramList
     * @return
     */
    public Integer batchBlackList(String progressKey, List<BlackListOperationParams> paramList) {
        BlackListRunnable runnable = context.getBean(BlackListRunnable.class);
        runnable.setProgressKey(progressKey);
        runnable.setParams(paramList);
        int enableCount = runnable.enableFilter();
        executorService.submit(runnable);
        return enableCount;
    }

    /**
     * 批量物流黑名单处理
     * @param paramsList
     * @throws BeansException
     */
    public Integer batchExpressBlackList(String progressKey, List<ExpressBlackListOperationParams> paramsList){
        ExpressBlackListRunnable runnable = context.getBean(ExpressBlackListRunnable.class);
        runnable.setProgressKey(progressKey);
        runnable.setParams(paramsList);
        int enableCount = runnable.enableFilter();
        //测试方便改成同步执行，真实环境放在线程池处理
//        runnable.run();
        executorService.submit(runnable);
        return enableCount;
    }

    @Override
    public void setApplicationContext(ApplicationContext applicationContext) throws BeansException {
        this.context = applicationContext;
    }

    /**
     * 防作弊黑名单校验
     *
     * @author chencheng
     * @param orderId
     */
    public void cheatProof(String orderId) {
        OrderDto orderDto = orderService.findByOrderId(orderId);
        if (orderDto == null) {
            return;
        }
        List<QihoAnticheateEntity> list = Lists.newArrayList();
        if (StringUtils.isNotBlank(orderDto.getMobile())) {
            // 手机号手动添加黑名单
            BlackListDto manualPhoneDto = blackListService.findOneBlacklist(QueryBlackListTypeEnum.MANUAL_PHONE_NUMBER,
                                                                            orderDto.getMobile());
            if (manualPhoneDto != null) {
                list.add(new QihoAnticheateEntity(orderId, QueryBlackListTypeEnum.MANUAL_PHONE_NUMBER.getDesc()));
            }
            // 手机号策略黑名单
            BlackListDto blackListDto = blackListService.findLatestDateByType(QueryBlackListTypeEnum.STRATEGY_PHONE_NUMBER);
            BlackListDto strategyPhoneDto = null;
            if (null != blackListDto){
                strategyPhoneDto = blackListService.findOneBlacklist(QueryBlackListTypeEnum.STRATEGY_PHONE_NUMBER,
                        orderDto.getMobile(),blackListDto.getCurDate());
            }
            if (strategyPhoneDto != null) {
                list.add(new QihoAnticheateEntity(orderId, QueryBlackListTypeEnum.STRATEGY_PHONE_NUMBER.getDesc()));
            }
        }

        if (StringUtils.isNotBlank(orderDto.getAreaNum())) {
            // 地区手动添加黑名单
            BlackListDto manualRegionDto = blackListService.findOneBlacklist(QueryBlackListTypeEnum.MANUAL_REGION,
                                                                             orderDto.getAreaNum());
            if (manualRegionDto != null) {
                list.add(new QihoAnticheateEntity(orderId, QueryBlackListTypeEnum.MANUAL_REGION.getDesc()));
            }
            // 地区策略黑名单
            BlackListDto blackListDto = blackListService.findLatestDateByType(QueryBlackListTypeEnum.STRATEGY_REGION);
            BlackListDto strategyRegionDto = null;
            if (null != blackListDto){
                strategyRegionDto = blackListService.findOneBlacklist(QueryBlackListTypeEnum.STRATEGY_REGION,
                        orderDto.getMobile(),blackListDto.getCurDate());
            }
            if (strategyRegionDto != null) {
                list.add(new QihoAnticheateEntity(orderId, QueryBlackListTypeEnum.STRATEGY_REGION.getDesc()));
            }
        }

        if (list.size() > 0) {
            for (QihoAnticheateEntity qihoAnticheateEntity2 : list) {
                anticheateService.insert(qihoAnticheateEntity2);
            }
        }
    }

}
