package com.qiho.center.biz.paychannel.pay;

import cn.com.duiba.wolf.dubbo.DubboResult;
import com.google.common.collect.Maps;
import com.qiho.center.api.constant.NowPayConstant;
import com.qiho.center.api.dto.FundOrderDto;
import com.qiho.center.api.dto.OrderDto;
import com.qiho.center.api.dto.PayDto;
import com.qiho.center.api.enums.FundStatusEnum;
import com.qiho.center.api.enums.PayTypeEnum;
import com.qiho.center.api.exception.QihoException;
import com.qiho.center.api.util.FormDataConvertUtil;
import com.qiho.center.api.util.MD5Facade;
import com.qiho.center.api.util.NowPayAppUtil;
import com.qiho.center.api.util.SignCheckUtil;
import com.qiho.center.biz.model.RefundResult;
import com.qiho.center.biz.service.order.FundOrderService;
import com.qiho.center.biz.service.order.OrderService;
import com.qiho.center.common.dao.QihoFundOrderDAO;
import com.qiho.center.common.util.AppLogUtil;
import com.qiho.center.common.util.HttpClientUtil;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.InitializingBean;
import org.springframework.stereotype.Component;

import javax.annotation.Resource;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * Created by danke on 2017/7/24.
 */
@Component
public class NowPayProcessor extends PayChannelProcessor implements InitializingBean {

    Logger logger = LoggerFactory.getLogger(getClass());

    //private static final String APP_ID = "149973835725631";
    //private static final String APP_KEY = "cBBfruBz2pT8RpuUFAj9PHICfs2itZpz";//密钥

    private static final String FRONT_NOTIFY_URL = "http://dev-www.qiho99.com/order/success?orderId=";//前段通知url


    @Resource
    private OrderService orderService;

    @Resource
    private QihoFundOrderDAO qihoFundOrderDAO;

    @Resource
    private FundOrderService fundOrderService;

    @Override
    public String payExecute(String orderId) {
        //组装参数
        try{
            Map<String,String> dataMap = packageParamToMap(orderId);
            String parmaStr = FormDataConvertUtil.postFormLinkReportWithURLEncode(dataMap);
            String httpResultStr = HttpClientUtil.postData(NowPayAppUtil.getNowpayUrl(),parmaStr);
            Map<String,String> resultMap = FormDataConvertUtil.parseFormDataPatternReportWithDecode(httpResultStr);
            if (null == resultMap)
                throw new QihoException("请求支付返回结果为空");
            //验证签名
            boolean isValidSignature = SignCheckUtil.signCheck(resultMap,Boolean.FALSE, NowPayAppUtil.getAppKey());
            //同步返回之后验证签名以及返回状态码
            if (isValidSignature){
                if(resultMap.get(NowPayConstant.NowPayKey.responseCode).equalsIgnoreCase(NowPayConstant.OrderResponseStatus.SUCCESS)){
                    return resultMap.get(NowPayConstant.NowPayKey.TN);
                }else{
                    AppLogUtil.warn(logger,"申请付款失败 msg={}",resultMap.get(NowPayConstant.NowPayKey.responseMsg));
                }
            }else{
                AppLogUtil.warn(logger,"验签失败 orderId={}",orderId);
            }
        }catch (Exception e){
            AppLogUtil.error(logger,"现在支付异常 msg={}",e.getMessage());
        }
        return StringUtils.EMPTY;
    }

    /**
     * HTTP请求发起退款
     * @param orderId   主订单ID
     * @param fundId    资金流水ID，标识一次退款请求，同一笔交易多次退款需要保证唯一，如需部分退款，则此参数必传
     * @param refundAmt 退款金额
     */
    @Override
    public void refund(String orderId, String fundId, Integer refundAmt) {
        try {
             Map<String,String> dataMap = buildRefundParam(orderId,fundId,refundAmt);
            String parmaStr = FormDataConvertUtil.postFormLinkReportWithURLEncode(dataMap);
            String httpResultStr = HttpClientUtil.postData(NowPayAppUtil.getRefundUrl(),parmaStr);
            Map<String,String> resultMap = FormDataConvertUtil.parseFormDataPatternReportWithDecode(httpResultStr);
            if (null == resultMap)
                throw new QihoException("请求退款返回结果为空");
            //验证签名
            boolean isValidSignature = SignCheckUtil.signCheck(resultMap,Boolean.TRUE,NowPayAppUtil.getAppKey());
            FundOrderDto dto = fundOrderService.findByFundId(fundId);
            if (isValidSignature){
                String transStatus = resultMap.get(NowPayConstant.NowPayKey.tradeStatus);
                String responseCode = resultMap.get(NowPayConstant.NowPayKey.responseCode);
                dto.setOutSeqNo(resultMap.get(NowPayConstant.NowPayKey.nowPayOrderNo));
                if(NowPayConstant.refundStatus.SUCCESS.equals(transStatus) && NowPayConstant.refundRespCode.SUCCESS.equals(responseCode)){//同步退款成功
                    //处理订单状态,记录流水号
                    dto.setFundStatus(FundStatusEnum.SUCCESS.getCode());
                }else if(NowPayConstant.refundStatus.FAIL.equals(transStatus)){
                    dto.setFundStatus(FundStatusEnum.FAILED.getCode());
                    //推送失败则打印失败日志推送失败则打印失败日志
                    AppLogUtil.warn(logger,"发送退款请求失败 orderId={} fundId={} reefundAmt={} responseMsg={}",orderId,fundId,refundAmt,
                            resultMap.get(NowPayConstant.NowPayKey.responseMsg));
                }
                fundOrderService.update(dto);
            }else{
                AppLogUtil.warn(logger,"验签失败 orderId={} fundId={} reefundAmt={}",orderId,fundId,refundAmt);
            }
        } catch (Exception e) {
            AppLogUtil.error(logger,"请求退款异常 msg={}",e.getMessage());
        }
        //申请退款成功
    }


    /**
     * 只返回退款查询结果,更新订单状态为退款成功交给上游业务
     * @param orderId 主订单ID
     * @param fundId 资金订单ID，请求退款接口时，传入的退款请求号，如果在退款请求时未传入，则该值为主订单ID
     * @return
     */
    @Override
    public RefundResult refundQuery(String orderId, String fundId) {
        RefundResult refundResult = new RefundResult();
        try {
            Map<String,String> dataMap = buildRefundQueryParam(fundId);
            String parmaStr = FormDataConvertUtil.postFormLinkReportWithURLEncode(dataMap);
            String httpResultStr = HttpClientUtil.postData(NowPayAppUtil.getRefundUrl(),parmaStr);
            Map<String,String> resultMap = FormDataConvertUtil.parseFormDataPatternReportWithDecode(httpResultStr);
            if (null == resultMap)
                throw new QihoException("查询退款请求返回结果为空");
            //验证签名
            boolean isValidSignature = SignCheckUtil.signCheck(resultMap,Boolean.TRUE,NowPayAppUtil.getAppKey());
            String transStatus = resultMap.get(NowPayConstant.NowPayKey.tradeStatus);
            String responseCode = resultMap.get(NowPayConstant.NowPayKey.responseCode);
            if (isValidSignature
                    && NowPayConstant.refundRespCode.SUCCESS.equals(responseCode) && NowPayConstant.refundStatus.SUCCESS.equals(transStatus)){//验签成功
                //退款请求处理成功
                refundResult.setFundId(resultMap.get(NowPayConstant.NowPayKey.mhtRefundNo));
                refundResult.setOrderId(resultMap.get(NowPayConstant.NowPayKey.mhtOrderNo));
                refundResult.setRefundAmt(Integer.valueOf(resultMap.get(NowPayConstant.NowPayKey.amount)));
                refundResult.setIsSuccess(Boolean.TRUE);
                refundResult.setRefundReason(resultMap.get(NowPayConstant.NowPayKey.reason));
                refundResult.setOutTradeNo(resultMap.get(NowPayConstant.NowPayKey.nowPayOrderNo));
            }else{
                refundResult.setIsSuccess(Boolean.FALSE);
            }
        } catch (Exception e) {
            AppLogUtil.error(logger,"请求退款异常 msg={}",e.getMessage());
        }
        return refundResult;
    }

    /**
     * 延签,上游需要对延签参数处理完全
     * @see com.qiho.center.biz.paychannel.pay.PayChannelProcessor#verify(java.util.Map)
     */
    @Override
    public boolean verify(Map<String, String> params) {
        //报文签名
        String signature = params.remove(NowPayConstant.NowPayKey.signature);
        //验证签名
        return MD5Facade.validateFormDataParamMD5(params,NowPayAppUtil.getAppKey(),signature);
    }

    /**
     * 构建退款申请参数
     * @param orderId
     * @param fundId
     * @param refundAmt
     * @return
     */
    private Map<String,String> buildRefundParam(String orderId, String fundId, Integer refundAmt){
        Map<String,String> dataMap = new HashMap<>();
        dataMap.put(NowPayConstant.NowPayKey.funcode,NowPayConstant.FunCode.REFUND_SUBMIT);
        dataMap.put(NowPayConstant.NowPayKey.appId, NowPayAppUtil.getAppId());
        dataMap.put(NowPayConstant.NowPayKey.mhtOrderNo,orderId);
        dataMap.put(NowPayConstant.NowPayKey.mhtRefundNo,fundId);
        dataMap.put(NowPayConstant.NowPayKey.reason,"订单超时");
        dataMap.put(NowPayConstant.NowPayKey.mhtCharset,NowPayConstant.DefaultValue.default_charset);
        dataMap.put(NowPayConstant.NowPayKey.amount,refundAmt.toString());
        String mhtSignature = MD5Facade.getFormDataParamMD5(dataMap, NowPayAppUtil.getAppKey(),NowPayConstant.DefaultValue.default_charset);
        dataMap.put(NowPayConstant.NowPayKey.signType,NowPayConstant.DefaultValue.default_code);
        dataMap.put(NowPayConstant.NowPayKey.mhtSignature,mhtSignature);
        return dataMap;
    }

    /**
     * 构建退款查询参数
     * @param fundId
     * @return
     */
    private Map<String,String> buildRefundQueryParam(String fundId){
        Map<String,String> dataMap = new HashMap<>();
        dataMap.put(NowPayConstant.NowPayKey.funcode,NowPayConstant.FunCode.REFUND_QUERY);
        dataMap.put(NowPayConstant.NowPayKey.mhtRefundNo,fundId);
        dataMap.put(NowPayConstant.NowPayKey.mhtCharset,NowPayConstant.DefaultValue.default_charset);
        dataMap.put(NowPayConstant.NowPayKey.appId, NowPayAppUtil.getAppId());
        String mhtSignature = MD5Facade.getFormDataParamMD5(dataMap, NowPayAppUtil.getAppKey(), NowPayConstant.DefaultValue.default_charset);
        dataMap.put(NowPayConstant.NowPayKey.signType,NowPayConstant.DefaultValue.default_code);
        dataMap.put(NowPayConstant.NowPayKey.mhtSignature,mhtSignature);
        return dataMap;
    }

    /**
     * 请求支付参数参数封装
     */
    private Map<String,String> packageParamToMap(String orderId){
        List<FundOrderDto> resultFund = fundOrderService.findByOrderId(orderId);
        OrderDto orderDto = orderService.findByOrderId(orderId);
        if (null == orderDto)
            throw new QihoException("通过订单id查询订单 || 付款订单异常");
        FundOrderDto fundOrderDto = resultFund.get(0);//唯一
        Map<String, String> dataMap = new HashMap<>();
        //定值
        dataMap.put(NowPayConstant.NowPayKey.appId, NowPayAppUtil.getAppId());//识别id
        dataMap.put(NowPayConstant.NowPayKey.deviceType, NowPayConstant.DefaultValue.default_deviceType);
        dataMap.put(NowPayConstant.NowPayKey.funcode,NowPayConstant.FunCode.PAY_SUBMIT);
        dataMap.put(NowPayConstant.NowPayKey.version, NowPayConstant.DefaultValue.default_version);
        dataMap.put(NowPayConstant.NowPayKey.mhtCurrencyType, NowPayConstant.DefaultValue.def_mhtCurrencyType);
        dataMap.put(NowPayConstant.NowPayKey.mhtOrderType, NowPayConstant.DefaultValue.def_mhtOrderType);
        dataMap.put(NowPayConstant.NowPayKey.mhtCharset, NowPayConstant.DefaultValue.default_charset);
        dataMap.put(NowPayConstant.NowPayKey.payChannelType, NowPayConstant.DefaultValue.def_payChannelType);
        dataMap.put(NowPayConstant.NowPayKey.mhtSignType, NowPayConstant.DefaultValue.default_code);
        dataMap.put(NowPayConstant.NowPayKey.outputType,NowPayConstant.DefaultValue.def_outtype);
        dataMap.put(NowPayConstant.NowPayKey.mhtOrderNo, orderDto.getOrderId());
        dataMap.put(NowPayConstant.NowPayKey.mhtOrderName, orderDto.getOrderItem().getItemName());
        dataMap.put(NowPayConstant.NowPayKey.mhtOrderAmt, fundOrderDto.getAmt().toString());
        dataMap.put(NowPayConstant.NowPayKey.mhtOrderDetail, fundOrderDto.getFundId());
        SimpleDateFormat dateFormat = new SimpleDateFormat("yyyyMMddHHmmss");
        String mhtOrderStartTime = dateFormat.format(new Date());
        dataMap.put(NowPayConstant.NowPayKey.mhtOrderStartTime, mhtOrderStartTime);
        dataMap.put(NowPayConstant.NowPayKey.notifyUrl, NowPayAppUtil.getNotifyUrl());
        dataMap.put(NowPayConstant.NowPayKey.frontNotifyUrl, FRONT_NOTIFY_URL+orderId);
        String mhtSignature = MD5Facade.getFormDataParamMD5(dataMap, NowPayAppUtil.getAppKey(), NowPayConstant.DefaultValue.default_charset);
        dataMap.put(NowPayConstant.NowPayKey.mhtSignature,mhtSignature);//签名
        return dataMap;
    }

    @Override
    public PayDto queryPayResult(String orderId) {
        PayDto payResult = new PayDto();
        Map<String,String> dataMap = Maps.newHashMap();
        // TODO Auto-generated method stub
        OrderDto orderResult = orderService.findByOrderId(orderId);
        if (null == orderResult)
            throw new QihoException("发起查询支付请求失败,订单id有误");
        //封装查询请求参数
        dataMap.put(NowPayConstant.NowPayKey.appId, NowPayAppUtil.getAppId());
        dataMap.put(NowPayConstant.NowPayKey.funcode,NowPayConstant.FunCode.PAY_QUERY);
        dataMap.put(NowPayConstant.NowPayKey.mhtOrderNo, orderResult.getOrderId());
        dataMap.put(NowPayConstant.NowPayKey.version, NowPayConstant.DefaultValue.default_version);
        dataMap.put(NowPayConstant.NowPayKey.deviceType, NowPayConstant.DefaultValue.default_deviceType);
        dataMap.put(NowPayConstant.NowPayKey.mhtCharset, NowPayConstant.DefaultValue.default_charset);
        dataMap.put(NowPayConstant.NowPayKey.mhtSignType, NowPayConstant.DefaultValue.default_code);
        try{
            dataMap.put(NowPayConstant.NowPayKey.mhtSignature, MD5Facade.getFormDataParamMD5(dataMap,NowPayAppUtil.getAppKey(), "UTF-8"));
            String queryForm = FormDataConvertUtil.postFormLinkReportWithURLEncode(dataMap);
            String resultCode = HttpClientUtil.postData(NowPayAppUtil.getNowpayUrl(),queryForm);
            //返回查询结果
            Map<String,String> resultMap = FormDataConvertUtil.parseFormDataPatternReportWithDecode(resultCode);
            //验签
            boolean isValidSignature = SignCheckUtil.signCheck(resultMap,Boolean.FALSE,NowPayAppUtil.getAppKey());
            String transStatus = resultMap.get(NowPayConstant.NowPayKey.transStatus);
            if (isValidSignature && NowPayConstant.refundStatus.SUCCESS.equals(transStatus)){
                payResult.setOrderId(resultMap.get(NowPayConstant.NowPayKey.mhtOrderNo));
                payResult.setPayAmt(Integer.valueOf(resultMap.get(NowPayConstant.NowPayKey.mhtOrderAmt)));
                payResult.setPayerId(resultMap.get(NowPayConstant.NowPayKey.payConsumerId));
                payResult.setSuccess(Boolean.TRUE);
                payResult.setOutTradeNo(resultMap.get(NowPayConstant.NowPayKey.nowPayOrderNo));
            }else{
                payResult.setSuccess(Boolean.FALSE);
            }
        }catch (Exception e){
            AppLogUtil.error(logger,"现在支付查询结果异常 msg={}",e.getMessage());
        }
        return payResult;
    }

    @Override
    PayTypeEnum getPayType() {
        return PayTypeEnum.NOWPAY;
    }

    /**
     * @see org.springframework.beans.factory.InitializingBean#afterPropertiesSet()
     */
    @Override
    public void afterPropertiesSet() throws Exception {
        PayChannelDecider.registPaychannel(getPayType(), this);
    }

}
