package com.qiho.center.biz.service.impl;

import java.util.Date;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang.StringUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import cn.com.duiba.wolf.utils.BeanUtils;

import com.google.common.base.Objects;
import com.google.common.collect.BiMap;
import com.google.common.collect.HashBiMap;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.google.common.collect.Sets;
import com.qiho.center.api.dto.ExpressBlackListDto;
import com.qiho.center.api.dto.PagenationDto;
import com.qiho.center.api.dto.blackList.BlackListDto;
import com.qiho.center.api.dto.blackList.BlacklistCheckDto;
import com.qiho.center.api.enums.BlackListEnum;
import com.qiho.center.api.enums.ExpressCompanyEnum;
import com.qiho.center.api.enums.QueryBlackListTypeEnum;
import com.qiho.center.api.params.BlackListCheckParams;
import com.qiho.center.api.params.BlackListOperationParams;
import com.qiho.center.api.params.BlackListParams;
import com.qiho.center.api.params.ExpressBlackListPageParams;
import com.qiho.center.biz.service.BlackListService;
import com.qiho.center.common.dao.QihoBlacklistDAO;
import com.qiho.center.common.dao.QihoStrategyBlacklistDAO;
import com.qiho.center.common.entity.QihoBlacklistEntity;
import com.qiho.center.common.params.BlackListPageParams;

/**
 * Created by liuyao on 2017/5/22.
 */
@Service
public class BlackListServiceImpl implements BlackListService {

    /**
     * 维护快递公司到名单之间的关系(可逆MAP)
     */
    private final BiMap<ExpressCompanyEnum, BlackListEnum> blackListTypeMap = HashBiMap.create();
    {
        blackListTypeMap.put(ExpressCompanyEnum.YTO, BlackListEnum.YUAN_TONG_COD);
        blackListTypeMap.put(ExpressCompanyEnum.YTO_PAID, BlackListEnum.YUAN_TONG_PAIED);
    }

    @Autowired
    private QihoBlacklistDAO                               qihoBlacklistDAO;

    @Autowired
    private QihoStrategyBlacklistDAO                       qihoStrategyBlacklistDAO;

    @Override
    public PagenationDto<ExpressBlackListDto> findExpressBlackPage(ExpressBlackListPageParams params) {
        PagenationDto<ExpressBlackListDto> page = new PagenationDto<>();
        BlackListPageParams daoParams = new BlackListPageParams();
        BeanUtils.copy(params, daoParams);
        daoParams.setIfPage(true);
        // 快递区域黑名单存在多种类型，此处不在适应，需要修改
        List<String> bkTypes = Lists.newArrayList();
        if (Objects.equal(ExpressCompanyEnum.ALL, params.getExpressCompany())) {
            for (BlackListEnum blackEnum : blackListTypeMap.values()) {
                bkTypes.add(blackEnum.getBkType());
            }
        } else {
            bkTypes.add(blackListTypeMap.get(params.getExpressCompany()).getBkType());
        }
        daoParams.setBkTypeList(bkTypes);
        if(StringUtils.isNotBlank(params.getAddressCode())){
            daoParams.setBkValue(params.getAddressCode());
        }

        List<QihoBlacklistEntity> list = qihoBlacklistDAO.findQihoBlacklistPage(daoParams);
        Integer total = qihoBlacklistDAO.findQihoBlacklistTotal(daoParams);

        List<ExpressBlackListDto> returnList = Lists.transform(list, (entity) -> {
            ExpressBlackListDto dto = new ExpressBlackListDto();
            dto.setExpressCompany(blackListTypeMap.inverse().get(BlackListEnum.getEnumByType(entity.getBkType())));
            dto.setAddressCode(entity.getBkValue());
            dto.setId(entity.getId());
            dto.setGmtCreate(entity.getGmtCreate());
            return dto;
        });

        page.setTotal(total);
        page.setList(returnList);

        return page;
    }

    @Override
    public Long findOneBlack(ExpressCompanyEnum expressCompany, String addressCode) {
        QihoBlacklistEntity entity = qihoBlacklistDAO.findOneBlacklist(blackListTypeMap.get(expressCompany),
                                                                       addressCode);
        if (Objects.equal(entity, null)) {
            return null;
        } else {
            return entity.getId();
        }
    }

    @Override
    public Boolean deleteById(Long id, QueryBlackListTypeEnum type) {
        int ret = 0;
        switch (type) {
            case EXPRESS_COMPANY:
            case MANUAL_PHONE_NUMBER:
            case MANUAL_REGION:
                // 手动黑名单操作
                ret = qihoBlacklistDAO.deleteBlackList(id);
                break;
            case STRATEGY_PHONE_NUMBER:
            case STRATEGY_REGION:
                // 策略黑名单操作
                ret = qihoStrategyBlacklistDAO.deleteBlackList(id);
                break;
            default:
                return false;
        }
        return ret == 1;
    }

    @Override
    public Integer deleteByValue(BlackListOperationParams params) {
        if (StringUtils.isBlank(params.getBkValue())) {
            return 0;
        }
        QihoBlacklistEntity entity = new QihoBlacklistEntity();
        BlackListEnum bkType = getBkTypeFromQueryType(params.getType());
        if (bkType == null) {
            return 0;
        }
        entity.setBkType(bkType.getBkType());
        entity.setBkValue(params.getBkValue());
        return qihoBlacklistDAO.deleteByValue(entity);
    }

    @Override
    public Integer batchAddExpressBlackList(ExpressCompanyEnum expressCompany, Set<String> addressCodes) {

        BlackListEnum blackListEnum = blackListTypeMap.get(expressCompany);
        List<String> list = qihoBlacklistDAO.findExistByTypeAndIdSet(blackListEnum, addressCodes);
        Set<String> existSet = Sets.newHashSet(list);
        return qihoBlacklistDAO.batchAddBlackList(blackListTypeMap.get(expressCompany),
                                                  Sets.difference(addressCodes, existSet));
    }

    @Override
    public Integer findBlackListCount(BlackListParams params) {
        BlackListPageParams daoParams = new BlackListPageParams();
        BeanUtils.copy(params, daoParams);
        if (StringUtils.isNotBlank(daoParams.getBkValue())) {
            daoParams.setBklength(daoParams.getBkValue().length());
        }
        Integer total = 0;
        BlackListEnum bkType = getBkTypeFromQueryType(params.getType());
        if (bkType == null) {
            return 0;
        }
        daoParams.setBkType(bkType.getBkType());

        switch (params.getType()) {
            case MANUAL_PHONE_NUMBER:
            case MANUAL_REGION:
                // 手动黑名单查询qiho表
                total = qihoBlacklistDAO.findQihoBlacklistTotal(daoParams);
                break;
            case STRATEGY_PHONE_NUMBER:
            case STRATEGY_REGION:
                // 策略黑名单查询qiho_statistics静态表
                total = qihoStrategyBlacklistDAO.findQihoBlacklistTotal(daoParams);
                break;
            default:
                return 0;
        }
        return total;
    }

    @Override
    public PagenationDto<BlackListDto> queryBlackListPage(BlackListParams params) {
        PagenationDto<BlackListDto> page = new PagenationDto<>();
        BlackListPageParams daoParams = new BlackListPageParams();
        BeanUtils.copy(params, daoParams);
        if (StringUtils.isNotBlank(daoParams.getBkValue())) {
            daoParams.setBklength(daoParams.getBkValue().length());
        }
        List<QihoBlacklistEntity> list = Lists.newArrayList();
        Integer total = 0;
        BlackListEnum bkType = getBkTypeFromQueryType(params.getType());
        if (bkType == null) {
            return page;
        }
        daoParams.setBkType(bkType.getBkType());

        switch (params.getType()) {
            case MANUAL_PHONE_NUMBER:
            case MANUAL_REGION:
                // 手动黑名单查询
                list = qihoBlacklistDAO.findQihoBlacklistPage(daoParams);
                total = qihoBlacklistDAO.findQihoBlacklistTotal(daoParams);
                break;
            case STRATEGY_PHONE_NUMBER:
            case STRATEGY_REGION:
                // 策略黑名单查询
                list = qihoStrategyBlacklistDAO.findQihoBlacklistPage(daoParams);
                total = qihoStrategyBlacklistDAO.findQihoBlacklistTotal(daoParams);
                break;
            default:
                return page;
        }

        List<BlackListDto> returnList = Lists.transform(list, (entity) -> {
            BlackListDto dto = BeanUtils.copy(entity, BlackListDto.class);
            return dto;
        });

        page.setTotal(total);
        page.setList(returnList);

        return page;
    }

    @Override
    public Integer addBlackValue(BlackListOperationParams params) {
        if (StringUtils.isBlank(params.getBkValue())) {
            return 0;
        }
        QihoBlacklistEntity entity = new QihoBlacklistEntity();
        BlackListEnum bkType = getBkTypeFromQueryType(params.getType());
        if (bkType == null) {
            return 0;
        }
        entity.setBkType(bkType.getBkType());
        entity.setBkValue(params.getBkValue());
        entity.setRemarks(params.getRemarks());
        return qihoBlacklistDAO.insert(entity);
    }

    @Override
    public BlackListDto findOneBlacklist(QueryBlackListTypeEnum type, String bkValue) {

        BlackListEnum bkType = getBkTypeFromQueryType(type);
        QihoBlacklistEntity entity = new QihoBlacklistEntity();
        switch (type) {
            case EXPRESS_COMPANY:
            case MANUAL_PHONE_NUMBER:
            case MANUAL_REGION:
                // 手动黑名单查询
                entity = qihoBlacklistDAO.findOneBlacklist(bkType, bkValue);
                break;
            default:
                return null;
        }

        return BeanUtils.copy(entity, BlackListDto.class);
    }

    @Override
    public BlackListDto findOneBlacklist(QueryBlackListTypeEnum type, String bkValue, Date curDate) {
        BlackListEnum bkType = getBkTypeFromQueryType(type);
        QihoBlacklistEntity entity = qihoStrategyBlacklistDAO.findOneBlacklist(bkType, bkValue, curDate);
        return BeanUtils.copy(entity, BlackListDto.class);
    }

    @Override
    public BlackListDto findLatestDateByType(QueryBlackListTypeEnum type) {
        BlackListEnum bkType = getBkTypeFromQueryType(type);
        QihoBlacklistEntity qihoBlacklistEntity = qihoStrategyBlacklistDAO.findLatestDateByType(bkType);
        return BeanUtils.copy(qihoBlacklistEntity, BlackListDto.class);
    }

    private BlackListEnum getBkTypeFromQueryType(QueryBlackListTypeEnum type) {
        switch (type) {
            case EXPRESS_COMPANY:
                return BlackListEnum.YUAN_TONG_COD;
            case EXPRESS_COMPANY_PUTO:
                return BlackListEnum.YUAN_TONG_PAIED;
            case MANUAL_PHONE_NUMBER:
            case STRATEGY_PHONE_NUMBER:
                return BlackListEnum.PHONE_NUMBER;
            case MANUAL_REGION:
            case STRATEGY_REGION:
                return BlackListEnum.REGION;
            default:
                return null;
        }
    }

    /**
     * @see com.qiho.center.biz.service.BlackListService#blacklistCheck(com.qiho.center.api.params.BlackListCheckParams)
     */
    @Override
    public BlacklistCheckDto blacklistCheck(BlackListCheckParams params) {
        BlacklistCheckDto dto = new BlacklistCheckDto();
        if(StringUtils.isBlank(params.getBkValue()) || CollectionUtils.isEmpty(params.getCheckTypes())){
            return dto;
        }
        // 奇货本地库黑名单
        List<Map<String, Object>> list = qihoBlacklistDAO.countByType(params.getBkValue(), params.getCheckTypes());
        Map<BlackListEnum, Long> bkTypeMap = Maps.newHashMap();
        fillMap(bkTypeMap, list);
        // 奇货离线库黑名单
        QihoBlacklistEntity qihoBlacklistEntity = qihoStrategyBlacklistDAO.findLatestDateByType(BlackListEnum.PHONE_NUMBER);
        Date curDate = qihoBlacklistEntity.getCurDate();
        List<Map<String, Object>> strategyList = qihoStrategyBlacklistDAO.countByType(params.getBkValue(), params.getCheckTypes(), curDate);
        fillMap(bkTypeMap, strategyList);
        dto.setBkValue(params.getBkValue());
        dto.setBkTypeMap(bkTypeMap);
        return dto;
    }

    @Override
    public Integer findExpressBlackNum(ExpressBlackListPageParams params) {
        BlackListPageParams daoParams = new BlackListPageParams();
        BeanUtils.copy(params, daoParams);
        daoParams.setIfPage(true);
        // 快递区域黑名单存在多种类型，此处不在适应，需要修改
        List<String> bkTypes = Lists.newArrayList();
        if (Objects.equal(ExpressCompanyEnum.ALL, params.getExpressCompany())) {
            for (BlackListEnum blackEnum : blackListTypeMap.values()) {
                bkTypes.add(blackEnum.getBkType());
            }
        } else {
            bkTypes.add(blackListTypeMap.get(params.getExpressCompany()).getBkType());
        }
        daoParams.setBkTypeList(bkTypes);
        if(StringUtils.isNotBlank(params.getAddressCode())){
            daoParams.setBkValue(params.getAddressCode());
        }

        Integer total = qihoBlacklistDAO.findQihoBlacklistTotal(daoParams);
        return total;
    }

    /**
     * @author zhanglihui
     * @param bkTypeMap
     * @param list
     */
    private void fillMap(Map<BlackListEnum, Long> bkTypeMap, List<Map<String, Object>> list) {
        if (CollectionUtils.isNotEmpty(list)) {
            for (Map<String, Object> map : list) {
                BlackListEnum bkType = BlackListEnum.getEnumByType((String) map.get("bkType"));
                Long count = (Long) map.get("count");
                Long icount = bkTypeMap.get(bkType);
                if (icount != null) {
                    count += icount;
                }
                bkTypeMap.put(bkType, count);
            }
        }
    }

}
