/**
 * Project Name:qiho-center-biz File Name:CategoryServiceImpl.java Package Name:com.qiho.center.biz.service.impl
 * Date:2017年5月17日上午9:21:55 Copyright (c) 2017, duiba.com.cn All Rights Reserved.
 */

package com.qiho.center.biz.service.impl;

import java.util.List;

import org.apache.commons.collections.CollectionUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import cn.com.duiba.wolf.utils.BeanUtils;

import com.google.common.base.Objects;
import com.google.common.collect.Lists;
import com.qiho.center.api.dto.CategoryDto;
import com.qiho.center.biz.service.CategoryService;
import com.qiho.center.common.constant.DsConstants;
import com.qiho.center.common.dao.QihoCategoryDAO;
import com.qiho.center.common.entity.item.QihoCategoryEntity;

/**
 * ClassName:CategoryServiceImpl <br/>
 * 类目service <br/>
 * Date: 2017年5月17日 上午9:21:55 <br/>
 * 
 * @author zhanglihui
 * @version
 * @since JDK 1.6
 * @see
 */
@Service
public class CategoryServiceImpl implements CategoryService {

    @Autowired
    private QihoCategoryDAO qihoCategoryDAO;
    
    /** 一级类目的父节点ID */
    private static final long TOP_CATEGORY_PID = 0;

    @Override
    @Transactional(DsConstants.DATABASE_QIHO)
    public void sortByRange(Long pid, Integer startPayload, Integer endPayload) {

        if (Objects.equal(startPayload, endPayload)) {
            return;
        }
        List<QihoCategoryEntity> list = qihoCategoryDAO.findListBetweenPayload(pid, startPayload, endPayload);
        if (CollectionUtils.isNotEmpty(list)) {
            // 被移动的类目payload更新为endPayload
            qihoCategoryDAO.updateByPayload(pid, startPayload, endPayload);
            List<Long> categoryIdList = Lists.newArrayList();
            for (QihoCategoryEntity category : list) {
                if (category.getPayload() != startPayload) {
                    categoryIdList.add(category.getId());
                }
            }
            int offset = 0;
            // 如果向下移动，区间内的类目排序值+1
            if (startPayload > endPayload) {
                offset = 1;
            } else {
                // 如果向上移动，区间内的类目排序值-1
                offset = -1;
            }
            qihoCategoryDAO.updateBatchByOffset(categoryIdList, offset);
        }
    }

    /**
     * @see com.qiho.center.biz.service.CategoryService#getCategoryByPid(java.lang.Long)
     */
    @Override
    public List<CategoryDto> getCategoryByPid(Long pid) {
        List<QihoCategoryEntity> list = qihoCategoryDAO.getCategoryByPid(pid);
        return BeanUtils.copyList(list, CategoryDto.class);
    }
    
    /**
     * @see com.qiho.center.biz.service.CategoryService#deleteCategoryById(java.lang.Long)
     */
    @Override
    @Transactional(DsConstants.DATABASE_QIHO)
    public void deleteCategoryById(Long categoryId) {
        qihoCategoryDAO.deleteByCategoryId(categoryId);
        QihoCategoryEntity category = qihoCategoryDAO.findByCategoryId(categoryId);
        if(category.getPid() != TOP_CATEGORY_PID){
            int payload = qihoCategoryDAO.findMaxPlayload(category.getPid());
            if(payload == 0){
                //如果删除子类目后，父类目下没有其他子类目，需要把父节点降级为叶子节点
                qihoCategoryDAO.updateCategoryIfParent(category.getPid(), false);
            }
        }
    }

    /**
     * @see com.qiho.center.biz.service.CategoryService#updateCategoryById(java.lang.Long, java.lang.String)
     */
    @Override
    public void updateCategoryById(Long categoryId, String categoryName) {
        qihoCategoryDAO.updateCategoryById(categoryId, categoryName);
    }

    /**
     * @see com.qiho.center.biz.service.CategoryService#saveCategory(java.lang.Long, java.util.List)
     */
    @Override
    @Transactional(DsConstants.DATABASE_QIHO)
    public List<CategoryDto> saveCategory(Long pid, List<String> categoryList) {
        List<QihoCategoryEntity> list = Lists.newArrayList();
        int payload = qihoCategoryDAO.findMaxPlayload(pid);
        if(payload == 0 && pid != TOP_CATEGORY_PID){
            //原来是叶子节点，标记成父节点
            qihoCategoryDAO.updateCategoryIfParent(pid, true);
        }
        for(String categoryName : categoryList){
            payload += 1;
            QihoCategoryEntity category = new QihoCategoryEntity();
            category.setCategoryName(categoryName);
            category.setDeleted(false);
            category.setIsParent(false);
            category.setPayload(payload);
            category.setPid(pid);
            list.add(category);
        }
        return BeanUtils.copyList(qihoCategoryDAO.saveCategory(list), CategoryDto.class);
    }
}
