package com.qiho.center.biz.bo;

import com.google.common.collect.Lists;
import com.qiho.center.api.dto.OrderDto;
import com.qiho.center.api.dto.blackList.BlackListDto;
import com.qiho.center.api.enums.QueryBlackListTypeEnum;
import com.qiho.center.api.params.BlackListOperationParams;
import com.qiho.center.api.params.ExpressBlackListOperationParams;
import com.qiho.center.biz.runnable.BlackListRunnable;
import com.qiho.center.biz.runnable.ExpressBlackListRunnable;
import com.qiho.center.biz.service.BlackListService;
import com.qiho.center.biz.service.order.AnticheateService;
import com.qiho.center.biz.service.order.OrderService;
import com.qiho.center.common.entity.order.QihoAnticheateEntity;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang.StringUtils;
import org.springframework.beans.BeansException;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.ApplicationContext;
import org.springframework.context.ApplicationContextAware;
import org.springframework.stereotype.Service;

import javax.annotation.Resource;
import java.util.List;
import java.util.concurrent.ExecutorService;

/**
 * ClassName: BlackListBo <br/>
 * Function: 黑名单处理. <br/>
 * date: 2017年6月26日 下午8:05:35 <br/>
 *
 * @author chencheng
 * @version
 */
@Service
public class BlackListBo implements ApplicationContextAware {

    private ApplicationContext context;

    @Resource
    private ExecutorService    executorService;

    @Autowired
    private BlackListService   blackListService;

    @Autowired
    private OrderService       orderService;

    @Resource
    private AnticheateService  anticheateService;

    /**
     * 批量黑名单处理
     *
     * @author chencheng
     * @param progressKey
     * @param paramList
     * @return
     */
    public Integer batchBlackList(String progressKey, List<BlackListOperationParams> paramList) {
        BlackListRunnable runnable = context.getBean(BlackListRunnable.class);
        runnable.setProgressKey(progressKey);
        runnable.setParams(paramList);
        int enableCount = runnable.enableFilter();
        executorService.submit(runnable);
        return enableCount;
    }

    /**
     * 批量物流黑名单处理
     * @param paramsList
     * @throws BeansException
     */
    public Integer batchExpressBlackList(String progressKey, List<ExpressBlackListOperationParams> paramsList){
        ExpressBlackListRunnable runnable = context.getBean(ExpressBlackListRunnable.class);
        runnable.setProgressKey(progressKey);
        runnable.setParams(paramsList);
        int enableCount = runnable.enableFilter();
        executorService.submit(runnable);
        return enableCount;
    }

    @Override
    public void setApplicationContext(ApplicationContext applicationContext) throws BeansException {
        this.context = applicationContext;
    }

    /**
     * 防作弊黑名单校验
     *
     * @author chencheng
     * @param orderId
     */
    public void cheatProof(String orderId) {
        OrderDto orderDto = orderService.findByOrderId(orderId);
        if (orderDto == null) {
            return;
        }
        List<QihoAnticheateEntity> list = Lists.newArrayList();
        //手机手动黑名单
        list.addAll(phoneBlack(orderDto));
        //地址黑名单
        list.addAll(areaBlack(orderDto));

        if (CollectionUtils.isNotEmpty(list)) {
            for (QihoAnticheateEntity qihoAnticheateEntity2 : list) {
                if(null==qihoAnticheateEntity2){
                    continue;
                }
                anticheateService.insert(qihoAnticheateEntity2);
            }
        }
    }

    private List<QihoAnticheateEntity> phoneBlack(OrderDto orderDto){
        List<QihoAnticheateEntity> list = Lists.newArrayList();
        if (StringUtils.isNotBlank(orderDto.getMobile())) {
            // 手机号手动添加黑名单
            BlackListDto manualPhoneDto = blackListService.findOneBlacklist(QueryBlackListTypeEnum.MANUAL_PHONE_NUMBER,orderDto.getMobile());
            if (manualPhoneDto != null) {
                list.add(new QihoAnticheateEntity(orderDto.getOrderId(), QueryBlackListTypeEnum.MANUAL_PHONE_NUMBER.getDesc()));
            }
            // 手机号策略黑名单
            BlackListDto blackListDto = blackListService.findLatestDateByType(QueryBlackListTypeEnum.STRATEGY_PHONE_NUMBER);
            BlackListDto strategyPhoneDto = null;
            if (null != blackListDto){
                strategyPhoneDto = blackListService.findOneBlacklist(QueryBlackListTypeEnum.STRATEGY_PHONE_NUMBER,
                        orderDto.getMobile(),blackListDto.getCurDate());
            }
            if (strategyPhoneDto != null) {
                list.add(new QihoAnticheateEntity(orderDto.getOrderId(), QueryBlackListTypeEnum.STRATEGY_PHONE_NUMBER.getDesc()));
            }
        }
        return list;
    }

    private List<QihoAnticheateEntity> areaBlack(OrderDto  orderDto){
        List<QihoAnticheateEntity> list = Lists.newArrayList();
        if (StringUtils.isNotBlank(orderDto.getAreaNum())) {
            // 地区手动添加黑名单
            BlackListDto manualRegionDto = blackListService.findOneBlacklist(QueryBlackListTypeEnum.MANUAL_REGION,
                    orderDto.getAreaNum());
            if (manualRegionDto != null) {
                list.add(new QihoAnticheateEntity(orderDto.getOrderId(), QueryBlackListTypeEnum.MANUAL_REGION.getDesc()));
            }
            // 地区策略黑名单
            BlackListDto blackListDto = blackListService.findLatestDateByType(QueryBlackListTypeEnum.STRATEGY_REGION);
            BlackListDto strategyRegionDto = null;
            if (null != blackListDto){
                strategyRegionDto = blackListService.findOneBlacklist(QueryBlackListTypeEnum.STRATEGY_REGION,
                        orderDto.getMobile(),blackListDto.getCurDate());
            }
            if (strategyRegionDto != null) {
                list.add(new QihoAnticheateEntity(orderDto.getOrderId(), QueryBlackListTypeEnum.STRATEGY_REGION.getDesc()));
            }
        }
        return list;
    }

}
