package com.qiho.center.biz.runnable;

import java.util.List;

import javax.annotation.Resource;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.annotation.Scope;
import org.springframework.data.redis.core.BoundHashOperations;
import org.springframework.data.redis.core.StringRedisTemplate;
import org.springframework.stereotype.Component;

import com.google.common.base.Objects;
import com.google.common.collect.Lists;
import com.qiho.center.api.dto.ErpOrderDto;
import com.qiho.center.api.enums.ErpStatusEnum;
import com.qiho.center.api.exception.QihoException;
import com.qiho.center.api.params.BatchResolveErpDeliverParams;
import com.qiho.center.api.params.ResolveDeliverParams;
import com.qiho.center.biz.bo.ErpOrderBo;
import com.qiho.center.biz.service.order.ErpOrderService;

/**
 * Created by liuyao on 2017/6/14.
 */
@Component
@Scope("prototype")
public class DealWithErpOrderDeliverRunnable implements Runnable{

    private Logger logger = LoggerFactory.getLogger(DealWithErpOrderDeliverRunnable.class);

    @Autowired
    private ErpOrderBo erpOrderBo;
    @Autowired
    private ErpOrderService erpOrderService;

    @Resource(name = "stringRedisTemplate")
    private StringRedisTemplate redisTemplate;

    private List<BatchResolveErpDeliverParams> paramsList = Lists.newArrayList();

    private BoundHashOperations<String,String,Object> ops;

    public Integer init(){
        return paramsList.size();
    }

    @Override
    public void run() {
        long failCount = 0L;
        try{
            for(BatchResolveErpDeliverParams param:paramsList){
                try{
                    switch (param.getDealResult()){
                        case BatchResolveErpDeliverParams.DEAL_RESULT_SUCCESS:
                            resolveDeliver(param) ;
                            break;
                        case BatchResolveErpDeliverParams.DEAL_RESULT_FAILD:
                            rejectDeliver(param);
                            break;
                        default:
                            break;
                    }
                }catch(Exception e){
                    failCount++;
                    logger.error("ERP处理",e);
                }finally {
                    ops.increment("successCount",1L);
                }
            }
        }finally {
            ops.increment("failCount",failCount);
            ops.increment("taskCount",-1L);
        }
    }

    private void resolveDeliver(BatchResolveErpDeliverParams params){
        ErpOrderDto erp = findEnableErpOrder(params.getOrderId());
        if(Objects.equal(null,erp)){
            throw new QihoException("该订单下未找到待通知的ERP订单");
        }

        ResolveDeliverParams pp = new ResolveDeliverParams();
        pp.setExpress(params.getExpress());
        pp.setExpressCode(params.getExpressCode());
        pp.setErpId(erp.getErpId());
        erpOrderBo.resolveDeliver(pp);
    }


    private void rejectDeliver(BatchResolveErpDeliverParams params){
        ErpOrderDto erp = findEnableErpOrder(params.getOrderId());
        if(Objects.equal(null,erp)){
            throw new QihoException("该订单下未找到待通知的ERP订单");
        }
        erpOrderBo.cancelDeliver(erp.getErpId());
    }

    private ErpOrderDto findEnableErpOrder(String orderId){
        List<ErpOrderDto> list = erpOrderService.findByOrderId(orderId);

        ErpOrderDto erp = null;
        for(ErpOrderDto dto:list) {
            if (Objects.equal(dto.getErpStatus(), ErpStatusEnum.TO_NOTIFY.getCode())) {
                erp = dto;
                break;
            }
        }
        return erp;
    }

    public void setProgressKey(String progressKey) {
        this.ops = redisTemplate.boundHashOps(progressKey);
    }

    public void setParamsList(List<BatchResolveErpDeliverParams> paramsList) {
        this.paramsList.addAll(paramsList);
    }
}
