/**
 * Project Name:qiho-center-biz File Name:ItemServiceImpl.java Package Name:com.qiho.center.biz.service.impl
 * Date:2017年5月18日上午10:23:17 Copyright (c) 2017, duiba.com.cn All Rights Reserved.
 */

package com.qiho.center.biz.service.impl;

import java.util.List;
import java.util.Map;
import java.util.Set;

import org.apache.commons.collections.CollectionUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import cn.com.duiba.stock.service.api.remoteservice.RemoteStockService;
import cn.com.duiba.wolf.utils.BeanUtils;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.TypeReference;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.qiho.center.api.dto.ItemDetailDto;
import com.qiho.center.api.dto.ItemDto;
import com.qiho.center.api.dto.ItemSimpleDto;
import com.qiho.center.api.dto.PagenationDto;
import com.qiho.center.api.dto.SkuPropertyDto;
import com.qiho.center.biz.event.ItemDeleteEvent;
import com.qiho.center.biz.service.ItemService;
import com.qiho.center.biz.service.ItemSkuService;
import com.qiho.center.biz.service.SkuPropertyService;
import com.qiho.center.common.constant.DsConstants;
import com.qiho.center.common.dao.QihoItemDAO;
import com.qiho.center.common.dao.QihoItemExtDAO;
import com.qiho.center.common.dao.QihoItemRecommendDAO;
import com.qiho.center.common.dao.QihoItemSkuDAO;
import com.qiho.center.common.entity.item.QihoItemDetailEntity;
import com.qiho.center.common.entity.item.QihoItemEntity;
import com.qiho.center.common.entity.item.QihoItemExtEntity;
import com.qiho.center.common.entity.item.QihoItemSkuEntity;
import com.qiho.center.common.support.BizEventBus;

/**
 * ClassName:ItemServiceImpl <br/>
 * 商品服务service <br/>
 * Date: 2017年5月18日 上午10:23:17 <br/>
 * 
 * @author zhanglihui
 * @version
 * @since JDK 1.6
 * @see
 */
@Service
public class ItemServiceImpl implements ItemService {

    @Autowired
    private QihoItemDAO          qihoItemDAO;

    @Autowired
    private QihoItemSkuDAO       qihoItemSkuDAO;

    @Autowired
    private QihoItemExtDAO       qihoItemExtDAO;

    @Autowired
    private QihoItemRecommendDAO qihoItemRecommendDAO;

    @Autowired
    private SkuPropertyService   skuPropertyService;

    @Autowired
    private ItemSkuService       itemSkuService;

    @Autowired
    private RemoteStockService   remoteStockService;

    @Autowired
    private BizEventBus          eventBus;

    @Override
    public PagenationDto<ItemDto> queryItemPage(ItemDto queryItem, Integer pageNum, Integer pageSize) {
        PagenationDto<ItemDto> pagenation = new PagenationDto<>();
        QihoItemEntity param = BeanUtils.copy(queryItem, QihoItemEntity.class);
        List<QihoItemDetailEntity> list = qihoItemDAO.queryByParam(param, pageNum, pageSize);
        int total = qihoItemDAO.countByParam(param);
        pagenation.setList(convertToItemDto(list));
        pagenation.setTotal(total);
        return pagenation;
    }

    /**
     * @author zhanglihui
     * @param list
     * @return
     */
    private List<ItemDto> convertToItemDto(List<QihoItemDetailEntity> list) {
        List<ItemDto> itemList = Lists.newArrayList();
        if (CollectionUtils.isEmpty(list)) {
            return itemList;
        }
        List<Long> idList = Lists.newArrayList();
        for (QihoItemDetailEntity entity : list) {
            idList.add(entity.getId());
        }
        List<QihoItemSkuEntity> skuList = qihoItemSkuDAO.getSkuByItemIds(idList);
        List<Long> stockIds = Lists.newArrayList();
        for (QihoItemSkuEntity skuEntity : skuList) {
            stockIds.add(skuEntity.getStockId());
        }
        // stockId : 库存 键值对
        Map<Long, Long> stockMap = remoteStockService.findBatch(stockIds).getResult();
        // itemId : 库存 键值对
        Map<Long, Long> itemStockMap = Maps.newHashMap();
        //有效库存
        Map<Long,Long> itemEnableStockMap = Maps.newHashMap();
        // itemId: 最小价格
        Map<Long, Integer> minPriceMap = Maps.newHashMap();
        // itemId: 最小原价
        Map<Long, Integer> minOriginalPriceMap = Maps.newHashMap();
        for (QihoItemSkuEntity skuEntity : skuList) {
            // sku库存
            Long stock = stockMap.get(skuEntity.getStockId());
            if (stock == null || stock <= 0) {
                stock = 0l;
            }
            Long itemId = skuEntity.getItemId();
            // 商品总库存
            Long itemStock = itemStockMap.get(itemId);
            if (itemStock == null) {
                itemStockMap.put(itemId, stock);
            } else {
                itemStockMap.put(itemId, itemStock + stock);
            }
            if(skuEntity.getSkuEnable()){
                Long itemEnableStock = itemEnableStockMap.get(itemId);
                if(itemEnableStock==null){
                    itemEnableStockMap.put(itemId,stock);
                }else{
                    itemEnableStockMap.put(itemId, itemEnableStock + stock);
                }
            }

            // 商品最小价格
            Integer minPrice = minPriceMap.get(itemId);
            Integer sellingPrice = skuEntity.getSellingPrice() == null ? 0 : skuEntity.getSellingPrice();
            if (minPrice == null) {
                minPriceMap.put(itemId, sellingPrice);
            } else {
                minPriceMap.put(itemId, Math.min(minPrice, sellingPrice));
            }
            // 商品最小原价
            Integer minOriginalPrice = minOriginalPriceMap.get(itemId);
            Integer originalPrice = skuEntity.getOriginalPrice() == null ? 0 : skuEntity.getOriginalPrice();
            if (minOriginalPrice == null) {
                minOriginalPriceMap.put(itemId, originalPrice);
            } else {
                minOriginalPriceMap.put(itemId, Math.min(minOriginalPrice, originalPrice));
            }
        }
        Set<Long> recommendIds = qihoItemRecommendDAO.findAllRecommendItemId();
        for (QihoItemDetailEntity entity : list) {
            ItemDto dto = new ItemDto();
            dto.setDeleted(entity.getDeleted());
            dto.setId(entity.getId());
            dto.setItemName(entity.getItemName());
            dto.setItemShortName(entity.getItemShortName());
            dto.setItemNo(entity.getItemNo());
            dto.setItemStatus(entity.getItemStatus());
            dto.setTopCategoryId(entity.getTopCategoryId());
            dto.setDelivery(entity.getDelivery());
            dto.setStock(itemStockMap.get(entity.getId()));
            Long enableStock = itemEnableStockMap.get(entity.getId());
            if(enableStock==null){
                dto.setEnableStock(0L);
            }else{
                dto.setEnableStock(enableStock);
            }
            dto.setMinPrice(minPriceMap.get(entity.getId()));
            dto.setMinOriginalPrice(minOriginalPriceMap.get(entity.getId()));
            dto.setIsRecommend(recommendIds.contains(entity.getId()));
            Map<String, String> extParam = JSON.parseObject(entity.getExtParam(),
                                                            new TypeReference<Map<String, String>>() {
                                                            });
            dto.setExtParam(extParam);
            itemList.add(dto);
        }
        return itemList;
    }

    /**
     * @see com.qiho.center.biz.service.ItemService#createItem(com.qiho.center.api.dto.ItemDetailDto)
     */
    @Override
    @Transactional(DsConstants.DATABASE_QIHO)
    public Long createItem(ItemDetailDto itemDetail) {
        QihoItemEntity item = BeanUtils.copy(itemDetail, QihoItemEntity.class);
        item.setDeleted(false);
        qihoItemDAO.insert(item);
        Long itemId = item.getId();

        QihoItemExtEntity itemExt = new QihoItemExtEntity();
        itemExt.setDetail(itemDetail.getDetail());
        itemExt.setExtParam(JSON.toJSONString(itemDetail.getExtParam()));
        itemExt.setItemId(itemId);
        qihoItemExtDAO.insert(itemExt);

        // 关联商品属性
        List<SkuPropertyDto> propertyList = itemDetail.getSkuPropertyList();
        skuPropertyService.updateItemId(propertyList, itemId);
        // 生效属性
        skuPropertyService.enableProperty(propertyList, itemId);

        // 创建SKU
        itemSkuService.updateItemSku(itemDetail.getSkuList(), itemId);
        return itemId;
    }

    /**
     * @see com.qiho.center.biz.service.ItemService#updateItem(com.qiho.center.api.dto.ItemDetailDto)
     */
    @Override
    @Transactional(DsConstants.DATABASE_QIHO)
    public void updateItem(ItemDetailDto itemDetail) {
        // 更新商品详情
        QihoItemEntity item = BeanUtils.copy(itemDetail, QihoItemEntity.class);
        qihoItemDAO.update(item);

        Long itemId = itemDetail.getId();
        QihoItemExtEntity itemExt = new QihoItemExtEntity();
        itemExt.setDetail(itemDetail.getDetail());
        itemExt.setExtParam(JSON.toJSONString(itemDetail.getExtParam()));
        itemExt.setItemId(itemId);
        qihoItemExtDAO.updateByItemId(itemExt);

        // 关联商品属性
        List<SkuPropertyDto> propertyList = itemDetail.getSkuPropertyList();
        skuPropertyService.updateItemId(propertyList, itemId);
        // 生效属性
        skuPropertyService.enableProperty(propertyList, itemId);

        // 更新sku信息
        itemSkuService.updateItemSku(itemDetail.getSkuList(), itemId);
    }

    /**
     * @see com.qiho.center.biz.service.ItemService#queryItemDetail(java.lang.Long)
     */
    @Override
    public ItemDetailDto queryItemDetail(Long itemId) {
        QihoItemDetailEntity qihoItemDetailEntity = qihoItemDAO.findById(itemId);
        if(qihoItemDetailEntity == null){
            return null;
        }
        ItemDto itemDto = convertToItemDto(Lists.newArrayList(qihoItemDetailEntity)).get(0);
        ItemDetailDto itemDetailDto = BeanUtils.copy(itemDto, ItemDetailDto.class);
        itemDetailDto.setDetail(qihoItemDetailEntity.getDetail());
        itemDetailDto.setPriceText(qihoItemDetailEntity.getPriceText());
        itemDetailDto.setGmtCreator(qihoItemDetailEntity.getGmtCreator());
        itemDetailDto.setGmtModifier(qihoItemDetailEntity.getGmtModifier());
        itemDetailDto.setGmtModified(qihoItemDetailEntity.getGmtModified());
        itemDetailDto.setSkuPropertyList(skuPropertyService.getSkuPropertyByItemId(itemId));
        itemDetailDto.setSkuList(itemSkuService.getSkuByItemId(itemId));
        itemDetailDto.setDeleted(qihoItemDetailEntity.getDeleted());
        return itemDetailDto;
    }

    /**
     * @see com.qiho.center.biz.service.ItemService#updateStatus(java.util.List, java.lang.String)
     */
    @Override
    public void updateStatus(List<Long> itemIds, String status) {
        qihoItemDAO.updateStatus(itemIds, status);
    }

    @Override
    public void deleteBatch(List<Long> itemIds) {
        qihoItemDAO.deleteBatch(itemIds);
        // 发布删除商品事件
        ItemDeleteEvent event = new ItemDeleteEvent();
        event.addAllItemId(itemIds);
        eventBus.post(event);
    }

    @Override
    public List<ItemDto> findItemDtoByItemIds(Set<Long> itemIds) {
        List<QihoItemDetailEntity> list = qihoItemDAO.findByIds(Lists.newArrayList(itemIds));
        List<QihoItemDetailEntity> newList = Lists.newArrayList();
        for (QihoItemDetailEntity entity : list) {
            if (!entity.getDeleted()) {
                newList.add(entity);
            }
        }
        return convertToItemDto(newList);
    }

    /**
     * @see com.qiho.center.biz.service.ItemService#findRecommendIndex()
     */
    @Override
    public List<ItemDto> findRecommendIndex() {
        List<QihoItemDetailEntity> recommendList = qihoItemDAO.findRecommendIndex();
        return convertToItemDto(recommendList);
    }

    /**
     * @see com.qiho.center.biz.service.ItemService#queryItemSimpleDto(java.lang.Long)
     */
    @Override
    public ItemSimpleDto queryItemSimpleDto(Long itemId) {
        QihoItemEntity entity = qihoItemDAO.findSimpleById(itemId);
        return BeanUtils.copy(entity, ItemSimpleDto.class);
    }
}
