package com.qiho.center.biz.service.impl;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;

import com.qiho.center.api.dto.FundOrderDto;
import com.qiho.center.api.enums.*;
import com.qiho.center.biz.service.order.ErpOrderService;
import com.qiho.center.biz.service.order.FundOrderService;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import cn.com.duiba.wolf.utils.BeanUtils;

import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.qiho.center.api.dto.ErpOrderDto;
import com.qiho.center.api.dto.PagenationDto;
import com.qiho.center.api.dto.logistics.ErpLogisticsSyncAckDto;
import com.qiho.center.api.dto.logistics.ErpLogisticsSyncDto;
import com.qiho.center.api.dto.logistics.LogisticsOrderDto;
import com.qiho.center.api.dto.logistics.QueryLogisticsOrderDto;
import com.qiho.center.api.params.ResolveDeliverParams;
import com.qiho.center.biz.event.LogisticsInfoUpdateEvent;
import com.qiho.center.biz.event.LogisticsUpdateEvent;
import com.qiho.center.biz.service.LogisticsOrderService;
import com.qiho.center.common.constant.DsConstants;
import com.qiho.center.common.dao.QihoErpOrderDAO;
import com.qiho.center.common.dao.QihoFundOrderDAO;
import com.qiho.center.common.dao.QihoLogisticsOrderDAO;
import com.qiho.center.common.dao.QihoOrderDAO;
import com.qiho.center.common.dao.QihoOrderSnapshotDAO;
import com.qiho.center.common.entity.fund.QihoFundOrderEntity;
import com.qiho.center.common.entity.logistics.LogisticsOrderEntity;
import com.qiho.center.common.entity.order.QihoErpOrderEntity;
import com.qiho.center.common.entity.order.QihoOrderEntity;
import com.qiho.center.common.entity.order.QihoOrderSnapshotEntity;
import com.qiho.center.common.enums.SeqBizTypeEnum;
import com.qiho.center.common.params.ErpLogisticsOrderParams;
import com.qiho.center.common.params.QueryYTOLogisticProcessParams;
import com.qiho.center.common.params.QueryYTOLogisticProcessParams.Result;
import com.qiho.center.common.params.QueryYTOLogisticProcessParams.Result.WaybillCode;
import com.qiho.center.common.support.BizEventBus;
import com.qiho.center.common.support.SequenceNoBuilder;
import com.qiho.center.common.util.AppLogUtil;
import com.qiho.center.common.util.YTOClientUtil;

@Service
public class LogisticsOrderServiceImpl implements LogisticsOrderService {

    private static final Logger   LOG = LoggerFactory.getLogger(LogisticsOrderServiceImpl.class);

    @Autowired
    private QihoLogisticsOrderDAO qihoLogisticsOrderDAO;
    @Autowired
    private SequenceNoBuilder     builder;
    @Autowired
    private BizEventBus           bizEventBus;
    @Autowired
    private QihoErpOrderDAO       qihoErpOrderDAO;
    @Autowired
    private QihoOrderDAO          qihoOrderDAO;
    @Autowired
    private QihoOrderSnapshotDAO  qihoOrderSnapshotDAO;
    @Autowired
    private QihoFundOrderDAO      qihoFundOrderDAO;
    @Autowired
    private FundOrderService      fundOrderService;
    @Autowired
    private ErpOrderService       erpOrderService;

    @Override
    public void initLogisticsOrder(ErpOrderDto erp, ResolveDeliverParams params) {
        // 构建物流同步对象
        ErpLogisticsSyncDto erpLogisticsSyncDto = new ErpLogisticsSyncDto();
        ExpressCompanyEnum expressCompany = params.getExpress();
        erpLogisticsSyncDto.setLogisticsCode(expressCompany.getCode());
        erpLogisticsSyncDto.setLogisticsName(expressCompany.getLogisticsName());
        erpLogisticsSyncDto.setLogisticsNo(params.getExpressCode());
        erpLogisticsSyncDto.setTid(erp.getErpId());
        LogisticsOrderEntity entity = convertToLogisticsOrderEntity(erpLogisticsSyncDto,
                                                                    BeanUtils.copy(erp, QihoErpOrderEntity.class));
        createLogisticsOrder(entity);
    }

    @Transactional(DsConstants.DATABASE_QIHO)
    public void createLogisticsOrder(LogisticsOrderEntity logisticsOrderEntity) {
        qihoLogisticsOrderDAO.insert(logisticsOrderEntity);
        qihoOrderDAO.updateOrderStatus(logisticsOrderEntity.getOrderId(), OrderStatusEnum.SEND_OUT);
        QihoOrderSnapshotEntity orderSnapshot = new QihoOrderSnapshotEntity();
        orderSnapshot.setOrderId(logisticsOrderEntity.getOrderId());
        orderSnapshot.setPostId(logisticsOrderEntity.getPostId());
        orderSnapshot.setOrderStatus(OrderStatusEnum.SEND_OUT.getCode());
        orderSnapshot.setLogisticsCode(logisticsOrderEntity.getLogisticsCode());
        orderSnapshot.setLogisticsName(logisticsOrderEntity.getLogisticsName());
        orderSnapshot.setLogisticsStatus(logisticsOrderEntity.getLogisticsStatus());
        qihoOrderSnapshotDAO.updateByOrderId(orderSnapshot);

        // 发布订单物流信息更新事件
        LogisticsInfoUpdateEvent event = BeanUtils.copy(logisticsOrderEntity, LogisticsInfoUpdateEvent.class);
        bizEventBus.post(event);
    }

    @Override
    public PagenationDto<LogisticsOrderDto> queryLogisticsOrderPage(QueryLogisticsOrderDto queryItem, Integer pageNum,
                                                                    Integer pageSize) {
        PagenationDto<LogisticsOrderDto> pagenation = new PagenationDto<>();
        LogisticsOrderEntity param = BeanUtils.copy(queryItem, LogisticsOrderEntity.class);
        int total = qihoLogisticsOrderDAO.countLogisticsOrderByParam(param, queryItem.getStartTime(),
                                                                     queryItem.getEndTime());
        List<LogisticsOrderEntity> list = null;
        if (total > 0) {
            list = qihoLogisticsOrderDAO.queryLogisticsOrderByParam(param, pageNum, pageSize, queryItem.getStartTime(),
                                                                    queryItem.getEndTime());
        }
        pagenation.setList(convertToLogisticsOrderDto(list));
        pagenation.setTotal(total);
        return pagenation;
    }

    /**
     * LogisticsOrderEntity转LogisticsOrderDto
     *
     * @author chencheng
     * @param list
     * @return
     */
    private List<LogisticsOrderDto> convertToLogisticsOrderDto(List<LogisticsOrderEntity> list) {
        List<LogisticsOrderDto> logisticsOrderDtoList = Lists.newArrayList();
        if (CollectionUtils.isEmpty(list)) {
            return logisticsOrderDtoList;
        }
        for (LogisticsOrderEntity entity : list) {
            LogisticsOrderDto dto = BeanUtils.copy(entity, LogisticsOrderDto.class);
            logisticsOrderDtoList.add(dto);
        }
        return logisticsOrderDtoList;
    }

    @Override
    @Transactional(DsConstants.DATABASE_QIHO)
    public Integer updateBylogisticsId(String logisticsId, String orderStatus) {
        LogisticsOrderDto dto = findByLogisticsId(logisticsId);
        // 查询不到，或者已签收
        if (dto == null || StringUtils.equals(dto.getOrderStatus(), LogisticsOrderStatusEnum.RETURN.getCode())
            || StringUtils.equals(dto.getOrderStatus(), LogisticsOrderStatusEnum.SUCCESS.getCode())) {
            AppLogUtil.info(LOG, "物流子订单已签收，或查询不到。无法修改订单状态,logisticsId={}", logisticsId);
            return 0;
        }
        String logisticsStatus = StringUtils.equals(orderStatus, LogisticsOrderStatusEnum.SUCCESS.getCode()) ? YTOLogisticsStatusEnum.SIGNED.getDesc() : YTOLogisticsStatusEnum.FAILED.getDesc();
        int success = qihoLogisticsOrderDAO.updateLogisticsOrderStatus(logisticsId, orderStatus, logisticsStatus);
        updateOrderInfo(dto, logisticsStatus, orderStatus);
        return success;
    }

    /**
     * @author zhanglihui
     * @param logisticsStatus
     * @param orderStatus
     */
    private void updateOrderInfo(LogisticsOrderDto logisticsOrderDto, String logisticsStatus, String orderStatus) {
        switch (LogisticsOrderStatusEnum.getByCode(orderStatus)) {
            case SUCCESS:
                updateToSuccess(logisticsOrderDto.getOrderId(), logisticsStatus);
                break;
            case RETURN:
            case FAILED:
                updateToFailed(logisticsOrderDto.getOrderId(), logisticsStatus, orderStatus);
                break;
            default:
                AppLogUtil.info(LOG, "不需要关心的物流子订单状态{}", orderStatus);
        }
        if (StringUtils.equals(orderStatus, LogisticsOrderStatusEnum.RETURN.getCode())) {
            //在线支付支付退件成功自动退款
            returnAmount(logisticsOrderDto);
            // 退货成功不需要发送短信
            return;
        }
        // 发布物流更新事件
        LogisticsUpdateEvent event = new LogisticsUpdateEvent();
        event.setLogisticsStatus(YTOLogisticsStatusEnum.getByName(logisticsStatus));
        event.setOrderId(logisticsOrderDto.getOrderId());
        if(StringUtils.equals(YTOLogisticsStatusEnum.SENT_SCAN.getDesc(),logisticsStatus)){
            event.setDeliveryMsg(logisticsOrderDto.getDeliveryMsg());
        }
        bizEventBus.post(event);
    }

    /**
     * @see com.qiho.center.biz.service.LogisticsOrderService#findByOrderId(java.lang.String)
     */
    @Override
    public List<LogisticsOrderDto> findByOrderId(String orderId) {
        return BeanUtils.copyList(qihoLogisticsOrderDAO.findByOrderId(orderId), LogisticsOrderDto.class);
    }

    private List<ErpLogisticsSyncAckDto> convertToListReurn(Map<String, ErpLogisticsOrderParams> errorCodeMap) {
        List<ErpLogisticsSyncAckDto> listReurn = new ArrayList<>();
        for (Entry<String, ErpLogisticsOrderParams> errorMap : errorCodeMap.entrySet()) {
            ErpLogisticsOrderParams erpLogisticsOrderParams = errorMap.getValue();
            listReurn.add(new ErpLogisticsSyncAckDto(erpLogisticsOrderParams.getRecId(),
                                                     erpLogisticsOrderParams.getErpErrorCodeReturnEnum()));
        }
        return listReurn;
    }

    private LogisticsOrderEntity convertToLogisticsOrderEntity(ErpLogisticsSyncDto erpLogisticsSyncDto,
                                                               QihoErpOrderEntity erpOrder) {
        LogisticsOrderEntity entity = new LogisticsOrderEntity();
        entity.setErpId(erpLogisticsSyncDto.getTid());
        entity.setPostId(erpLogisticsSyncDto.getLogisticsNo());
        entity.setLogisticsCode(erpLogisticsSyncDto.getLogisticsCode());
        entity.setLogisticsName(erpLogisticsSyncDto.getLogisticsName());
        entity.setLogisticsStatus(YTOLogisticsStatusEnum.ACCEPT.getDesc());
        entity.setOrderStatus(LogisticsOrderStatusEnum.UNDER_WAY.getCode());
        entity.setConsumerName(erpOrder.getConsumerName());
        entity.setItemId(erpOrder.getItemId());
        entity.setItemName(erpOrder.getItemName());
        entity.setItemNo(erpOrder.getItemNo());
        entity.setLogisticsId(builder.createSeqNo(SeqBizTypeEnum.LOGISTICS, erpOrder.getItemId()));
        entity.setMobile(erpOrder.getMobile());
        entity.setOrderId(erpOrder.getOrderId());
        entity.setSkuNo(erpOrder.getSkuNo());
        return entity;
    }

    @Override
    @Transactional(DsConstants.DATABASE_QIHO)
    public Integer update(LogisticsOrderDto logisticsOrderDto) {
        LogisticsOrderDto dto = findByPostIdCode(logisticsOrderDto.getPostId(), logisticsOrderDto.getLogisticsCode());
        int success = 0;
        // 查询不到，或者已签收(客户签收或仓库签收)
        if (dto == null || StringUtils.equals(dto.getOrderStatus(), LogisticsOrderStatusEnum.RETURN.getCode())
            || StringUtils.equals(dto.getOrderStatus(), LogisticsOrderStatusEnum.SUCCESS.getCode())) {
            AppLogUtil.info(LOG, "物流子订单已签收，或查询不到。无法修改订单状态,postId={}", logisticsOrderDto.getPostId());
            return success;
        }
        if (StringUtils.equals(dto.getOrderStatus(), LogisticsOrderStatusEnum.FAILED.getCode())
            && StringUtils.equals(logisticsOrderDto.getOrderStatus(), LogisticsOrderStatusEnum.UNDER_WAY.getCode())) {
            // 物流子订单状态为签收失败，只能更新为签收成功或退件成功
            return success;
        }
        LogisticsOrderEntity param = BeanUtils.copy(logisticsOrderDto, LogisticsOrderEntity.class);
        success = qihoLogisticsOrderDAO.update(param);
        //派件状态
        if(StringUtils.equals(YTOLogisticsStatusEnum.SENT_SCAN.getDesc(),logisticsOrderDto.getLogisticsStatus())){
            dto.setDeliveryMsg(logisticsOrderDto.getDeliveryMsg());
        }
        updateOrderInfo(dto, logisticsOrderDto.getLogisticsStatus(), logisticsOrderDto.getOrderStatus());
        return success;
    }

    /**
     * 退件成功，ERP子订单为发货类型&发货状态时，交易金额自动退还
     */
    private void returnAmount(LogisticsOrderDto logisticsOrderDto) {
        ErpOrderDto erpOrderDto = erpOrderService.findErpOrderByErpId(logisticsOrderDto.getErpId());
        if (ErpTypeEnum.SEND.getCode().equals(erpOrderDto.getErpType())) {
            String orderId = logisticsOrderDto.getOrderId();
            List<FundOrderDto> fundOrderDtoList = fundOrderService.findByOrderIdAndBizType(orderId,
                                                                                           FundBizTypeEnum.PAY.getCode());
            // 一个主订单只会有一笔付款资金流水，对应一次发货类型ERP订单
            FundOrderDto fundOrder = fundOrderDtoList.get(0);
            // 付款成功的在线支付订单需要发起退款
            if (!StringUtils.equals(PayTypeEnum.COD.getCode(), fundOrder.getPayType())
                && StringUtils.equals(FundStatusEnum.SUCCESS.getCode(), fundOrder.getFundStatus())) {
                fundOrderService.refund(orderId, fundOrder.getPayType(), fundOrder.getAmt());
            }
        }
    }

    /**
     * @author zhanglihui
     * @param orderId
     * @param logisticsStatus
     */
    private void updateToSuccess(String orderId, String logisticsStatus) {
        QihoOrderEntity updateOrder = new QihoOrderEntity();
        updateOrder.setOrderId(orderId);
        updateOrder.setOrderStatus(OrderStatusEnum.SUCCESS.getCode());
        updateOrder.setRemark(LogisticsOrderStatusEnum.SUCCESS.getDesc());
        qihoOrderDAO.updateByOrderId(updateOrder);

        QihoOrderSnapshotEntity orderSnapshot = new QihoOrderSnapshotEntity();
        orderSnapshot.setOrderId(orderId);
        orderSnapshot.setLogisticsStatus(logisticsStatus);
        orderSnapshot.setOrderStatus(OrderStatusEnum.SUCCESS.getCode());
        orderSnapshot.setFundStatus(FundStatusEnum.SUCCESS.getCode());
        orderSnapshot.setRemark(LogisticsOrderStatusEnum.SUCCESS.getDesc());
        qihoOrderSnapshotDAO.updateByOrderId(orderSnapshot);
        List<QihoFundOrderEntity> list = qihoFundOrderDAO.findByOrderIdAndBizType(orderId,
                                                                                  FundBizTypeEnum.PAY.getCode());
        if (CollectionUtils.isNotEmpty(list)) {
            QihoFundOrderEntity fundOrder = list.get(0);
            if (StringUtils.equals(fundOrder.getPayType(), PayTypeEnum.COD.getCode())) {
                qihoFundOrderDAO.updateFundOrderStatus(fundOrder.getFundId(), FundStatusEnum.SUCCESS.getCode());
            }
        }
    }

    /**
     * @author zhanglihui
     */
    private void updateToFailed(String orderId, String logisticsStatus, String logisticsOrderStatus) {
        LogisticsOrderStatusEnum logisticsOrderStatusEnum = LogisticsOrderStatusEnum.getByCode(logisticsOrderStatus);
        QihoOrderEntity updateOrder = new QihoOrderEntity();
        updateOrder.setOrderId(orderId);
        updateOrder.setOrderStatus(OrderStatusEnum.CLOSED.getCode());
        updateOrder.setRemark(logisticsOrderStatusEnum.getDesc());
        qihoOrderDAO.updateByOrderId(updateOrder);

        QihoOrderSnapshotEntity orderSnapshot = new QihoOrderSnapshotEntity();
        orderSnapshot.setOrderId(orderId);
        orderSnapshot.setLogisticsStatus(logisticsStatus);
        orderSnapshot.setOrderStatus(OrderStatusEnum.CLOSED.getCode());
        orderSnapshot.setRemark(logisticsOrderStatusEnum.getDesc());

        List<QihoFundOrderEntity> list = qihoFundOrderDAO.findByOrderIdAndBizType(orderId,
                                                                                  FundBizTypeEnum.PAY.getCode());
        if (CollectionUtils.isNotEmpty(list)) {
            QihoFundOrderEntity fundOrder = list.get(0);
            if (StringUtils.equals(fundOrder.getPayType(), PayTypeEnum.COD.getCode())) {
                qihoFundOrderDAO.updateFundOrderStatus(fundOrder.getFundId(), FundStatusEnum.FAILED.getCode());
                orderSnapshot.setFundStatus(FundStatusEnum.FAILED.getCode());
            }
        }
        qihoOrderSnapshotDAO.updateByOrderId(orderSnapshot);
    }

    /**
     * @see com.qiho.center.biz.service.LogisticsOrderService#createLogisticsOrderBatch(java.util.List)
     */
    @Override
    public List<ErpLogisticsSyncAckDto> createLogisticsOrderBatch(List<ErpLogisticsSyncDto> list) {
        List<ErpLogisticsSyncAckDto> returnList = Lists.newArrayList();
        if (CollectionUtils.isEmpty(list)) {
            return returnList;
        }
        // 订单状态map
        Map<String, ErpLogisticsOrderParams> errorCodeMap = new HashMap<>(list.size());
        Map<String, ErpLogisticsSyncDto> syncMap = Maps.newHashMap();
        List<String> erpIds = Lists.newArrayList();
        for (ErpLogisticsSyncDto erpLogisticsSyncDto : list) {
            // 先置为订单不存在状态，如果数据已存在或更新成功会更改
            errorCodeMap.put(erpLogisticsSyncDto.getTid(),
                             new ErpLogisticsOrderParams(erpLogisticsSyncDto.getRecId(),
                                                         ErpErrorCodeReturnEnum.ORDER_CODE_NON_ERROR));
            erpIds.add(erpLogisticsSyncDto.getTid());
            syncMap.put(erpLogisticsSyncDto.getTid(), erpLogisticsSyncDto);
        }

        List<QihoErpOrderEntity> erpOrderList = qihoErpOrderDAO.findByErpIds(erpIds);
        if (CollectionUtils.isEmpty(erpOrderList)) {
            return convertToListReurn(errorCodeMap);
        }
        // 查询已同步的物流子订单
        List<LogisticsOrderEntity> existList = qihoLogisticsOrderDAO.findByErpIds(erpIds);
        List<String> syncErpIds = Lists.newArrayList();
        if (CollectionUtils.isNotEmpty(existList)) {
            for (LogisticsOrderEntity logisticsOrder : existList) {
                syncErpIds.add(logisticsOrder.getErpId());
            }
        }
        for (QihoErpOrderEntity erpOrder : erpOrderList) {
            errorCodeMap.get(erpOrder.getErpId()).setErpErrorCodeReturnEnum(ErpErrorCodeReturnEnum.SUCCESS);
            if (syncErpIds.contains(erpOrder.getErpId())) {
                // 已经同步的不再创建物流子订单
                continue;
            }
            LogisticsOrderEntity logisticsOrderEntity = convertToLogisticsOrderEntity(syncMap.get(erpOrder.getErpId()),
                                                                                      erpOrder);
            // 创建物流子订单、创建付款子订单、更新订单快照、发送发货短信
            createLogisticsOrder(logisticsOrderEntity);
        }
        return convertToListReurn(errorCodeMap);
    }

    @Override
    public LogisticsOrderDto findByPostIdCode(String postId, String logisticsCode) {
        return BeanUtils.copy(qihoLogisticsOrderDAO.findByPostIdCode(postId, logisticsCode), LogisticsOrderDto.class);
    }

    @Override
    public LogisticsOrderDto findByLogisticsId(String logisticsId) {
        return BeanUtils.copy(qihoLogisticsOrderDAO.findByLogisticsId(logisticsId), LogisticsOrderDto.class);
    }

    @Override
    public LogisticsOrderDto findByErpId(String erpId) {
        List<String> erpids = Lists.newArrayList();
        erpids.add(erpId);
        List<LogisticsOrderEntity> list = qihoLogisticsOrderDAO.findByErpIds(erpids);
        return list.size() > 0 ? BeanUtils.copy(list.get(0), LogisticsOrderDto.class) : null;
    }

    /**
     * @author zhanglihui
     * @param logisticsOrderStatus
     */
    public void syncLogisticsStatus(String logisticsOrderStatus) {
        LogisticsOrderEntity param = new LogisticsOrderEntity();
        param.setOrderStatus(logisticsOrderStatus);
        int pageSize = 100;
        int pageNum = 1;
        while (true) {
            List<LogisticsOrderEntity> list = qihoLogisticsOrderDAO.queryLogisticsOrderByParam(param, pageNum,
                                                                                               pageSize, null, null);
            if (CollectionUtils.isEmpty(list)) {
                break;
            }
            doSync(list);
            pageNum++;
        }
    }

    /**
     * @author zhanglihui
     * @param list
     */
    public void doSync(List<LogisticsOrderEntity> list) {
        for (LogisticsOrderEntity entity : list) {
            if(!StringUtils.equals(entity.getLogisticsCode(), LogisticsNameEnum.YTO.getCode())){
                continue;
            }
            QueryYTOLogisticProcessParams queryXml = new QueryYTOLogisticProcessParams(
                                                                                       new Result(
                                                                                                  new WaybillCode(
                                                                                                                  entity.getPostId())));
            // 请求查询走件流程
            String logisticsString = YTOClientUtil.execute(YTOClientUtil.getQueryLogisticsUrl(), queryXml);
            String[] refuseRemark = { "签收人: 退件" };
            if (StringUtils.indexOfAny(logisticsString, refuseRemark) > 0
                && StringUtils.contains(logisticsString, "【浙江省杭州市富阳市公司】 派件人")) {
                // if (StringUtils.equals(entity.getOrderStatus(), LogisticsOrderStatusEnum.RETURN.getCode())) {
                // continue;
                // }
                entity.setLogisticsStatus(YTOLogisticsStatusEnum.FAILED.getDesc());
                entity.setOrderStatus(LogisticsOrderStatusEnum.RETURN.getCode());
            } else if (StringUtils.contains(logisticsString, "已签收 感谢使用圆通速递，期待再次为您服务")) {
                // if (StringUtils.equals(entity.getOrderStatus(), LogisticsOrderStatusEnum.SUCCESS.getCode())) {
                // continue;
                // }
                entity.setLogisticsStatus(YTOLogisticsStatusEnum.SIGNED.getDesc());
                entity.setOrderStatus(LogisticsOrderStatusEnum.SUCCESS.getCode());
            } else {
                continue;
            }
            LogisticsOrderDto dto = BeanUtils.copy(entity, LogisticsOrderDto.class);
            update(dto);
        }
    }

    /**
     * @see com.qiho.center.biz.service.LogisticsOrderService#queryListToFix(int, int,long)
     */
    @Override
    public List<LogisticsOrderEntity> queryListToFix(int pageNum, int pageSize, long minId) {
        return qihoLogisticsOrderDAO.queryListToFix(pageNum, pageSize, minId);
    }

}
