/**
 * Project Name:qiho-center-biz
 * File Name:SkuPropertyServiceImpl.java
 * Package Name:com.qiho.center.biz.service.impl
 * Date:2017年5月20日下午3:47:17
 * Copyright (c) 2017, duiba.com.cn All Rights Reserved.
 *
*/

package com.qiho.center.biz.service.impl;

import java.util.List;
import java.util.Map;

import org.apache.commons.collections.CollectionUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import cn.com.duiba.wolf.utils.BeanUtils;

import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.qiho.center.api.dto.SkuPropertyDto;
import com.qiho.center.api.dto.SkuPropertyValueDto;
import com.qiho.center.biz.service.SkuPropertyService;
import com.qiho.center.common.constant.DsConstants;
import com.qiho.center.common.dao.QihoSkuPropertyNameDAO;
import com.qiho.center.common.dao.QihoSkuPropertyValueDAO;
import com.qiho.center.common.entity.item.QihoSkuPropertyNameEntity;
import com.qiho.center.common.entity.item.QihoSkuPropertyValueEntity;


/**
 * ClassName:SkuPropertyServiceImpl <br/>
 * 商品属性服务实现类. <br/>
 * Date:     2017年5月20日 下午3:47:17 <br/>
 * @author   zhanglihui
 * @version  
 * @since    JDK 1.6
 * @see 	 
 */
@Service
public class SkuPropertyServiceImpl implements SkuPropertyService {
    @Autowired
    private QihoSkuPropertyNameDAO    qihoSkuPropertyNameDAO;

    @Autowired
    private QihoSkuPropertyValueDAO   qihoSkuPropertyValueDAO;
    
    /**
     * @see com.qiho.center.biz.service.SkuPropertyService#getSkuPropertyByItemId(java.lang.Long)
     */
    @Override
    public List<SkuPropertyDto> getSkuPropertyByItemId(Long itemId) {
        // 获取商品属性名
        List<QihoSkuPropertyNameEntity> propertyNameEntityList = qihoSkuPropertyNameDAO.findByItemId(itemId);
        List<SkuPropertyDto> skuPropertyList = Lists.newArrayList();
        if(CollectionUtils.isEmpty(propertyNameEntityList)){
            return skuPropertyList;
        }
        List<Long> propertyNameIds = Lists.newArrayList();
        for(QihoSkuPropertyNameEntity propertyName : propertyNameEntityList){
            propertyNameIds.add(propertyName.getId());
        }
        // 根据属性名获取对应的所有属性值
        List<QihoSkuPropertyValueEntity> propertyValueEntityList = qihoSkuPropertyValueDAO.findByPropertyNameIds(propertyNameIds);
        // 根据属性名ID对属性值进行分组
        Map<Long, List<SkuPropertyValueDto>> propertyValueMap = Maps.newHashMap();
        for(QihoSkuPropertyValueEntity value : propertyValueEntityList){
            SkuPropertyValueDto dto = BeanUtils.copy(value, SkuPropertyValueDto.class);
            Long propertyNameId = value.getPropertyNameId();
            List<SkuPropertyValueDto> list = propertyValueMap.get(propertyNameId);
            if(CollectionUtils.isEmpty(list)){
                propertyValueMap.put(propertyNameId, Lists.newArrayList(dto));
            }else{
                list.add(dto);
            }
        }
        // 组装商品属性
        for(QihoSkuPropertyNameEntity propertyName : propertyNameEntityList){
            SkuPropertyDto dto = BeanUtils.copy(propertyName, SkuPropertyDto.class);
            dto.setPropertyValueList(propertyValueMap.get(propertyName.getId()));
            skuPropertyList.add(dto);
        }
        return skuPropertyList;
    }

    /**
     * @see com.qiho.center.biz.service.SkuPropertyService#updateItemId(java.util.List, java.lang.Long)
     */
    @Override
    public void updateItemId(List<SkuPropertyDto> propertyList, Long itemId) {
        if(CollectionUtils.isEmpty(propertyList)){
            return;
        }
        List<Long> propertyNameIds = Lists.newArrayList();
        for(SkuPropertyDto dto : propertyList){
            propertyNameIds.add(dto.getId());
        }
        qihoSkuPropertyNameDAO.updateItemIdByIds(propertyNameIds, itemId);
    }

    /**
     * @see com.qiho.center.biz.service.SkuPropertyService#createPropertyNames(java.util.List)
     */
    @Override
    public Map<Long, String> createPropertyNames(List<String> propertyNames) {
        Map<Long, String> map = Maps.newHashMap();
        if(CollectionUtils.isEmpty(propertyNames)){
            return map;
        }
        List<QihoSkuPropertyNameEntity> list = Lists.newArrayList();
        for(String name : propertyNames){
            QihoSkuPropertyNameEntity entity = new QihoSkuPropertyNameEntity();
            entity.setPropertyName(name);
            entity.setNameEnable(false);
            list.add(entity);
        }
        qihoSkuPropertyNameDAO.insertBatch(list);
        for(QihoSkuPropertyNameEntity entity : list){
            map.put(entity.getId(), entity.getPropertyName());
        }
        return map;
    }

    /**
     * @see com.qiho.center.biz.service.SkuPropertyService#createPropertyValues(java.util.List, java.lang.Long)
     */
    @Override
    public Map<Long, String> createPropertyValues(List<SkuPropertyValueDto> propertyValues) {
        Map<Long, String> map = Maps.newHashMap();
        if(CollectionUtils.isEmpty(propertyValues)){
            return map;
        }
        List<QihoSkuPropertyValueEntity> list = Lists.newArrayList();
        for(SkuPropertyValueDto propertyValue : propertyValues){
            QihoSkuPropertyValueEntity entity = new QihoSkuPropertyValueEntity();
            entity.setPropertyNameId(propertyValue.getPropertyNameId());
            entity.setPropertyValue(propertyValue.getPropertyValue());
            entity.setValueEnable(false);
            list.add(entity);
        }
        qihoSkuPropertyValueDAO.insertBatch(list);
        for(QihoSkuPropertyValueEntity entity : list){
            map.put(entity.getId(), entity.getPropertyValue());
        }
        return map;
    }

    /**
     * @see com.qiho.center.biz.service.SkuPropertyService#enableProperty(java.util.List)
     */
    @Override
    @Transactional(DsConstants.DATABASE_QIHO)
    public void enableProperty(List<SkuPropertyDto> propertyList, Long itemId) {
        List<Long> propertyNameIds = Lists.newArrayList();
        List<Long> propertyValueIds = Lists.newArrayList();
        for(SkuPropertyDto property : propertyList){
            propertyNameIds.add(property.getId());
            for(SkuPropertyValueDto value : property.getPropertyValueList()){
                propertyValueIds.add(value.getId());
            }
        }
        List<QihoSkuPropertyNameEntity> dbNameList = qihoSkuPropertyNameDAO.findByItemId(itemId);
        List<Long> dbPropertyNameIds = Lists.newArrayList();
        if(CollectionUtils.isNotEmpty(dbNameList)){
            List<Long> delNameIds = Lists.newArrayList();
            for(QihoSkuPropertyNameEntity entity : dbNameList){
                dbPropertyNameIds.add(entity.getId());
                delNameIds.add(entity.getId());
            }
            delNameIds.removeAll(propertyNameIds);
            if(CollectionUtils.isNotEmpty(delNameIds)){
                qihoSkuPropertyNameDAO.deleteByIds(delNameIds);
            }
        }
        if(CollectionUtils.isNotEmpty(propertyNameIds)){
            qihoSkuPropertyNameDAO.enablePropertyName(propertyNameIds);
        }
        List<QihoSkuPropertyValueEntity> dbValueList = Lists.newArrayList();
        if(CollectionUtils.isNotEmpty(dbPropertyNameIds)){
            dbValueList = qihoSkuPropertyValueDAO.findByPropertyNameIds(dbPropertyNameIds);
        }
        if(CollectionUtils.isNotEmpty(dbValueList)){
            List<Long> delValueIds = Lists.newArrayList();
            for(QihoSkuPropertyValueEntity entity : dbValueList){
                delValueIds.add(entity.getId());
            }
            delValueIds.removeAll(propertyValueIds);
            if(CollectionUtils.isNotEmpty(delValueIds)){
                qihoSkuPropertyValueDAO.deleteByIds(delValueIds);
            }
        }
        if(CollectionUtils.isNotEmpty(propertyValueIds)){
            qihoSkuPropertyValueDAO.enablePropertyValue(propertyValueIds);
        }
    }

    /**
     * @see com.qiho.center.biz.service.SkuPropertyService#updatePropertyValue(java.lang.Long, java.lang.String)
     */
    @Override
    public void updatePropertyValue(Long id, String propertyValue) {
        qihoSkuPropertyValueDAO.updatePropertyValue(id, propertyValue);
    }

}

