package com.qiho.center.biz.service.impl.order;

import java.util.List;
import java.util.Map;

import javax.annotation.Resource;

import com.qiho.center.common.util.*;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import cn.com.duiba.wolf.utils.BeanUtils;

import com.google.common.base.Function;
import com.google.common.base.Objects;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.google.common.eventbus.Subscribe;
import com.qiho.center.api.dto.ChannelInfoDto;
import com.qiho.center.api.dto.OrderSnapshotDto;
import com.qiho.center.api.dto.PagenationDto;
import com.qiho.center.api.enums.DeliveryEnum;
import com.qiho.center.api.enums.FundStatusEnum;
import com.qiho.center.api.enums.OrderApprovalStatusEnum;
import com.qiho.center.api.enums.OrderStatusEnum;
import com.qiho.center.api.enums.PayTypeEnum;
import com.qiho.center.api.exception.QihoException;
import com.qiho.center.api.params.OrderPageParams;
import com.qiho.center.biz.bo.BlackListBo;
import com.qiho.center.biz.engine.impl.ShotOrderEngine;
import com.qiho.center.biz.event.LogisticsInfoUpdateEvent;
import com.qiho.center.biz.event.OrderCreateEvent;
import com.qiho.center.biz.event.UpdateToCodEvent;
import com.qiho.center.biz.service.order.ChannelInfoService;
import com.qiho.center.biz.service.order.OrderSnapshotService;
import com.qiho.center.biz.service.order.SmsService;
import com.qiho.center.common.annotations.BizEventListener;
import com.qiho.center.common.constant.DomainConstantUtil;
import com.qiho.center.common.dao.QihoAnticheateDAO;
import com.qiho.center.common.dao.QihoOrderSnapshotDAO;
import com.qiho.center.common.dao.QihoShortUrlDAO;
import com.qiho.center.common.dao.QihoTemplateDAO;
import com.qiho.center.common.entity.QihoShortUrlEntity;
import com.qiho.center.common.entity.QihoTemplateEntity;
import com.qiho.center.common.entity.order.QihoAnticheateEntity;
import com.qiho.center.common.entity.order.QihoOrderSnapshotEntity;
import com.qiho.center.common.enums.SmsTemplateEnum;
import com.qiho.center.common.model.tongdun.FraudApiResponse;

/**
 * Created by liuyao on 2017/6/3.
 */
@Service
@BizEventListener
public class OrderSnapshotServiceImpl implements OrderSnapshotService {

    private static final Logger LOG = LoggerFactory.getLogger(OrderSnapshotServiceImpl.class);
    
    @Autowired
    private QihoOrderSnapshotDAO qihoOrderSnapshotDAO;
    
    @Autowired
    private QihoShortUrlDAO qihoShortUrlDAO;
    
    @Autowired
    private QihoTemplateDAO qihoTemplateDAO;
    
    @Autowired
    private SmsService smsService;
    
    @Autowired
    private BlackListBo blackListBo;

    @Resource
    private ShotOrderEngine shotOrderEngine;
    
    @Autowired
    private QihoAnticheateDAO    qihoAnticheateDAO;
    
    @Autowired
    private ChannelInfoService   channelInfoService;

    @Override
    public PagenationDto<OrderSnapshotDto> findOrderPage(OrderPageParams params) {
        PagenationDto<OrderSnapshotDto> page = new PagenationDto<>();
        List<QihoOrderSnapshotEntity> list = qihoOrderSnapshotDAO.findOrderPage(params);
        page.setTotal(qihoOrderSnapshotDAO.findOrderPageCount(params));
        DtoTransform transform = new DtoTransform();
        List<OrderSnapshotDto> dtoList = Lists.transform(list,transform);
        page.setList(dtoList);
        return page;
    }

    @Override
    public List<OrderSnapshotDto> findOrderPageForExport(OrderPageParams params) {
        List<QihoOrderSnapshotEntity> list = Lists.newArrayList();
        if(!Objects.equal(null,params.getOrderId())){
            QihoOrderSnapshotEntity order = qihoOrderSnapshotDAO.findByOrderId(params.getOrderId());
            if(order!=null){
                list.add(order);
            }
        }else{
            list.addAll(qihoOrderSnapshotDAO.exportOrderPage(params));
        }
        if(CollectionUtils.isEmpty(list)){
            return Lists.newArrayList();
        }
        List<String> orderIds = Lists.newArrayList();
        for(QihoOrderSnapshotEntity order : list){
            orderIds.add(order.getOrderId());
        }
        List<QihoAnticheateEntity> anticheatList = qihoAnticheateDAO.findByOrderIds(orderIds);
        Map<String, String> map = Maps.newHashMap();
        if(CollectionUtils.isNotEmpty(anticheatList)){
            for(QihoAnticheateEntity anticheat : anticheatList){
                String rules = map.get(anticheat.getOrderId());
                if(StringUtils.isNotBlank(rules)){
                    map.put(anticheat.getOrderId(), rules + "," +anticheat.getRule());
                }else{
                    map.put(anticheat.getOrderId(), anticheat.getRule());
                }
            }
        }
        DtoTransform transform = new DtoTransform();
        transform.setAnticheatMap(map);
        return Lists.transform(list,transform);
    }

    @Override
    public Integer findAllPageCount(OrderPageParams params) {
        if(!Objects.equal(null,params.getOrderId())){
            QihoOrderSnapshotEntity order = qihoOrderSnapshotDAO.findByOrderId(params.getOrderId());
            if(order!=null) {
                return 1;
            }else{
                return 0;
            }
        }else{
            return qihoOrderSnapshotDAO.findOrderPageCount(params);
        }
    }


    private class DtoTransform implements Function<QihoOrderSnapshotEntity,OrderSnapshotDto>{
        private Map<String, String> anticheatMap;
        
        
        /**
         * anticheatMap.
         *
         * @param   anticheatMap    the anticheatMap to set
         */
        public void setAnticheatMap(Map<String, String> anticheatMap) {
            this.anticheatMap = anticheatMap;
        }

        @Override
        public OrderSnapshotDto apply(QihoOrderSnapshotEntity input) {
            if(input==null){
                return null;
            }
            OrderSnapshotDto dto = new OrderSnapshotDto();
            BeanUtils.copy(input,dto);
            dto.setFundStatusEnum(FundStatusEnum.getEnumByCode(input.getFundStatus()));
            dto.setOrderStatusEnum(OrderStatusEnum.getByCode(input.getOrderStatus()));
            dto.setDeliveryEnum(DeliveryEnum.getByCode(input.getDelivery()));
            if(anticheatMap != null){
                dto.setAnticheatRules(anticheatMap.get(input.getOrderId()));
            }
            return dto;
        }
    }

    @Override
    public Integer exportOrderPageCount(OrderPageParams params) {
        if(!Objects.equal(null,params.getOrderId())){
            QihoOrderSnapshotEntity order = qihoOrderSnapshotDAO.findByOrderId(params.getOrderId());
            if(order!=null) {
                return 1;
            }else{
                return 0;
            }
        }else{
            return qihoOrderSnapshotDAO.exportOrderPageCount(params);
        }
    }

    @Override
    public Integer updateOrderStatus(String orderId, String remark, OrderStatusEnum orderStatus, OrderApprovalStatusEnum orderApprovalStatusEnum) {
        return qihoOrderSnapshotDAO.updateOrderStatus(orderId, remark, orderStatus, orderApprovalStatusEnum,null);
    }

    @Override
    public OrderSnapshotDto findByOrderId(String orderId) {
        QihoOrderSnapshotEntity entity = qihoOrderSnapshotDAO.findByOrderId(orderId);
        if(entity == null){
            return null;
        }
        OrderSnapshotDto dto = BeanUtils.copy(entity, OrderSnapshotDto.class);
        dto.setOrderStatusEnum(OrderStatusEnum.getByCode(entity.getOrderStatus()));
        dto.setDeliveryEnum(DeliveryEnum.getByCode(entity.getDelivery()));
        dto.setFundStatusEnum(FundStatusEnum.getEnumByCode(entity.getFundStatus()));
        return dto;
    }
    
    /**
     * 监听订单创建事件
     * @author zhanglihui
     * @param orderCreateEvent
     */
    @Subscribe
    public void createOrderListener(OrderCreateEvent orderCreateEvent){
        OrderSnapshotDto orderSnapshotDto = orderCreateEvent.getOrderSnapshot();
        String orderId = orderSnapshotDto.getOrderId();
        // 保存订单详情页短链接
        String shortUrl = saveShortUrl(orderId);
        if(StringUtils.equals(orderSnapshotDto.getPayType(), PayTypeEnum.COD.getCode())){
            // 防作弊规则扫描
            shotOrderEngine.process(orderSnapshotDto);
            // 发送下单成功短信
            sendMessage(orderSnapshotDto.getMobile(), orderSnapshotDto.getItemShortName(), shortUrl);
        }
        // 获取设备指纹ID
        updateDeviceId(orderId, orderCreateEvent.getTokenId());
    }

    /**
     * 监听付款方式更新成货到付款事件
     * @author zhanglihui
     * @param event
     */
    @Subscribe
    public void updateToCodListener(UpdateToCodEvent event){
        QihoOrderSnapshotEntity order = qihoOrderSnapshotDAO.findByOrderId(event.getOrderId());
        // 防作弊规则扫描
        OrderSnapshotDto orderSnapshotDto = BeanUtils.copy(order,OrderSnapshotDto.class);
        shotOrderEngine.process(orderSnapshotDto);
        String shortUrl = ShortUrlUtil.getShortUrlByOrderId(event.getOrderId());
        // 发送下单成功短信
        sendMessage(order.getMobile(), order.getItemShortName(), shortUrl);
    }
    /**
     *
     * @author zhanglihui
     * @param orderId
     * @return
     */
    private String saveShortUrl(String orderId) {
        
        String qihoWebUrl = DomainConstantUtil.getQihoWebUrl();
        // 手机端域名
        String domain = StringUtils.substring(qihoWebUrl, StringUtils.indexOf(qihoWebUrl, "//")+2);
        // 订单详情页长链接完整地址
        String longUrl = qihoWebUrl + "/order/detail?orderId="+orderId;
        // 短链接地址
        int section = Integer.parseInt(StringUtils.substring(orderId, 15, 19));
        String shortUrl = ShortUrlUtil.generateShortUrl(longUrl, section);
        // 保存长链接和短链接的映射关系
        QihoShortUrlEntity qihoShortUrlEntity = new QihoShortUrlEntity();
        qihoShortUrlEntity.setShortUrl(shortUrl);
        qihoShortUrlEntity.setLongUrl(longUrl);
        qihoShortUrlDAO.insert(qihoShortUrlEntity);
        return domain+"/"+shortUrl;
    }

    /**
     *
     * @author zhanglihui
     * @param tokenId
     * @param string 
     */
    private void updateDeviceId(String orderId, String tokenId) {
        FraudApiResponse fraudApiResponse = TongdunUtil.getRiskInfo(tokenId);
        String tongdunId = fraudApiResponse.getDeviceId();
        ChannelInfoDto channelInfo = new ChannelInfoDto();
        channelInfo.setOrderId(orderId);
        channelInfo.addDeviceId(ChannelInfoDto.TONGDUN_DEVICE_ID, tongdunId);
        
        String maxentId = MaxentUtil.getDeviceId(tokenId);
        channelInfo.addDeviceId(ChannelInfoDto.MAXENT_DEVICE_ID, maxentId);
        channelInfoService.updateByOrderId(channelInfo);
    }

    /**
     *
     * @author zhanglihui
     * @param order
     */
    private void sendMessage(String mobile, String itemShortName, String orderDetailUrl) {
        
        // 用户下单成功短信通知
        QihoTemplateEntity template = qihoTemplateDAO.findByCode(SmsTemplateEnum.ORDER_SUCCESS.getCode());
        if(template.getTemplateEnable()){
            Map<String, String> context = Maps.newHashMap();
            context.put("orderDetailUrl", orderDetailUrl);
            context.put("itemName", itemShortName);
            String text = SimpleTemplateUtil.render(template.getTemplateContext(), context);
            smsService.singleSend(text, mobile);
        }

        try {
            Thread.sleep(500l);
        } catch (InterruptedException e) {
            AppLogUtil.error(LOG,"发送短信线程中断",e);
            Thread.currentThread().interrupt();
        }
        // 发送下单后温馨提示
        template = qihoTemplateDAO.findByCode(SmsTemplateEnum.ORDER_NOTIFY.getCode());
        if(template.getTemplateEnable()){
            smsService.singleSend(template.getTemplateContext(), mobile);
        }
    }

    /**
     * 物流子订单更新事件
     * @param event
     */
    @Subscribe
    public void logisticsInfoUpdateListener(LogisticsInfoUpdateEvent event) {
        
        // 发送发货短信
        String postId = event.getPostId();
        String mobile = event.getMobile();
        
        QihoTemplateEntity template = qihoTemplateDAO.findByCode(SmsTemplateEnum.ORDER_SEND.getCode());
        if(template.getTemplateEnable()){
            Map<String, String> context = Maps.newHashMap();
            context.put("postId", postId);
            context.put("orderDetailUrl", ShortUrlUtil.getShortUrlByOrderId(event.getOrderId()));
            String text = SimpleTemplateUtil.render(template.getTemplateContext(), context);
            smsService.singleSend(text, mobile);
        }
    }

    /**
     * @see com.qiho.center.biz.service.order.OrderSnapshotService#findByMobile(java.lang.String)
     */
    @Override
    public List<OrderSnapshotDto> findByMobile(String mobile, List<String> statusList) {
        List<QihoOrderSnapshotEntity> list = qihoOrderSnapshotDAO.findByMobile(mobile, statusList);
        List<OrderSnapshotDto> orderList = Lists.newArrayList();
        if(CollectionUtils.isEmpty(list)){
            return orderList;
        }
        for(QihoOrderSnapshotEntity input : list){
            OrderSnapshotDto dto = new OrderSnapshotDto();
            BeanUtils.copy(input,dto);
            dto.setFundStatusEnum(FundStatusEnum.getEnumByCode(input.getFundStatus()));
            dto.setOrderStatusEnum(OrderStatusEnum.getByCode(input.getOrderStatus()));
            dto.setDeliveryEnum(DeliveryEnum.getByCode(input.getDelivery()));
            orderList.add(dto);
        }
        return orderList;
    }
}
