/**
 * Project Name:qiho-center-biz File Name:SmsServiceImpl.java Package Name:com.qiho.center.biz.service.impl.order
 * Date:2017年6月3日下午12:32:06 Copyright (c) 2017, duiba.com.cn All Rights Reserved.
 */

package com.qiho.center.biz.service.impl.order;

import java.util.Map;

import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Service;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import com.google.common.base.Objects;
import com.google.common.collect.Maps;
import com.google.common.eventbus.Subscribe;
import com.qiho.center.api.enums.PayTypeEnum;
import com.qiho.center.api.enums.YTOLogisticsStatusEnum;
import com.qiho.center.biz.event.FundInfoUpdateEvent;
import com.qiho.center.biz.event.LogisticsUpdateEvent;
import com.qiho.center.biz.service.logistics.YtoQueryProcessor;
import com.qiho.center.biz.service.order.SmsService;
import com.qiho.center.common.annotations.BizEventListener;
import com.qiho.center.common.constant.Environment;
import com.qiho.center.common.dao.QihoOrderDAO;
import com.qiho.center.common.dao.QihoOrderSnapshotDAO;
import com.qiho.center.common.dao.QihoTemplateDAO;
import com.qiho.center.common.entity.QihoTemplateEntity;
import com.qiho.center.common.entity.order.QihoOrderSnapshotEntity;
import com.qiho.center.common.enums.SmsTemplateEnum;
import com.qiho.center.common.util.AppLogUtil;
import com.qiho.center.common.util.HttpClientUtil;
import com.qiho.center.common.util.ShortUrlUtil;
import com.qiho.center.common.util.SimpleTemplateUtil;

/**
 * ClassName:SmsServiceImpl <br/>
 * 短信服务实现类. <br/>
 * Date: 2017年6月3日 下午12:32:06 <br/>
 * 
 * @author zhanglihui
 * @version
 * @since JDK 1.6
 * @see
 */
@Service
@BizEventListener
public class SmsServiceImpl implements SmsService {

    private static final Logger  LOG = LoggerFactory.getLogger(SmsServiceImpl.class);
    @Value("${yunpian.sms.url}")
    private String               smsUrl;

    @Value("${yunpian.apikey}")
    private String               apikey;

    @Autowired
    private QihoOrderDAO         qihoOrderDAO;

    @Autowired
    private QihoOrderSnapshotDAO qihoOrderSnapshotDAO;

    @Autowired
    private QihoTemplateDAO      qihoTemplateDAO;

    @Autowired
    private YtoQueryProcessor    ytoQueryProcessor;

    /**
     * @see com.qiho.center.biz.service.order.SmsService#sendSmsCode(java.lang.String, java.lang.String)
     */
    @Override
    public boolean sendSmsCode(String mobile, String smsCode) {
        Map<String, String> context = Maps.newHashMap();
        context.put("code", smsCode);
        return sendMessage(SmsTemplateEnum.SMS_CODE, context, mobile);
    }

    @Override
    public boolean singleSend(String text, String mobile) {
        if (Environment.isDaily() || Environment.isDev()) {
            AppLogUtil.warn(LOG, "测试环境不发送短信, text={}, mobile={}", text, mobile);
            return true;
        }
        Map<String, String> params = Maps.newHashMap();
        params.put("apikey", apikey);
        params.put("text", text);
        params.put("mobile", mobile);
        Integer returnCode = -1;
        try {
            String resp = HttpClientUtil.sendPost(smsUrl, params);
            if (StringUtils.isNotBlank(resp)) {
                JSONObject json = JSON.parseObject(resp);
                returnCode = json.getInteger("code");
                if (returnCode != 0) {
                    AppLogUtil.warn(LOG, "发送短信异常， mobile={}, text={}, resp={}", mobile, text, resp);
                }
            }
        } catch (Exception e) {
            LOG.warn("发送短信失败：mobile=" + mobile, e);
        }

        return returnCode == 0;
    }

    /**
     * 监听物流状态更新事件
     * 
     * @author zhanglihui
     * @param event
     */
    @Subscribe
    public void updateLogisticsListener(LogisticsUpdateEvent event) {
        // 获取要发送短信的手机号码
        String orderId = event.getOrderId();
        QihoOrderSnapshotEntity order = qihoOrderSnapshotDAO.findByOrderId(orderId);
        if (Objects.equal(null, order) || StringUtils.isBlank(order.getMobile())) {
            LOG.info("订单不存在，或者手机号为空，无法发送短信.orderId={}", orderId);
            return;
        }
        String mobile = order.getMobile();
        YTOLogisticsStatusEnum status = event.getLogisticsStatus();
        switch (status) {
            case SENT_SCAN:
                SmsTemplateEnum sendTemplate = StringUtils.equals(order.getPayType(), PayTypeEnum.COD.getCode()) ? SmsTemplateEnum.ORDER_DELIVERY : SmsTemplateEnum.ORDER_DELIVERY_PAID;
                sendMessage(sendTemplate, event.getDeliveryMsg(), mobile);
                break;
            case SIGNED:
                sendMessage(SmsTemplateEnum.ORDER_RECIVED, null, mobile);
                break;
            default:
                LOG.info("不需要发送短信的状态，status=" + status.getCode());
                break;
        }
    }

    /**
     * @author zhanglihui
     * @param event
     */
    @Subscribe
    public void paySuccessListener(FundInfoUpdateEvent event) {

        // 用户下单成功短信通知
        QihoTemplateEntity template = qihoTemplateDAO.findByCode(SmsTemplateEnum.ORDER_SUCCESS.getCode());
        String mobile = null;
        if (template.getTemplateEnable()) {
            QihoOrderSnapshotEntity order = qihoOrderSnapshotDAO.findByOrderId(event.getOrderId());
            mobile = order.getMobile();
            Map<String, String> context = Maps.newHashMap();
            context.put("orderDetailUrl", ShortUrlUtil.getShortUrlByOrderId(event.getOrderId()));
            context.put("itemName", order.getItemShortName());
            String text = SimpleTemplateUtil.render(template.getTemplateContext(), context);
            singleSend(text, mobile);
        }
        //如果是在线支付的订单,还需要发送短信
        if(null!=event.getPayType() && !StringUtils.equals(event.getPayType(),PayTypeEnum.COD.getCode())){
            QihoTemplateEntity paySuccessTemp = qihoTemplateDAO.findByCode(SmsTemplateEnum.PAY_SUCCESS.getCode());
            if (paySuccessTemp.getTemplateEnable()){
                singleSend(paySuccessTemp.getTemplateContext(),mobile);
            }
        }
    }

    private boolean sendMessage(SmsTemplateEnum template, Map<String, String> context, String mobile) {
        QihoTemplateEntity entity = qihoTemplateDAO.findByCode(template.getCode());
        if (entity.getTemplateEnable()) {
            String text = SimpleTemplateUtil.render(entity.getTemplateContext(), context);
            return singleSend(text, mobile);
        }
        return true;
    }
}
