package com.qiho.center.biz.bo;

import java.util.Map;
import java.util.concurrent.Callable;
import java.util.concurrent.ExecutorService;

import javax.annotation.PostConstruct;
import javax.annotation.Resource;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import com.alibaba.fastjson.JSONArray;
import com.alibaba.fastjson.JSONObject;
import com.google.common.base.Objects;
import com.google.common.collect.Maps;
import com.google.common.eventbus.Subscribe;
import com.google.common.util.concurrent.*;
import com.qiho.center.api.dto.ErpOrderDto;
import com.qiho.center.api.dto.OrderDto;
import com.qiho.center.api.dto.OrderItemDto;
import com.qiho.center.api.enums.DeliveryEnum;
import com.qiho.center.api.enums.ErpTypeEnum;
import com.qiho.center.api.enums.OrderStatusEnum;
import com.qiho.center.api.params.OrderAuditParams;
import com.qiho.center.api.params.ResolveDeliverParams;
import com.qiho.center.biz.event.OrderAuditEvent;
import com.qiho.center.biz.service.LogisticsOrderService;
import com.qiho.center.biz.service.order.ErpOrderService;
import com.qiho.center.biz.service.order.FundOrderService;
import com.qiho.center.biz.service.order.OrderService;
import com.qiho.center.common.annotations.BizEventListener;
import com.qiho.center.common.constant.DsConstants;
import com.qiho.center.common.entity.order.QihoErpOrderEntity;
import com.qiho.center.common.util.ERPClientUtil;

import cn.com.duiba.wolf.utils.DateUtils;

/**
 * Created by liuyao on 2017/6/12.
 */
@Service
@BizEventListener
public class ErpOrderBo {

    private static Logger log = LoggerFactory.getLogger(ErpOrderBo.class);

    @Autowired
    private ErpOrderService erpOrderService;
    @Autowired
    private OrderBo orderBo;
    @Autowired
    private OrderService orderService;
    @Autowired
    private LogisticsOrderService logisticsOrderService;
    @Autowired
    private FundOrderService fundOrderService;
    @Resource
    private ExecutorService executorService;
    @Value("${qiho.erp.shop.no}")
    private String shopNo;
    @Value("${qiho.erp.auto.reliver.url}")
    private String erpAutoReliverUrl;



    private ListeningExecutorService service;

    @PostConstruct
    public void init(){
        service = MoreExecutors.listeningDecorator(executorService);
    }

    /**
     * 取消发货
     * @param erpId
     * @return
     */
    public void cancelDeliver(String erpId){
        ErpOrderDto erpOrderDto = erpOrderService.cancelDeliver(erpId);
        if(Objects.equal(erpOrderDto.getErpType(),ErpTypeEnum.SEND.getCode())){
            orderBo.rejectOrder(erpOrderDto.getOrderId(),"ERP子订单通知失败");
        }
    }

    /**
     * 确认发货
     * @param params
     */
    @Transactional(DsConstants.DATABASE_QIHO)
    public void resolveDeliver(ResolveDeliverParams params) {
        ErpOrderDto erpOrderDto = erpOrderService.resolveDeliver(params.getErpId());
        //创建物流子订单
        logisticsOrderService.initLogisticsOrder(erpOrderDto,params);

        OrderDto order = orderService.findByOrderId(erpOrderDto.getOrderId());
        //初始化付款子订单
        fundOrderService.initFundOrder(order);
        //变更主订单状态
        orderBo.updateOrderStatus(order.getOrderId(), OrderStatusEnum.SEND_OUT);
    }

    /**
     * 监听订单审核事件
     * @param event
     */
    @Subscribe
    public void OrderAuditEventListener(OrderAuditEvent event){
        if(!Objects.equal(event.getAuditResult(), OrderAuditParams.AUDIT_SUCCESS)){
            return;
        }
        OrderDto order = orderService.findByOrderId(event.getOrderId());
        OrderItemDto itemEntity = order.getOrderItem();

        QihoErpOrderEntity erpOrder = new QihoErpOrderEntity();
        erpOrder.setOrderId(order.getOrderId());
        erpOrder.setDelivery(itemEntity.getDelivery());
        erpOrder.setItemId(itemEntity.getItemId());
        erpOrder.setItemName(itemEntity.getItemName());
        erpOrder.setItemNo(itemEntity.getItemNo());
        erpOrder.setSkuNo(itemEntity.getSkuNo());
        erpOrder.setConsumerName(order.getConsumerName());
        erpOrder.setMobile(order.getMobile());
        erpOrder.setProvince(order.getProvince());
        erpOrder.setCity(order.getCity());
        erpOrder.setDistrict(order.getDistrict());
        erpOrder.setAddress(order.getAddress());
        erpOrder.setErpType(ErpTypeEnum.SEND.getCode());

        ErpOrderDto newERPOrder = erpOrderService.createErpOrder(erpOrder);

        //自动发货
        autoErpReliver(order,newERPOrder);
    }

    private void autoErpReliver(OrderDto order,ErpOrderDto erp){
        //如果是ERP发货,自动调用的旺店通的接口
        if(!Objects.equal(DeliveryEnum.ERP.getCode(),erp.getDelivery())){
            return;
        }
        AutoErpReliverRunnable runnable = new AutoErpReliverRunnable();
        runnable.setErpOrderDto(erp);
        runnable.setOrder(order);
        AutoErpReliverFutureCallback callback = new AutoErpReliverFutureCallback();
        callback.setErpOrderDto(erp);
        callback.setOrder(order);

        ListenableFuture future = service.submit(runnable);
        Futures.addCallback(future, callback);
    }


    /**
     * ERP自动发货
     */
    private class AutoErpReliverRunnable implements Callable<JSONObject> {

        private ErpOrderDto erpOrderDto;

        private OrderDto order;

        @Override
        public JSONObject call() throws Exception {
            Map<String,String> params = getParams();
            String result = ERPClientUtil.execute(erpAutoReliverUrl,params);
            return JSONObject.parseObject(result);
        }

        public void setErpOrderDto(ErpOrderDto erpOrderDto) {
            this.erpOrderDto = erpOrderDto;
        }

        public void setOrder(OrderDto order) {
            this.order = order;
        }

        private Map<String,String> getParams(){
            Map<String,String> parmas = Maps.newHashMap();

            parmas.put("shop_no",shopNo);

            JSONArray trade_list = new JSONArray();
            JSONObject trade_item = new JSONObject();
            trade_item.put("tid",erpOrderDto.getErpId());
            trade_item.put("trade_status",30);
            trade_item.put("pay_status",0);
            trade_item.put("delivery_term",2);
            trade_item.put("trade_time", DateUtils.getSecondStr(erpOrderDto.getGmtCreate()));
            trade_item.put("buyer_nick",erpOrderDto.getConsumerName());
            trade_item.put("receiver_name",erpOrderDto.getConsumerName());
            trade_item.put("receiver_province",erpOrderDto.getProvince());
            trade_item.put("receiver_city",erpOrderDto.getCity());
            trade_item.put("receiver_district",erpOrderDto.getDistrict());
            trade_item.put("receiver_address",erpOrderDto.getAddress());
            trade_item.put("receiver_mobile",erpOrderDto.getMobile());
            trade_item.put("logistics_type",4);//文档中4代表圆通速递
            trade_item.put("buyer_message",order.getMessage());
            trade_item.put("post_amount",0);

            trade_item.put("cod_amount",(float)order.getOrderAmt()/100);
            trade_item.put("ext_cod_fee",0);
            trade_item.put("other_amount",0);
            trade_item.put("paid",0);

            JSONArray order_list = new JSONArray();
            JSONObject order_item = new JSONObject();
            order_item.put("oid",erpOrderDto.getErpId()+"-1");
            order_item.put("num",order.getOrderItem().getQuantity());
            order_item.put("price",(float)order.getOrderItem().getSellingPrice()/100);
            order_item.put("status",10);
            order_item.put("refund_status",0);
            order_item.put("goods_id",order.getOrderItem().getItemId());
            order_item.put("spec_id",order.getOrderItem().getSkuId());
            order_item.put("goods_no",order.getOrderItem().getItemNo());
            order_item.put("goods_name",order.getOrderItem().getItemName());
            order_item.put("spec_no",order.getOrderItem().getSkuNo());
            order_item.put("spec_name",order.getOrderItem().getSkuName());
            order_item.put("cid",1);//无用
            order_list.add(order_item);
            trade_item.put("order_list",order_list);
            trade_list.add(trade_item);
            parmas.put("trade_list",trade_list.toJSONString());
            return parmas;
        }
    }

    /**
     * 自动发货回调
     */
    private class AutoErpReliverFutureCallback implements FutureCallback<JSONObject> {

        private int REPEAT_COUNT = 2;//请求失败的重试次数(总共请求三次)

        private ErpOrderDto erpOrderDto;

        private OrderDto order;

        @Override
        public void onSuccess(JSONObject result) {
            if(result.getInteger("code")!=0){
                erpOrderService.cancelDeliver(erpOrderDto.getErpId());
                log.error("ERP自动发货返回失败:"+result.getString("message"));
            }else{
                erpOrderService.resolveDeliver(erpOrderDto.getErpId());
            }
        }

        @Override
        public void onFailure(Throwable t) {
            log.error("ERP订单自动发货请求失败,ERP={}",t,erpOrderDto.getErpId());
            REPEAT_COUNT--;
            //重试请求
            if(REPEAT_COUNT>0){
                autoErpReliver(order,erpOrderDto);
            }
        }

        public void setErpOrderDto(ErpOrderDto erpOrderDto) {
            this.erpOrderDto = erpOrderDto;
        }

        public void setOrder(OrderDto order) {
            this.order = order;
        }
    }


}
