package com.qiho.center.biz.bo;

import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ExecutorService;

import javax.annotation.Resource;

import org.apache.commons.collections.ListUtils;
import org.apache.commons.lang.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.BeansException;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.ApplicationContext;
import org.springframework.context.ApplicationContextAware;
import org.springframework.stereotype.Service;

import cn.com.duiba.wolf.utils.BeanUtils;

import com.alibaba.dubbo.common.utils.CollectionUtils;
import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONArray;
import com.alibaba.fastjson.JSONObject;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.qiho.center.api.dto.logistics.ErpLogisticsSyncAckDto;
import com.qiho.center.api.dto.logistics.ErpLogisticsSyncDto;
import com.qiho.center.api.dto.logistics.LogisticsOrderDto;
import com.qiho.center.api.dto.logistics.LogisticsProcessDto;
import com.qiho.center.api.enums.ErpErrorCodeReturnEnum;
import com.qiho.center.api.enums.ExpressCompanyEnum;
import com.qiho.center.biz.runnable.LogisticsOrderSignRunnable;
import com.qiho.center.biz.service.LogisticsOrderService;
import com.qiho.center.common.params.QueryYTOLogisticProcessParams;
import com.qiho.center.common.params.QueryYTOLogisticProcessParams.Result;
import com.qiho.center.common.params.QueryYTOLogisticProcessParams.Result.WaybillCode;
import com.qiho.center.common.params.YTOLogisticProcessParams;
import com.qiho.center.common.params.YTOLogisticProcessParams.Result.LogisticDetail;
import com.qiho.center.common.util.ERPClientUtil;
import com.qiho.center.common.util.JaxbUtil;
import com.qiho.center.common.util.YTOClientUtil;

/**
 * ClassName: LogisticsOrderBo <br/>
 * date: 2017年6月8日 下午6:04:21 <br/>
 *
 * @author chencheng
 * @version
 */
@Service
public class LogisticsOrderBo implements ApplicationContextAware {

    private static final Logger   LOG                            = LoggerFactory.getLogger(LogisticsOrderBo.class);

    /** ERP系统返回物流信息条数，ERP系统最多支持100条 */
    private static final String   QUERY_ERP_LOGISTICS_SYNC_LIMIT = "100";

    /** 是否支持拆单发货 0 否 1 是 */
    private static final String   IS_PART_SYNC_ABLE              = "0";

    private static final String   YTO_FAILED_RETURN              = "<success>false</success>";

    private ApplicationContext    context;

    @Autowired
    private LogisticsOrderService logisticsOrderService;

    @Resource
    private ExecutorService       executorService;

    /**
     * 定时任务，同步订单发货物流信息
     *
     * @author chencheng
     */
    public void erpLogisticsSyncQuery() {

        while (true) {
            // ERP系统同步物流信息
            Map<String, String> params = Maps.newHashMap();
            params.put("shop_no", ERPClientUtil.getShopNo());
            params.put("limit", QUERY_ERP_LOGISTICS_SYNC_LIMIT);
            params.put("is_part_sync_able", IS_PART_SYNC_ABLE);
            String logisticsJson = null;
            try {
                logisticsJson = ERPClientUtil.execute(ERPClientUtil.getLogisticsSyncUrl(), params);
            } catch (Exception e) {
                LOG.error("ERP系统同步物流信息失败，发送请求失败");
                break;
            }

            List<ErpLogisticsSyncDto> list = getLogisticsOrderDtoList(logisticsJson);
            if (CollectionUtils.isEmpty(list)) {
                LOG.info("ERP系统同步物流信息完成。无待同步物流信息");
                break;
            }
            // 创建物流子订单
            List<ErpLogisticsSyncAckDto> listReurn = logisticsOrderService.createLogisticsOrderBatch(list);
            if (CollectionUtils.isEmpty(listReurn)) {
                continue;
            }
            Map<String, String> returnParams = Maps.newHashMap();
            returnParams.put("logistics_list", JSON.toJSONString(listReurn));
            try {
                ERPClientUtil.execute(ERPClientUtil.getLogisticsAckUrl(), returnParams);
            } catch (Exception e) {
                LOG.error("返回ERP系统同步物流信息处理结果，请求失败");
            }
        }
    }

    /**
     * 解析erp系统返回物流信息json
     *
     * @author chencheng
     * @param logisticsJson
     * @return
     */
    @SuppressWarnings("unchecked")
    public List<ErpLogisticsSyncDto> getLogisticsOrderDtoList(String logisticsJson) {
        List<ErpLogisticsSyncDto> list = Lists.newArrayList();
        try {
            if (StringUtils.isBlank(logisticsJson)) {
                return ListUtils.EMPTY_LIST;
            }
            JSONObject json = JSON.parseObject(logisticsJson);
            String code = json.getString("code");
            // 错误码为失败
            if (!StringUtils.equals(code, ErpErrorCodeReturnEnum.SUCCESS.getCode())) {
                return ListUtils.EMPTY_LIST;
            }
            JSONArray temp = JSON.parseArray(json.get("trades").toString());
            // 无待同步物流订单信息
            if (temp.size() <= 0) {
                return ListUtils.EMPTY_LIST;
            }
            for (int j = 0; j < temp.size(); j++) {
                JSONObject t = temp.getJSONObject(j);
                ErpLogisticsSyncDto queryLogisticsReturnDto = JSON.toJavaObject(t, ErpLogisticsSyncDto.class);
                queryLogisticsReturnDto.setLogisticsCode(ExpressCompanyEnum.YTO.getCode());
                list.add(queryLogisticsReturnDto);
            }
        } catch (Exception e) {
            return ListUtils.EMPTY_LIST;
        }
        return list;
    }

    /**
     * 查询圆通快递运单走件流程
     *
     * @author chencheng
     * @param postId
     * @return
     */
    @SuppressWarnings("unchecked")
    public List<LogisticsProcessDto> queryYTOLogisticsProcessByPostId(String postId) {
        if (StringUtils.isBlank(postId)) {
            return ListUtils.EMPTY_LIST;
        }
        QueryYTOLogisticProcessParams queryXml = new QueryYTOLogisticProcessParams(new Result(new WaybillCode(postId)));
        // 请求查询走件流程
        List<LogisticsProcessDto> list = Lists.newArrayList();
        try {
            String logisticsString = YTOClientUtil.execute(YTOClientUtil.getQueryLogisticsUrl(), queryXml);
            // 结果返回
            if (StringUtils.isNotBlank(logisticsString) && !StringUtils.contains(logisticsString, YTO_FAILED_RETURN)) {
                YTOLogisticProcessParams returnLogistics = JaxbUtil.converyToJavaBean(logisticsString,
                                                                                      YTOLogisticProcessParams.class);
                if (returnLogistics != null) {
                    List<LogisticDetail> logisticDetail = returnLogistics.getResult().getLogisticDetail();
                    list = BeanUtils.copyList(logisticDetail, LogisticsProcessDto.class);
                }
            }
            // 倒序排序
            if (CollectionUtils.isNotEmpty(list)) {
                Collections.reverse(list);
            }
        } catch (Exception e) {
            LOG.error("查询圆通快递运单走件流程异常");
            return ListUtils.EMPTY_LIST;
        }
        return list;
    }

    public Integer batchSignOrder(String progressKey, List<LogisticsOrderDto> paramList) {
        LogisticsOrderSignRunnable runnable = context.getBean(LogisticsOrderSignRunnable.class);
        runnable.setProgressKey(progressKey);
        runnable.setParams(paramList);
        int enableCount = runnable.enableFilter();
        executorService.submit(runnable);
        return enableCount;
    }

    @Override
    public void setApplicationContext(ApplicationContext applicationContext) throws BeansException {
        this.context = applicationContext;
    }

}
