/**
 * Project Name:qiho-center-biz File Name:ItemSkuServiceImpl.java Package Name:com.qiho.center.biz.service.impl
 * Date:2017年5月20日下午6:01:54 Copyright (c) 2017, duiba.com.cn All Rights Reserved.
 */

package com.qiho.center.biz.service.impl;

import java.util.List;
import java.util.Map;

import org.apache.commons.collections.CollectionUtils;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import cn.com.duiba.stock.service.api.dto.NewStockReqDto;
import cn.com.duiba.stock.service.api.dto.NewStockRespDto;
import cn.com.duiba.stock.service.api.dto.UpdateStockReqDto;
import cn.com.duiba.stock.service.api.remoteservice.RemoteStockBackendService;
import cn.com.duiba.stock.service.api.remoteservice.RemoteStockService;
import cn.com.duiba.wolf.dubbo.DubboResult;
import cn.com.duiba.wolf.utils.BeanUtils;

import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.qiho.center.api.dto.ItemSkuDto;
import com.qiho.center.biz.service.ItemSkuService;
import com.qiho.center.common.constant.DsConstants;
import com.qiho.center.common.dao.QihoItemSkuDAO;
import com.qiho.center.common.entity.item.QihoItemSkuEntity;

/**
 * ClassName:ItemSkuServiceImpl <br/>
 * Date: 2017年5月20日 下午6:01:54 <br/>
 * 
 * @author zhanglihui
 * @version
 * @since JDK 1.6
 * @see
 */
@Service
public class ItemSkuServiceImpl implements ItemSkuService {

    @Autowired
    private QihoItemSkuDAO            qihoItemSkuDAO;

    @Autowired
    private RemoteStockService        remoteStockService;

    @Autowired
    private RemoteStockBackendService remoteStockBackendService;

    /**
     * @see com.qiho.center.biz.service.ItemSkuService#getSkuByItemId(java.lang.Long)
     */
    @Override
    public List<ItemSkuDto> getSkuByItemId(Long itemId) {
        List<QihoItemSkuEntity> skuEntityList = qihoItemSkuDAO.getSkuByItemIds(Lists.newArrayList(itemId));
        List<ItemSkuDto> skuDtoList = Lists.newArrayList();
        if (CollectionUtils.isEmpty(skuEntityList)) {
            return skuDtoList;
        }
        List<Long> stockIds = Lists.newArrayList();
        for (QihoItemSkuEntity skuEntity : skuEntityList) {
            stockIds.add(skuEntity.getStockId());
        }
        // stockId : 库存 键值对
        Map<Long, Long> stockMap = remoteStockService.findBatch(stockIds).getResult();
        for (QihoItemSkuEntity skuEntity : skuEntityList) {
            ItemSkuDto skuDto = BeanUtils.copy(skuEntity, ItemSkuDto.class);
            skuDto.setStock(stockMap.get(skuEntity.getStockId()));
            skuDto.setStockId(skuEntity.getStockId());
            skuDtoList.add(skuDto);
        }
        return skuDtoList;
    }

    /**
     * @see com.qiho.center.biz.service.ItemSkuService#updateItemSku(java.util.List, java.lang.Long)
     */
    @Override
    @Transactional(DsConstants.DATABASE_QIHO)
    public void updateItemSku(List<ItemSkuDto> skuList, Long itemId) {
        if (CollectionUtils.isEmpty(skuList) || itemId == null) {
            return;
        }
        List<Long> skuIds = Lists.newArrayList();
        // 要创建的sku列表
        List<QihoItemSkuEntity> createSkuList = Lists.newArrayList();
        // 要更新的sku列表
        List<ItemSkuDto> modifySkuList = Lists.newArrayList();
        for (ItemSkuDto skuDto : skuList) {
            if (skuDto.getId() != null) {
                skuIds.add(skuDto.getId());
                modifySkuList.add(skuDto);
            } else {
                QihoItemSkuEntity skuEntity = BeanUtils.copy(skuDto, QihoItemSkuEntity.class);
                skuEntity.setItemId(itemId);
                skuEntity.setDeleted(false);
                createSkuList.add(skuEntity);
            }
        }
        if (CollectionUtils.isNotEmpty(skuIds)) {
            // 删除sku
            qihoItemSkuDAO.deleteNotIn(skuIds, itemId);
        }else{
            qihoItemSkuDAO.deleteByItemId(itemId);
        }
        // 新建sku
        createSku(createSkuList);
        // 更新sku
        modifySku(modifySkuList);

    }

    /**
     * @author zhanglihui
     * @param modifySkuList
     */
    private void modifySku(List<ItemSkuDto> modifySkuList) {
        if(CollectionUtils.isEmpty(modifySkuList)){
            return;
        }
        // 批量更新sku
        List<QihoItemSkuEntity> skuList = Lists.newArrayList();
        List<UpdateStockReqDto> decreaseList = Lists.newArrayList();
        List<UpdateStockReqDto> increaseList = Lists.newArrayList();
        for (ItemSkuDto skuDto : modifySkuList) {
            skuList.add(BeanUtils.copy(skuDto, QihoItemSkuEntity.class));
            Long stockChanges = skuDto.getStockChanges();
            if (stockChanges != null) {
                UpdateStockReqDto updateStockReqDto = new UpdateStockReqDto();
                if(skuDto.getStockId() != null){
                    updateStockReqDto.setStockId(skuDto.getStockId());
                    updateStockReqDto.setStock(Math.abs(stockChanges));
                    if (stockChanges < 0) {
                        decreaseList.add(updateStockReqDto);
                    } else if (stockChanges > 0) {
                        increaseList.add(updateStockReqDto);
                    }
                }
            }
        }
        qihoItemSkuDAO.updateBatch(skuList);
        // 批量更新库存
        if(CollectionUtils.isNotEmpty(decreaseList)){
            remoteStockBackendService.decreaseItemStockBatch(decreaseList);
        }
        if(CollectionUtils.isNotEmpty(increaseList)){
            remoteStockBackendService.increaseItemStockBatch(increaseList);
        }

    }

    /**
     * @author zhanglihui
     * @param createSkuList
     */
    private void createSku(List<QihoItemSkuEntity> createSkuList) {
        if (CollectionUtils.isEmpty(createSkuList)) {
            return;
        }
        qihoItemSkuDAO.insertBatch(createSkuList);
        // 批量创建库存，并把库存ID回写到SKU表
        List<NewStockReqDto> stockReqList = Lists.newArrayList();
        Map<Long, QihoItemSkuEntity> skuMap = Maps.newHashMap();
        for (QihoItemSkuEntity skuEntity : createSkuList) {
            skuMap.put(skuEntity.getId(), skuEntity);
            NewStockReqDto stockReq = new NewStockReqDto();
            stockReq.setGid(skuEntity.getId());
            stockReq.setStock(skuEntity.getStock());
            stockReqList.add(stockReq);
        }
        List<NewStockRespDto> stockRespList = remoteStockBackendService.newStockBatch(stockReqList).getResult();
        for (NewStockRespDto stockResp : stockRespList) {
            QihoItemSkuEntity skuEntity = skuMap.get(stockResp.getGid());
            skuEntity.setStockId(stockResp.getStockId());
        }
        qihoItemSkuDAO.updateBatch(createSkuList);
    }

    /**
     * @see com.qiho.center.biz.service.ItemSkuService#querySku(java.lang.Long)
     */
    @Override
    public ItemSkuDto querySku(Long skuId) {
        QihoItemSkuEntity entity = qihoItemSkuDAO.findById(skuId);
        if(entity == null){
            return null;
        }
        ItemSkuDto sku = BeanUtils.copy(entity, ItemSkuDto.class);
        DubboResult<Long> result = remoteStockService.find(entity.getStockId());
        sku.setStock(result.getResult());
        return sku;
    }

}
