package com.qiho.center.biz.service.impl;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;

import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import cn.com.duiba.wolf.utils.BeanUtils;

import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.qiho.center.api.dto.ErpOrderDto;
import com.qiho.center.api.dto.PagenationDto;
import com.qiho.center.api.dto.logistics.ErpLogisticsSyncAckDto;
import com.qiho.center.api.dto.logistics.ErpLogisticsSyncDto;
import com.qiho.center.api.dto.logistics.LogisticsOrderDto;
import com.qiho.center.api.dto.logistics.QueryLogisticsOrderDto;
import com.qiho.center.api.enums.ErpErrorCodeReturnEnum;
import com.qiho.center.api.enums.ExpressCompanyEnum;
import com.qiho.center.api.enums.LogisticsOrderStatusEnum;
import com.qiho.center.api.enums.OrderStatusEnum;
import com.qiho.center.api.enums.YTOLogisticsStatusEnum;
import com.qiho.center.api.params.ResolveDeliverParams;
import com.qiho.center.biz.event.LogisticsInfoUpdateEvent;
import com.qiho.center.biz.event.LogisticsUpdateEvent;
import com.qiho.center.biz.service.LogisticsOrderService;
import com.qiho.center.biz.service.order.OrderService;
import com.qiho.center.common.constant.DsConstants;
import com.qiho.center.common.dao.QihoErpOrderDAO;
import com.qiho.center.common.dao.QihoLogisticsOrderDAO;
import com.qiho.center.common.entity.logistics.LogisticsOrderEntity;
import com.qiho.center.common.entity.order.QihoErpOrderEntity;
import com.qiho.center.common.enums.SeqBizTypeEnum;
import com.qiho.center.common.params.ErpLogisticsOrderParams;
import com.qiho.center.common.support.BizEventBus;
import com.qiho.center.common.support.SequenceNoBuilder;

@Service
public class LogisticsOrderServiceImpl implements LogisticsOrderService {

    private static final Logger   LOG = LoggerFactory.getLogger(LogisticsOrderServiceImpl.class);

    @Autowired
    private QihoLogisticsOrderDAO qihoLogisticsOrderDAO;
    @Autowired
    private SequenceNoBuilder     builder;
    @Autowired
    private BizEventBus           bizEventBus;
    @Autowired
    private QihoErpOrderDAO       qihoErpOrderDAO;
    @Autowired
    private OrderService orderService;

    @Override
    public void initLogisticsOrder(ErpOrderDto erp, ResolveDeliverParams params) {
        // 构建物流同步对象
        ErpLogisticsSyncDto erpLogisticsSyncDto = new ErpLogisticsSyncDto();
        ExpressCompanyEnum expressCompany = params.getExpress();
        erpLogisticsSyncDto.setLogisticsCode(expressCompany.getCode());
        erpLogisticsSyncDto.setLogisticsName(expressCompany.getName());
        erpLogisticsSyncDto.setLogisticsNo(params.getExpressCode());
        erpLogisticsSyncDto.setTid(erp.getErpId());
        LogisticsOrderEntity entity = convertToLogisticsOrderEntity(erpLogisticsSyncDto,
                                                                    BeanUtils.copy(erp, QihoErpOrderEntity.class));
        createLogisticsOrder(entity);
    }

    @Transactional(DsConstants.DATABASE_QIHO)
    private void createLogisticsOrder(LogisticsOrderEntity logisticsOrderEntity) {
        qihoLogisticsOrderDAO.insert(logisticsOrderEntity);
        orderService.updateOrderStatus(logisticsOrderEntity.getOrderId(), OrderStatusEnum.SEND_OUT);
        // 发布订单物流信息更新事件
        LogisticsInfoUpdateEvent event = new LogisticsInfoUpdateEvent();
        BeanUtils.copy(logisticsOrderEntity, event);
        bizEventBus.post(event);
    }

    @Override
    public PagenationDto<LogisticsOrderDto> queryLogisticsOrderPage(QueryLogisticsOrderDto queryItem, Integer pageNum,
                                                                    Integer pageSize) {
        PagenationDto<LogisticsOrderDto> pagenation = new PagenationDto<>();
        LogisticsOrderEntity param = BeanUtils.copy(queryItem, LogisticsOrderEntity.class);
        int total = qihoLogisticsOrderDAO.countLogisticsOrderByParam(param, queryItem.getStartTime(),
                                                                     queryItem.getEndTime());
        List<LogisticsOrderEntity> list = null;
        if (total > 0) {
            list = qihoLogisticsOrderDAO.queryLogisticsOrderByParam(param, pageNum, pageSize, queryItem.getStartTime(),
                                                                    queryItem.getEndTime());
        }
        pagenation.setList(convertToLogisticsOrderDto(list));
        pagenation.setTotal(total);
        return pagenation;
    }

    /**
     * LogisticsOrderEntity转LogisticsOrderDto
     *
     * @author chencheng
     * @param list
     * @return
     */
    private List<LogisticsOrderDto> convertToLogisticsOrderDto(List<LogisticsOrderEntity> list) {
        List<LogisticsOrderDto> logisticsOrderDtoList = Lists.newArrayList();
        if (CollectionUtils.isEmpty(list)) {
            return logisticsOrderDtoList;
        }
        for (LogisticsOrderEntity entity : list) {
            LogisticsOrderDto dto = BeanUtils.copy(entity, LogisticsOrderDto.class);
            dto.setOrderStatus(LogisticsOrderStatusEnum.getByCode(entity.getOrderStatus()) != null ? LogisticsOrderStatusEnum.getByCode(entity.getOrderStatus()).getDesc() : "未知");
            logisticsOrderDtoList.add(dto);
        }
        return logisticsOrderDtoList;
    }

    @Override
    public Integer updateBylogisticsId(String logisticsId, String orderStatus) {
        LogisticsOrderDto dto = findByLogisticsId(logisticsId);
        // 查询不到，或者已签收
        if (dto == null || StringUtils.equals(dto.getOrderStatus(), LogisticsOrderStatusEnum.FAILED.getCode())
            || StringUtils.equals(dto.getOrderStatus(), LogisticsOrderStatusEnum.SUCCESS.getCode())) {
            LOG.warn("物流子订单已签收，或查询不到。无法修改订单状态,logisticsId={}", logisticsId);
            return 0;
        }
        String logisticsStatus = StringUtils.equals(orderStatus, LogisticsOrderStatusEnum.FAILED.getCode()) ? YTOLogisticsStatusEnum.FAILED.getDesc() : YTOLogisticsStatusEnum.SIGNED.getDesc();
        int success = qihoLogisticsOrderDAO.updateLogisticsOrderStatus(logisticsId, orderStatus, logisticsStatus);
        // 发布物流更新事件
        LogisticsUpdateEvent event = new LogisticsUpdateEvent();
        event.setLogisticsStatus(StringUtils.equals(LogisticsOrderStatusEnum.FAILED.getCode(), orderStatus) ? YTOLogisticsStatusEnum.FAILED : YTOLogisticsStatusEnum.SIGNED);
        event.setOrderId(dto.getOrderId());
        event.setOrderStatus(LogisticsOrderStatusEnum.getByCode(orderStatus));
        // erpId由前面传过来。如果是物流推送的话，要走更新订单表流程要查询一下。
        event.setErpId(dto.getErpId());
        bizEventBus.post(event);

        return success;
    }

    /**
     * @see com.qiho.center.biz.service.LogisticsOrderService#findByOrderId(java.lang.String)
     */
    @Override
    public List<LogisticsOrderDto> findByOrderId(String orderId) {
        return BeanUtils.copyList(qihoLogisticsOrderDAO.findByOrderId(orderId), LogisticsOrderDto.class);
    }

    private List<ErpLogisticsSyncAckDto> convertToListReurn(Map<String, ErpLogisticsOrderParams> errorCodeMap) {
        List<ErpLogisticsSyncAckDto> listReurn = new ArrayList<>();
        for (Entry<String, ErpLogisticsOrderParams> errorMap : errorCodeMap.entrySet()) {
            ErpLogisticsOrderParams erpLogisticsOrderParams = errorMap.getValue();
            listReurn.add(new ErpLogisticsSyncAckDto(erpLogisticsOrderParams.getRecId(),
                                                     erpLogisticsOrderParams.getErpErrorCodeReturnEnum()));
        }
        return listReurn;
    }

    private LogisticsOrderEntity convertToLogisticsOrderEntity(ErpLogisticsSyncDto erpLogisticsSyncDto,
                                                               QihoErpOrderEntity erpOrder) {
        LogisticsOrderEntity entity = new LogisticsOrderEntity();
        entity.setErpId(erpLogisticsSyncDto.getTid());
        entity.setPostId(erpLogisticsSyncDto.getLogisticsNo());
        entity.setLogisticsCode(erpLogisticsSyncDto.getLogisticsCode());
        entity.setLogisticsName(erpLogisticsSyncDto.getLogisticsName());
        entity.setLogisticsStatus(YTOLogisticsStatusEnum.ACCEPT.getCode());
        entity.setOrderStatus(LogisticsOrderStatusEnum.UNDER_WAY.getCode());
        entity.setConsumerName(erpOrder.getConsumerName());
        entity.setItemId(erpOrder.getItemId());
        entity.setItemName(erpOrder.getItemName());
        entity.setItemNo(erpOrder.getItemNo());
        entity.setLogisticsId(builder.createSeqNo(SeqBizTypeEnum.LOGISTICS, erpOrder.getItemId()));
        entity.setMobile(erpOrder.getMobile());
        entity.setOrderId(erpOrder.getOrderId());
        entity.setSkuNo(erpOrder.getSkuNo());
        return entity;
    }

    @Override
    public Integer update(LogisticsOrderDto logisticsOrderDto) {
        LogisticsOrderDto dto = findByPostIdCode(logisticsOrderDto.getPostId(), logisticsOrderDto.getLogisticsCode());
        // 查询不到，或者已签收
        if (dto == null || StringUtils.equals(dto.getOrderStatus(), LogisticsOrderStatusEnum.FAILED.getCode())
            || StringUtils.equals(dto.getOrderStatus(), LogisticsOrderStatusEnum.SUCCESS.getCode())) {
            LOG.warn("物流子订单已签收，或查询不到。无法修改订单状态,postId={}", logisticsOrderDto.getPostId());
            return 0;
        }
        LogisticsOrderEntity param = BeanUtils.copy(logisticsOrderDto, LogisticsOrderEntity.class);
        int success = qihoLogisticsOrderDAO.update(param);
        if (StringUtils.isNotBlank(param.getLogisticsStatus())) {
            // 发布物流更新事件
            LogisticsUpdateEvent event = new LogisticsUpdateEvent();
            event.setLogisticsStatus(YTOLogisticsStatusEnum.getByName(param.getLogisticsStatus()));
            event.setOrderId(dto.getOrderId());
            event.setOrderStatus(LogisticsOrderStatusEnum.getByCode(param.getOrderStatus()));
            event.setErpId(dto.getErpId());
            bizEventBus.post(event);
        }
        return success;
    }

    /**
     * @see com.qiho.center.biz.service.LogisticsOrderService#createLogisticsOrderBatch(java.util.List)
     */
    @Override
    public List<ErpLogisticsSyncAckDto> createLogisticsOrderBatch(List<ErpLogisticsSyncDto> list) {
        List<ErpLogisticsSyncAckDto> returnList = Lists.newArrayList();
        if (CollectionUtils.isEmpty(list)) {
            return returnList;
        }
        // 订单状态map
        Map<String, ErpLogisticsOrderParams> errorCodeMap = new HashMap<>(list.size());
        Map<String, ErpLogisticsSyncDto> syncMap = Maps.newHashMap();
        List<String> erpIds = Lists.newArrayList();
        for (ErpLogisticsSyncDto erpLogisticsSyncDto : list) {
            // 先置为订单不存在状态，如果数据已存在或更新成功会更改
            errorCodeMap.put(erpLogisticsSyncDto.getTid(),
                             new ErpLogisticsOrderParams(erpLogisticsSyncDto.getRecId(),
                                                         ErpErrorCodeReturnEnum.ORDER_CODE_NON_ERROR));
            erpIds.add(erpLogisticsSyncDto.getTid());
            syncMap.put(erpLogisticsSyncDto.getTid(), erpLogisticsSyncDto);
        }

        List<QihoErpOrderEntity> erpOrderList = qihoErpOrderDAO.findByErpIds(erpIds);
        if (CollectionUtils.isEmpty(erpOrderList)) {
            return convertToListReurn(errorCodeMap);
        }
        // 查询已同步的物流子订单
        List<LogisticsOrderEntity> existList = qihoLogisticsOrderDAO.findByErpIds(erpIds);
        List<String> syncErpIds = Lists.newArrayList();
        if (CollectionUtils.isNotEmpty(existList)) {
            for (LogisticsOrderEntity logisticsOrder : existList) {
                syncErpIds.add(logisticsOrder.getErpId());
            }
        }
        for (QihoErpOrderEntity erpOrder : erpOrderList) {
            errorCodeMap.get(erpOrder.getErpId()).setErpErrorCodeReturnEnum(ErpErrorCodeReturnEnum.SUCCESS);
            if (syncErpIds.contains(erpOrder.getErpId())) {
                // 已经同步的不再创建物流子订单
                continue;
            }
            LogisticsOrderEntity logisticsOrderEntity = convertToLogisticsOrderEntity(syncMap.get(erpOrder.getErpId()),
                                                                                      erpOrder);
            // 创建物流子订单、更新订单快照、发送发货短信
            createLogisticsOrder(logisticsOrderEntity);
        }
        return convertToListReurn(errorCodeMap);
    }

    @Override
    public LogisticsOrderDto findByPostIdCode(String postId, String logisticsCode) {
        return BeanUtils.copy(qihoLogisticsOrderDAO.findByPostIdCode(postId, logisticsCode), LogisticsOrderDto.class);
    }

    @Override
    public LogisticsOrderDto findByLogisticsId(String logisticsId) {
        return BeanUtils.copy(qihoLogisticsOrderDAO.findByLogisticsId(logisticsId), LogisticsOrderDto.class);
    }

}
