/**
 * Project Name:qiho-center-biz
 * File Name:FundOrderServiceImpl.java
 * Package Name:com.qiho.center.biz.service.impl.order
 * Date:2017年6月6日下午1:03:34
 * Copyright (c) 2017, duiba.com.cn All Rights Reserved.
 *
*/

package com.qiho.center.biz.service.impl.order;

import java.util.List;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;

import cn.com.duiba.wolf.utils.BeanUtils;

import com.google.common.eventbus.Subscribe;
import com.qiho.center.api.dto.FundOrderDto;
import com.qiho.center.api.dto.OrderDto;
import com.qiho.center.api.dto.OrderItemDto;
import com.qiho.center.api.dto.PagenationDto;
import com.qiho.center.api.enums.FundStatusEnum;
import com.qiho.center.api.enums.LogisticsOrderStatusEnum;
import com.qiho.center.api.enums.PayTypeEnum;
import com.qiho.center.api.params.FundPageParam;
import com.qiho.center.biz.event.FundInfoUpdateEvent;
import com.qiho.center.biz.event.LogisticsUpdateEvent;
import com.qiho.center.biz.service.order.FundOrderService;
import com.qiho.center.common.annotations.BizEventListener;
import com.qiho.center.common.dao.QihoFundOrderDAO;
import com.qiho.center.common.entity.fund.QihoFundOrderEntity;
import com.qiho.center.common.entity.order.QihoOrderEntity;
import com.qiho.center.common.enums.SeqBizTypeEnum;
import com.qiho.center.common.enums.SmsTemplateEnum;
import com.qiho.center.common.support.BizEventBus;
import com.qiho.center.common.support.SequenceNoBuilder;


/**
 * ClassName:FundOrderServiceImpl <br/>
 * Date:     2017年6月6日 下午1:03:34 <br/>
 * @author   zhanglihui
 * @version  
 * @since    JDK 1.6
 * @see 	 
 */
@Service
@BizEventListener
public class FundOrderServiceImpl implements FundOrderService {
    
    private static final Logger LOG = LoggerFactory.getLogger(FundOrderServiceImpl.class);
    
    @Autowired
    private QihoFundOrderDAO qihoFundOrderDAO;
    @Autowired
    private SequenceNoBuilder builder;
    @Autowired
    private BizEventBus bizEventBus;
    
    
    @Override
    public List<FundOrderDto> findByOrderId(String orderId) {
        return BeanUtils.copyList(qihoFundOrderDAO.findByOrderId(orderId), FundOrderDto.class);
    }

    /**
     * @see com.qiho.center.biz.service.order.FundOrderService#queryFundPage(com.qiho.center.api.params.FundPageParam)
     */
    @Override
    public PagenationDto<FundOrderDto> queryFundPage(FundPageParam param) {
        PagenationDto<FundOrderDto> pagenation = new PagenationDto<>();
        int count = qihoFundOrderDAO.countByParam(param);
        pagenation.setTotal(count);
        if(count>0){
            List<QihoFundOrderEntity> list = qihoFundOrderDAO.queryPage(param);
            List<FundOrderDto> fundList = BeanUtils.copyList(list, FundOrderDto.class);
            pagenation.setList(fundList);
        }
        return pagenation;
    }

    @Override
    public void initFundOrder(OrderDto order) {
        QihoFundOrderEntity entity = new QihoFundOrderEntity();
        entity.setOrderId(order.getOrderId());
        OrderItemDto item = order.getOrderItem();
        String fundId = builder.createSeqNo(SeqBizTypeEnum.FUND,item.getItemId());
        entity.setFundId(fundId);
        entity.setPayType(PayTypeEnum.COD.getCode());
        entity.setAmt(order.getOrderAmt());
        entity.setFundStatus(FundStatusEnum.TO_PAY.getCode());
        qihoFundOrderDAO.initFundOrder(entity);

        FundInfoUpdateEvent event = new FundInfoUpdateEvent();
        event.setOrderId(order.getOrderId());
        event.setPayType(entity.getPayType());
        event.setFundStatus(entity.getFundStatus());
        event.setOrderAmt(entity.getAmt());
        bizEventBus.post(event);
    }

    @Override
    public Integer updateFundOrderStatus(String orderId, String fundStatus) {
        return qihoFundOrderDAO.updateFundOrderStatus(orderId, fundStatus);
    }

    /**
     * 监听物流状态更新事件
     * @author chencheng
     * @param event
     */
    @Subscribe
    public void updateLogisticsListener(LogisticsUpdateEvent event) {
        LogisticsOrderStatusEnum status = event.getOrderStatus();
        switch (status) {
            case SUCCESS:
                updateFundOrderStatus(event.getOrderId(), FundStatusEnum.SUCCESS.getCode());
                break;
            case FAILED:
                updateFundOrderStatus(event.getOrderId(), FundStatusEnum.FAILED.getCode());
                break;
            default:
                LOG.info("不需要更改订单的状态，status="+status.getCode());
                break;
        }
    }
}

