/**
 * Project Name:qiho-center-biz File Name:OrderServiceImpl.java Package Name:com.qiho.center.biz.service.impl.order
 * Date:2017年6月2日下午9:01:40 Copyright (c) 2017, duiba.com.cn All Rights Reserved.
 */

package com.qiho.center.biz.service.impl.order;

import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import cn.com.duiba.stock.service.api.constant.ConsumeStockTypes;
import cn.com.duiba.stock.service.api.remoteservice.RemoteStockBackendService;
import cn.com.duiba.stock.service.api.remoteservice.RemoteStockService;
import cn.com.duiba.wolf.dubbo.DubboResult;
import cn.com.duiba.wolf.utils.BeanUtils;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import com.google.common.base.Objects;
import com.google.common.eventbus.Subscribe;
import com.qiho.center.api.dto.ChannelInfoDto;
import com.qiho.center.api.dto.OrderDto;
import com.qiho.center.api.dto.OrderItemDto;
import com.qiho.center.api.enums.ErpTypeEnum;
import com.qiho.center.api.enums.ExpressCompanyEnum;
import com.qiho.center.api.enums.LogisticsOrderStatusEnum;
import com.qiho.center.api.enums.OrderStatusEnum;
import com.qiho.center.api.exception.QihoException;
import com.qiho.center.api.params.OrderAuditParams;
import com.qiho.center.api.params.OrderConsumerParam;
import com.qiho.center.api.params.OrderItemParam;
import com.qiho.center.biz.event.LogisticsUpdateEvent;
import com.qiho.center.biz.event.OrderAuditEvent;
import com.qiho.center.biz.event.OrderCreateEvent;
import com.qiho.center.biz.service.ExpressBlackListService;
import com.qiho.center.biz.service.order.OrderService;
import com.qiho.center.common.annotations.BizEventListener;
import com.qiho.center.common.constant.DsConstants;
import com.qiho.center.common.dao.QihoChannelInfoDAO;
import com.qiho.center.common.dao.QihoErpOrderDAO;
import com.qiho.center.common.dao.QihoItemDAO;
import com.qiho.center.common.dao.QihoItemSkuDAO;
import com.qiho.center.common.dao.QihoOrderDAO;
import com.qiho.center.common.dao.QihoOrderItemInfoDAO;
import com.qiho.center.common.dao.QihoOrderSnapshotDAO;
import com.qiho.center.common.entity.item.QihoItemDetailEntity;
import com.qiho.center.common.entity.item.QihoItemSkuEntity;
import com.qiho.center.common.entity.order.QihoChannelInfoEntity;
import com.qiho.center.common.entity.order.QihoErpOrderEntity;
import com.qiho.center.common.entity.order.QihoOrderEntity;
import com.qiho.center.common.entity.order.QihoOrderItemInfoEntity;
import com.qiho.center.common.entity.order.QihoOrderSnapshotEntity;
import com.qiho.center.common.enums.SeqBizTypeEnum;
import com.qiho.center.common.support.BizEventBus;
import com.qiho.center.common.support.SequenceNoBuilder;

/**
 * ClassName:OrderServiceImpl <br/>
 * Date: 2017年6月2日 下午9:01:40 <br/>
 *
 * @author zhanglihui
 * @see
 * @since JDK 1.6
 */
@Service
@BizEventListener
public class OrderServiceImpl implements OrderService {

    private static final Logger       LOG = LoggerFactory.getLogger(OrderServiceImpl.class);

    @Autowired
    private QihoOrderDAO              qihoOrderDAO;
    @Autowired
    private QihoOrderItemInfoDAO      qihoOrderItemInfoDAO;
    @Autowired
    private QihoItemDAO               qihoItemDAO;
    @Autowired
    private QihoItemSkuDAO            qihoItemSkuDAO;
    @Autowired
    private QihoOrderSnapshotDAO      qihoOrderSnapshotDAO;
    @Autowired
    private QihoChannelInfoDAO        qihoChannelInfoDAO;
    @Autowired
    private QihoErpOrderDAO           qihoErpOrderDAO;
    @Autowired
    private SequenceNoBuilder         sequenceNoBuilder;
    @Autowired
    private RemoteStockService        remoteStockService;
    @Autowired
    private RemoteStockBackendService remoteStockBackendService;
    @Autowired
    private BizEventBus               eventBus;
    @Autowired
    private ExpressBlackListService   expressBlackListService;

    @Override
    public String createOrder(OrderDto order, ChannelInfoDto channelInfo) {
        OrderItemDto orderItemDto = order.getOrderItem();
        // 生成订单编号
        String orderId = sequenceNoBuilder.createSeqNo(SeqBizTypeEnum.ORDER, orderItemDto.getItemId());
        QihoItemSkuEntity skuEntity = qihoItemSkuDAO.findById(orderItemDto.getSkuId());
        // 扣减库存
        DubboResult<Boolean> result = remoteStockService.consumeStockNum(ConsumeStockTypes.QihoTakeOrder.getType(),
                                                                         orderId, skuEntity.getStockId(),
                                                                         orderItemDto.getQuantity());
        if (result.isSuccess() || result.getResult()) {
            // 创建订单
            try {
                insertOrder(orderId, order, channelInfo);
            } catch (Exception e) {
                // 创建订单失败，回滚库存
                remoteStockService.rollbackStock(ConsumeStockTypes.QihoTakeOrder.getType(), orderId);
                throw new QihoException("创建订单失败", e);
            }
            return orderId;
        }
        throw new QihoException("创建订单失败-扣减库存失败");
    }

    /**
     * 新增订单
     * 
     * @author zhanglihui
     * @param orderId
     * @param order
     * @param channelInfoDto
     */
    @Transactional(DsConstants.DATABASE_QIHO)
    public void insertOrder(String orderId, OrderDto order, ChannelInfoDto channelInfoDto) {
        // 新增订单
        QihoOrderEntity qihoOrder = BeanUtils.copy(order, QihoOrderEntity.class);
        qihoOrder.setOrderId(orderId);
        qihoOrder.setConsumerId(order.getMobile());
        qihoOrder.setOrderStatus(OrderStatusEnum.TO_APPROVE.getCode());
        qihoOrderDAO.insert(qihoOrder);
        // 新增订单商品详情
        QihoOrderItemInfoEntity qihoOrderItemInfo = BeanUtils.copy(order.getOrderItem(), QihoOrderItemInfoEntity.class);
        qihoOrderItemInfo.setOrderId(orderId);
        qihoOrderItemInfoDAO.insert(qihoOrderItemInfo);

        // 记录渠道信息
        QihoChannelInfoEntity qihoChannelInfo = BeanUtils.copy(channelInfoDto, QihoChannelInfoEntity.class);
        qihoChannelInfo.setOrderId(orderId);
        qihoChannelInfoDAO.insert(qihoChannelInfo);
        // 创建订单快照
        QihoItemDetailEntity itemEntity = qihoItemDAO.findById(qihoOrderItemInfo.getItemId());
        QihoOrderSnapshotEntity snapshot = BeanUtils.copy(qihoOrder, QihoOrderSnapshotEntity.class);
        snapshot.setItemName(qihoOrderItemInfo.getItemName());
        snapshot.setItemShortName(qihoOrderItemInfo.getItemShortName());
        snapshot.setItemId(qihoOrderItemInfo.getItemId());
        snapshot.setItemNo(qihoOrderItemInfo.getItemNo());
        snapshot.setDelivery(qihoOrderItemInfo.getDelivery());
        snapshot.setSellingPrice(qihoOrderItemInfo.getSellingPrice());
        snapshot.setOriginalPrice(qihoOrderItemInfo.getOriginalPrice());
        snapshot.setQuantity(qihoOrderItemInfo.getQuantity());
        snapshot.setSkuName(qihoOrderItemInfo.getSkuName());
        snapshot.setSkuNo(qihoOrderItemInfo.getSkuNo());
        snapshot.setChannelId(qihoChannelInfo.getChannelId());
        snapshot.setCountChannelId(qihoChannelInfo.getCountChannelId());
        snapshot.setUrl(qihoChannelInfo.getUrl());
        snapshot.setIp(qihoChannelInfo.getIp());
        snapshot.setUserAgent(qihoChannelInfo.getUserAgent());
        JSONObject json = JSON.parseObject(itemEntity.getExtParam());
        snapshot.setImage(json.getString("image"));
        qihoOrderSnapshotDAO.insert(snapshot);
        // 发布新增订单事件
        OrderCreateEvent event = new OrderCreateEvent(orderId);
        eventBus.post(event);
    }
    
    @Override
    public Boolean auditOrder(OrderAuditParams params) {
        Integer result;
        switch (params.getAuditResult()) {
            case OrderAuditParams.AUDIT_SUCCESS:
                result = qihoOrderDAO.auditOrderToSuccess(params);
                break;
            case OrderAuditParams.AUDIT_FAILED:
                result = qihoOrderDAO.auditOrderToFailed(params);
                break;
            default:
                throw new QihoException("订单审核参数异常");
        }
        if (result == 1) {
            // 发布订单审核事件
            OrderAuditEvent event = new OrderAuditEvent();
            event.setOrderId(params.getOrderId());
            event.setAuditResult(params.getAuditResult());
            eventBus.post(event);
            return true;
        } else {
            return false;
        }
    }

    /**
     * @see com.qiho.center.biz.service.order.OrderService#findByOrderId(java.lang.String)
     */
    @Override
    public OrderDto findByOrderId(String orderId) {
        QihoOrderEntity qihoOrderEntity = qihoOrderDAO.findByOrderId(orderId);
        OrderDto orderDto = BeanUtils.copy(qihoOrderEntity, OrderDto.class);
        QihoOrderItemInfoEntity qihoOrderItemInfoEntity = qihoOrderItemInfoDAO.findByOrderId(orderId);
        OrderItemDto orderItemDto = BeanUtils.copy(qihoOrderItemInfoEntity, OrderItemDto.class);
        orderDto.setOrderItem(orderItemDto);
        return orderDto;
    }

    @Override
    public void updateOrderItem(OrderItemParam param) {
        QihoOrderEntity order = qihoOrderDAO.findByOrderId(param.getOrderId());
        if (!StringUtils.equals(order.getOrderStatus(), OrderStatusEnum.TO_APPROVE.getCode())) {
            throw new QihoException("更新订单商品详情失败，订单状态错误，orderId=" + param.getOrderId() + ", status="
                                    + order.getOrderStatus());
        }
        // 更新库存
        Long decreaseStockId = StringUtils.isBlank(param.getDecreaseStockId()) ? null : Long.valueOf(param.getDecreaseStockId());
        Integer decreaseStock = param.getDecreaseStock();
        Long increaseStockId = StringUtils.isBlank(param.getIncreaseStockId()) ? null : Long.valueOf(param.getIncreaseStockId());
        Integer increaseStock = param.getIncreaseStock();
        if (decreaseStockId != null && decreaseStock != null) {
            remoteStockBackendService.decreaseItemStock(decreaseStockId, decreaseStock);
        }
        if (increaseStockId != null && increaseStock != null) {
            remoteStockBackendService.increaseItemStock(increaseStockId, increaseStock);
        }
        try {
            updateOrderItemInTransaction(param);
        } catch (Exception e) {
            // 更新失败反向操作库存
            if (decreaseStockId != null && decreaseStock != null) {
                remoteStockBackendService.increaseItemStock(decreaseStockId, decreaseStock);
            }
            if (increaseStockId != null && increaseStock != null) {
                remoteStockBackendService.decreaseItemStock(increaseStockId, increaseStock);
            }
            throw new QihoException("更新订单商品详情失败，param=" + param, e);
        }
    }

    /**
     * 事务内更新订单商品详情
     * 
     * @author zhanglihui
     * @param param
     */
    @Transactional(DsConstants.DATABASE_QIHO)
    public void updateOrderItemInTransaction(OrderItemParam param) {
        QihoOrderItemInfoEntity qihoOrderItemInfo = new QihoOrderItemInfoEntity();
        qihoOrderItemInfo.setOrderId(param.getOrderId());
        qihoOrderItemInfo.setSkuId(param.getSkuId());
        qihoOrderItemInfo.setSkuName(param.getSkuName());
        qihoOrderItemInfo.setSkuNo(param.getSkuNo());
        qihoOrderItemInfo.setSellingPrice(param.getSellingPrice());
        qihoOrderItemInfo.setOriginalPrice(param.getOriginalPrice());
        qihoOrderItemInfo.setQuantity(param.getQuantity());
        qihoOrderItemInfoDAO.updateByOrderId(qihoOrderItemInfo);

        // 更新订单金额
        Integer orderAmt = param.getSellingPrice() * param.getQuantity();
        qihoOrderDAO.updateOrderAmt(param.getOrderId(), orderAmt);

        QihoOrderSnapshotEntity orderSnapshot = BeanUtils.copy(param, QihoOrderSnapshotEntity.class);
        orderSnapshot.setOrderAmt(orderAmt);
        qihoOrderSnapshotDAO.updateByOrderId(orderSnapshot);
    }

    /**
     * @see com.qiho.center.biz.service.order.OrderService#updateOrderConsumer(com.qiho.center.api.params.OrderConsumerParam)
     */
    @Override
    @Transactional(DsConstants.DATABASE_QIHO)
    public void updateOrderConsumer(OrderConsumerParam param) {
        QihoOrderEntity order = qihoOrderDAO.findByOrderId(param.getOrderId());
        if (!StringUtils.equals(order.getOrderStatus(), OrderStatusEnum.TO_APPROVE.getCode())) {
            LOG.warn("更新订单买家信息失败，订单状态错误，orderId=" + param.getOrderId() + ", status=" + order.getOrderStatus());
            throw new QihoException("操作失败");
        }
        Long blackId = expressBlackListService.findOneBlack(ExpressCompanyEnum.YTO, param.getAreaNum());
        if (blackId != null) {
            LOG.warn("更新订单买家信息失败，寄送地址不在货到付款配送区域内，orderId=" + param.getOrderId() + ", 行政区划代码=" + param.getAreaNum());
            throw new QihoException("下单失败！您的寄送地址不在货到付款配送区域内，暂时无法下单!");
        }
        QihoOrderEntity updateOrder = BeanUtils.copy(param, QihoOrderEntity.class);
        qihoOrderDAO.updateByOrderId(updateOrder);
        QihoOrderSnapshotEntity orderSnapshot = BeanUtils.copy(param, QihoOrderSnapshotEntity.class);
        qihoOrderSnapshotDAO.updateByOrderId(orderSnapshot);
    }

    @Override
    public void rejectOrder(String orderId, String remark) {
        qihoOrderDAO.rejectOrder(orderId,remark);
    }

    @Override
    @Transactional(DsConstants.DATABASE_QIHO)
    public void updateOrderStatus(String orderId, OrderStatusEnum orderStatus) {
        qihoOrderDAO.updateOrderStatus(orderId,orderStatus);
        qihoOrderSnapshotDAO.updateOrderStatus(orderId, orderStatus);
    }
    
    /**
     * 监听物流状态更新事件
     * @author chencheng
     * @param event
     */
    @Subscribe
    public void updateLogisticsListener(LogisticsUpdateEvent event) {
        String erpId = event.getErpId();
        if (StringUtils.isBlank(erpId)) {
            LOG.info("不需要更改订单的状态，erpId为空");
        }
        QihoErpOrderEntity entity = qihoErpOrderDAO.findByErpId(erpId);
        if(Objects.equal(null,entity) || !StringUtils.equals(entity.getErpType(),ErpTypeEnum.SEND.getCode())) {
            LOG.info("不需要更改订单的状态，erpId为{}",erpId);
        }
        LogisticsOrderStatusEnum status = event.getOrderStatus();
        switch (status) {
            case SUCCESS:
                updateOrderStatus(event.getOrderId(), OrderStatusEnum.SUCCESS);
                break;
            case FAILED:
                QihoOrderEntity updateOrder = new QihoOrderEntity();
                updateOrder.setOrderId(event.getOrderId());
                updateOrder.setOrderStatus(OrderStatusEnum.CLOSED.getCode());
                updateOrder.setRemark(event.getLogisticsStatus().getDesc());
                qihoOrderDAO.updateByOrderId(updateOrder);
                qihoOrderSnapshotDAO.updateOrderStatus(event.getOrderId(), OrderStatusEnum.CLOSED);
                break;
            default:
                LOG.info("不需要更改订单的状态，status:{}"+status.getCode());
                break;
        }
    }
}
